package com.chenyang.nse.bussiness.tools.dataprocess.hive.hdfs.operator;

import ch.qos.logback.classic.Logger;
import com.chenyang.nse.bussiness.tools.dataprocess.hive.hdfs.HdfsFileInfo;
import com.chenyang.nse.bussiness.tools.dataprocess.hive.hdfs.TableInfo;
import com.chenyang.nse.bussiness.tools.dataprocess.hive.hdfs.format.HdfsItemIterator;
import com.chenyang.nse.bussiness.tools.jdbc.JdbcTool;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public abstract class HdfsOperator {
   protected Logger logger;
   protected final TableInfo table;
   protected final Connection connection;
   protected final Properties properties;

   public void setLogger(Logger logger) {
      this.logger = logger;
   }

   public static HdfsOperator createHdfsOperator(HdfsType type, Connection connection, Properties properties, TableInfo table) {
      switch (type) {
         case webHdfs:
            return new WebHdfsOperator(connection, properties, table);
         case keberos:
            return new KeberosHdfsOperator(connection, properties, table);
         case ldap:
            return new LdapHdfsOperator(connection, properties, table);
         default:
            return null;
      }
   }

   public HdfsOperator(Connection connection, Properties properties, TableInfo table) {
      this.connection = connection;
      this.properties = properties;
      this.table = table;
   }

   public abstract void prepare(String schema, String table) throws Exception;

   public abstract List<HdfsFileInfo> listFiles(String schema, String table, String location) throws Exception;

   public abstract InputStream loadFile(String schema, String table, String location) throws IOException;

   public abstract HdfsItemIterator loadFileIter(String schema, String table, String location) throws IOException, URISyntaxException;

   public abstract OutputStream createFile(String schema, String table, String location) throws Exception;

   public abstract void saveBatch(String schema, String table, String location, OutputStream stream) throws Exception;

   public abstract void fininshed(String schema, String table) throws Exception;

   public abstract void closeFile(String schema, String table, String location) throws Exception;

   public abstract void deleteFile(String schema, String table, String file) throws IOException, URISyntaxException;

   public abstract void rename(String schema, String table, String file, String newname) throws Exception;

   protected String getTableLocation(String schema, String table) {
      String location = JdbcTool.queryHdfsLocationForTDH(this.connection, schema, table);
      this.logger.info("表 " + schema + "." + table + " 在HDFS的路径为 ：" + location);
      return location;
   }

   protected Map<String, String> getTableInfo(String schema, String table) {
      Map<String, String> map = JdbcTool.queryHdfsFileInfoForTDH(this.connection, schema, table);
      return map;
   }

   public static enum HdfsType {
      webHdfs,
      keberos,
      ldap;
   }
}
