package com.chenyang.nse.bussiness.tools.quartz;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.UUID;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.quartz.impl.StdSchedulerFactory;

public class QuartzUtil {
   private static Scheduler scheduler;
   private static final String TRIGGER_GROUP = "SCH_TRIGGER_GROUP_NAME";
   private static final String JOB_GROUP = "SCH_JOB_GROUP_NAME";

   private QuartzUtil() {
   }

   public static void startScheduler() throws SchedulerException {
      scheduler.start();
   }

   public static void closeScheduler() throws SchedulerException {
      scheduler.shutdown();
   }

   private static String getUUID() {
      return UUID.randomUUID().toString();
   }

   private static Boolean isCronExpression(String cronExpression) {
      return CronExpression.isValidExpression(cronExpression);
   }

   private static Boolean isRepeatTrigger(TriggerKey triggerKey) throws SchedulerException {
      return scheduler.checkExists(triggerKey);
   }

   private static Boolean isRepeatJob(JobKey jobKey) throws SchedulerException {
      return scheduler.checkExists(jobKey);
   }

   private static Trigger getCronTrigger(TriggerKey triggerKey, String cronExpression) throws ParseException {
      if (triggerKey == null) {
         triggerKey = new TriggerKey("SCH_TRIGGER_GROUP_NAME", getUUID());
      }

      if (!isCronExpression(cronExpression)) {
         throw new ParseException(cronExpression, 0);
      } else {
         Trigger trigger = TriggerBuilder.newTrigger().withIdentity(triggerKey).withSchedule(CronScheduleBuilder.cronSchedule(cronExpression)).build();
         return trigger;
      }
   }

   private static JobDetail getJobDetail(JobKey jobKey, Class<? extends Job> jobClass, JobDataMap jobDataMap) {
      if (jobKey == null) {
         jobKey = new JobKey("SCH_JOB_GROUP_NAME", getUUID());
      }

      if (jobClass == null) {
         throw new NullPointerException();
      } else {
         if (jobDataMap == null) {
            jobDataMap = new JobDataMap();
         }

         JobDetail jobDetail = JobBuilder.newJob(jobClass).withIdentity(jobKey).storeDurably().usingJobData(jobDataMap).build();
         return jobDetail;
      }
   }

   public static Boolean addJob(Class<? extends Job> jobClass, String cronExpression, JobKey jobKey, TriggerKey triggerKey, JobDataMap jobDataMap) throws SchedulerException, ParseException {
      Boolean bool = false;
      if (isRepeatTrigger(triggerKey)) {
         return bool;
      } else if (isRepeatJob(jobKey)) {
         return bool;
      } else {
         Trigger trigger = getCronTrigger(triggerKey, cronExpression);
         JobDetail jobDetail = getJobDetail(jobKey, jobClass, jobDataMap);
         scheduler.scheduleJob(jobDetail, trigger);
         return !bool;
      }
   }

   public static void removeTrigger(TriggerKey triggerKey) throws SchedulerException {
      scheduler.unscheduleJob(triggerKey);
   }

   public static void removeJob(JobKey jobKey) throws SchedulerException {
      scheduler.deleteJob(jobKey);
   }

   public static Boolean modifyTriggerTime(TriggerKey triggerKey, String cronExpression) throws SchedulerException, ParseException {
      Boolean bool = false;
      if (triggerKey == null) {
         return bool;
      } else if (!isCronExpression(cronExpression)) {
         throw new ParseException(cronExpression, 0);
      } else {
         CronTrigger cronTrigger = (CronTrigger)scheduler.getTrigger(triggerKey);
         if (cronTrigger == null) {
            return bool;
         } else {
            cronTrigger = (CronTrigger)cronTrigger.getTriggerBuilder().withSchedule(CronScheduleBuilder.cronSchedule(cronExpression)).build();
            scheduler.rescheduleJob(triggerKey, cronTrigger);
            return !bool;
         }
      }
   }

   public static Boolean modifyJob(JobKey jobKey, Class<? extends Job> jobClass, JobDataMap jobDataMap) throws SchedulerException {
      Boolean bool = false;
      if (!isRepeatJob(jobKey)) {
         return bool;
      } else {
         JobDetail jobDetail = getJobDetail(jobKey, jobClass, jobDataMap);
         scheduler.addJob(jobDetail, true);
         return !bool;
      }
   }

   public static Boolean modifyJobDataMap(JobKey jobKey, JobDataMap jobDataMap) throws SchedulerException {
      Boolean bool = false;
      if (!isRepeatJob(jobKey)) {
         return bool;
      } else {
         JobDetail jobDetail = scheduler.getJobDetail(jobKey);
         jobDetail = jobDetail.getJobBuilder().setJobData(jobDataMap).build();
         scheduler.addJob(jobDetail, true);
         return !bool;
      }
   }

   public static String getCronExpression(String rulesType, String rulesDay, String rulesHour, String rulesMinute, String xloptions, String howtime) {
      StringBuffer sb = new StringBuffer();
      String[] days = rulesDay.split("\\s+|-|:");
      String year = days[0];
      String month = days[1];
      String day = days[2];
      if (rulesType.equals("0")) {
         sb.append("0 ");
         sb.append(rulesMinute + " ");
         sb.append(rulesHour + " ");
         sb.append(day + " ");
         sb.append(month + " ");
         sb.append("? ");
         sb.append(year);
      } else {
         if (xloptions.equals("1")) {
            sb.append("0 ");
            sb.append(rulesMinute + " ");
            sb.append(rulesHour + " ");
            sb.append("*/" + howtime + " ");
            sb.append("* ");
            sb.append("?");
         }

         if (xloptions.equals("7")) {
            String[] weekDays = new String[]{"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"};
            Calendar calendar = Calendar.getInstance();
            String week = weekDays[calendar.get(7) - 1];
            sb.append("0 ");
            sb.append(rulesMinute + " ");
            sb.append(rulesHour + " ");
            sb.append("? ");
            sb.append("1/1 ");
            sb.append(week + "#" + howtime);
         }

         if (xloptions.equals("30")) {
            sb.append("0 ");
            sb.append(rulesMinute + " ");
            sb.append(rulesHour + " ");
            sb.append(day + " ");
            sb.append(month + "/" + howtime + " ");
            sb.append("?");
         }

         if (xloptions.equals("10")) {
            sb.append("0 ");
            sb.append(rulesMinute + " ");
            sb.append("*/" + howtime + " ");
            sb.append("* ");
            sb.append("* ");
            sb.append("?");
         }

         if (xloptions.equals("60")) {
            sb.append("0 ");
            sb.append("0/" + howtime + " ");
            sb.append("* ");
            sb.append("* ");
            sb.append("* ");
            sb.append("?");
         }
      }

      return sb.toString();
   }

   public static Boolean compareNowDate(String days, String hour, String minute) throws ParseException {
      Boolean bool = false;
      SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
      Date date = sdf.parse(days + " " + hour + ":" + minute);
      return date.getTime() > (new Date()).getTime() ? !bool : bool;
   }

   static {
      try {
         scheduler = StdSchedulerFactory.getDefaultScheduler();
         startScheduler();
      } catch (SchedulerException e) {
         e.printStackTrace();
      }

   }
}
