package com.chenyang.druid.support.json;

import com.chenyang.druid.sql.visitor.SQLEvalVisitor;
import com.chenyang.druid.util.Utils;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.TabularData;

public class JSONWriter {
   private StringBuilder out = new StringBuilder();

   public void writeArrayStart() {
      this.write('[');
   }

   public void writeComma() {
      this.write(',');
   }

   public void writeArrayEnd() {
      this.write(']');
   }

   public void writeNull() {
      this.write("null");
   }

   public void writeObject(Object o) {
      if (o == null) {
         this.writeNull();
      } else if (o instanceof String) {
         this.writeString((String)o);
      } else if (o instanceof Number) {
         this.write(o.toString());
      } else if (o instanceof Boolean) {
         this.write(o.toString());
      } else if (o instanceof Date) {
         this.writeDate((Date)o);
      } else if (o instanceof Collection) {
         this.writeArray((Collection)o);
      } else if (o instanceof Throwable) {
         this.writeError((Throwable)o);
      } else if (o instanceof int[]) {
         int[] array = (int[])o;
         this.write('[');

         for(int i = 0; i < array.length; ++i) {
            if (i != 0) {
               this.write(',');
            }

            this.write(array[i]);
         }

         this.write(']');
      } else if (o instanceof long[]) {
         long[] array = (long[])o;
         this.write('[');

         for(int i = 0; i < array.length; ++i) {
            if (i != 0) {
               this.write(',');
            }

            this.write(array[i]);
         }

         this.write(']');
      } else if (o instanceof TabularData) {
         this.writeTabularData((TabularData)o);
      } else if (o instanceof CompositeData) {
         this.writeCompositeData((CompositeData)o);
      } else if (o instanceof Map) {
         this.writeMap((Map)o);
      } else if (o == SQLEvalVisitor.EVAL_VALUE_NULL) {
         this.write("null");
      } else {
         throw new IllegalArgumentException("not support type : " + o.getClass());
      }
   }

   public void writeDate(Date date) {
      if (date == null) {
         this.writeNull();
      } else {
         SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
         this.writeString(dateFormat.format(date));
      }
   }

   public void writeError(Throwable error) {
      if (error == null) {
         this.writeNull();
      } else {
         this.write("{\"Class\":");
         this.writeString(error.getClass().getName());
         this.write(",\"Message\":");
         this.writeString(error.getMessage());
         this.write(",\"StackTrace\":");
         this.writeString(Utils.getStackTrace(error));
         this.write('}');
      }
   }

   public void writeArray(Object[] array) {
      if (array == null) {
         this.writeNull();
      } else {
         this.write('[');

         for(int i = 0; i < array.length; ++i) {
            if (i != 0) {
               this.write(',');
            }

            this.writeObject(array[i]);
         }

         this.write(']');
      }
   }

   public void writeArray(Collection<Object> list) {
      if (list == null) {
         this.writeNull();
      } else {
         int entryIndex = 0;
         this.write('[');

         for(Object entry : list) {
            if (entryIndex != 0) {
               this.write(',');
            }

            this.writeObject(entry);
            ++entryIndex;
         }

         this.write(']');
      }
   }

   public void writeString(String text) {
      if (text == null) {
         this.writeNull();
      } else {
         this.write('"');

         for(int i = 0; i < text.length(); ++i) {
            char c = text.charAt(i);
            if (c == '"') {
               this.write("\\\"");
            } else if (c == '\n') {
               this.write("\\n");
            } else if (c == '\r') {
               this.write("\\r");
            } else if (c == '\\') {
               this.write("\\\\");
            } else if (c == '\t') {
               this.write("\\t");
            } else if (c < 16) {
               this.write("\\u000");
               this.write(Integer.toHexString(c));
            } else if (c < ' ') {
               this.write("\\u00");
               this.write(Integer.toHexString(c));
            } else if (c >= 127 && c <= 160) {
               this.write("\\u00");
               this.write(Integer.toHexString(c));
            } else {
               this.write(c);
            }
         }

         this.write('"');
      }
   }

   public void writeTabularData(TabularData tabularData) {
      if (tabularData == null) {
         this.writeNull();
      } else {
         int entryIndex = 0;
         this.write('[');

         for(Object item : tabularData.values()) {
            if (entryIndex != 0) {
               this.write(',');
            }

            CompositeData row = (CompositeData)item;
            this.writeCompositeData(row);
            ++entryIndex;
         }

         this.write(']');
      }
   }

   public void writeCompositeData(CompositeData compositeData) {
      if (compositeData == null) {
         this.writeNull();
      } else {
         int entryIndex = 0;
         this.write('{');

         for(Object key : compositeData.getCompositeType().keySet()) {
            if (entryIndex != 0) {
               this.write(',');
            }

            this.writeString((String)key);
            this.write(':');
            Object value = compositeData.get((String)key);
            this.writeObject(value);
            ++entryIndex;
         }

         this.write('}');
      }
   }

   public void writeMap(Map<String, Object> map) {
      if (map == null) {
         this.writeNull();
      } else {
         int entryIndex = 0;
         this.write('{');

         for(Map.Entry<String, Object> entry : map.entrySet()) {
            if (entryIndex != 0) {
               this.write(',');
            }

            this.writeString((String)entry.getKey());
            this.write(':');
            this.writeObject(entry.getValue());
            ++entryIndex;
         }

         this.write('}');
      }
   }

   protected void write(String text) {
      this.out.append(text);
   }

   protected void write(char c) {
      this.out.append(c);
   }

   protected void write(int c) {
      this.out.append(c);
   }

   protected void write(long c) {
      this.out.append(c);
   }

   public String toString() {
      return this.out.toString();
   }
}
