package org.chenyang.http.impl.client;

import org.chenyang.commons.logging.Log;
import org.chenyang.commons.logging.LogFactory;
import org.chenyang.http.HttpException;
import org.chenyang.http.HttpHost;
import org.chenyang.http.HttpRequest;
import org.chenyang.http.annotation.Contract;
import org.chenyang.http.annotation.ThreadingBehavior;
import org.chenyang.http.auth.AuthSchemeProvider;
import org.chenyang.http.auth.AuthState;
import org.chenyang.http.client.ClientProtocolException;
import org.chenyang.http.client.CookieStore;
import org.chenyang.http.client.CredentialsProvider;
import org.chenyang.http.client.config.RequestConfig;
import org.chenyang.http.client.methods.CloseableHttpResponse;
import org.chenyang.http.client.methods.Configurable;
import org.chenyang.http.client.methods.HttpExecutionAware;
import org.chenyang.http.client.methods.HttpRequestWrapper;
import org.chenyang.http.client.params.HttpClientParamConfig;
import org.chenyang.http.client.protocol.HttpClientContext;
import org.chenyang.http.config.Lookup;
import org.chenyang.http.conn.ClientConnectionManager;
import org.chenyang.http.conn.ClientConnectionRequest;
import org.chenyang.http.conn.HttpClientConnectionManager;
import org.chenyang.http.conn.ManagedClientConnection;
import org.chenyang.http.conn.routing.HttpRoute;
import org.chenyang.http.conn.routing.HttpRoutePlanner;
import org.chenyang.http.conn.scheme.SchemeRegistry;
import org.chenyang.http.cookie.CookieSpecProvider;
import org.chenyang.http.impl.execchain.ClientExecChain;
import org.chenyang.http.params.HttpParams;
import org.chenyang.http.params.HttpParamsNames;
import org.chenyang.http.protocol.BasicHttpContext;
import org.chenyang.http.protocol.HttpContext;
import org.chenyang.http.util.Args;
import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Contract(
   threading = ThreadingBehavior.SAFE_CONDITIONAL
)
class InternalHttpClient extends CloseableHttpClient implements Configurable {
   private final Log log = LogFactory.getLog(this.getClass());
   private final ClientExecChain execChain;
   private final HttpClientConnectionManager connManager;
   private final HttpRoutePlanner routePlanner;
   private final Lookup<CookieSpecProvider> cookieSpecRegistry;
   private final Lookup<AuthSchemeProvider> authSchemeRegistry;
   private final CookieStore cookieStore;
   private final CredentialsProvider credentialsProvider;
   private final RequestConfig defaultConfig;
   private final List<Closeable> closeables;

   public InternalHttpClient(ClientExecChain execChain, HttpClientConnectionManager connManager, HttpRoutePlanner routePlanner, Lookup<CookieSpecProvider> cookieSpecRegistry, Lookup<AuthSchemeProvider> authSchemeRegistry, CookieStore cookieStore, CredentialsProvider credentialsProvider, RequestConfig defaultConfig, List<Closeable> closeables) {
      Args.notNull(execChain, "HTTP client exec chain");
      Args.notNull(connManager, "HTTP connection manager");
      Args.notNull(routePlanner, "HTTP route planner");
      this.execChain = execChain;
      this.connManager = connManager;
      this.routePlanner = routePlanner;
      this.cookieSpecRegistry = cookieSpecRegistry;
      this.authSchemeRegistry = authSchemeRegistry;
      this.cookieStore = cookieStore;
      this.credentialsProvider = credentialsProvider;
      this.defaultConfig = defaultConfig;
      this.closeables = closeables;
   }

   private HttpRoute determineRoute(HttpHost target, HttpRequest request, HttpContext context) throws HttpException {
      HttpHost host = target;
      if (target == null) {
         host = (HttpHost)request.getParams().getParameter("http.default-host");
      }

      return this.routePlanner.determineRoute(host, request, context);
   }

   private void setupContext(HttpClientContext context) {
      if (context.getAttribute("http.auth.target-scope") == null) {
         context.setAttribute("http.auth.target-scope", new AuthState());
      }

      if (context.getAttribute("http.auth.proxy-scope") == null) {
         context.setAttribute("http.auth.proxy-scope", new AuthState());
      }

      if (context.getAttribute("http.authscheme-registry") == null) {
         context.setAttribute("http.authscheme-registry", this.authSchemeRegistry);
      }

      if (context.getAttribute("http.cookiespec-registry") == null) {
         context.setAttribute("http.cookiespec-registry", this.cookieSpecRegistry);
      }

      if (context.getAttribute("http.cookie-store") == null) {
         context.setAttribute("http.cookie-store", this.cookieStore);
      }

      if (context.getAttribute("http.auth.credentials-provider") == null) {
         context.setAttribute("http.auth.credentials-provider", this.credentialsProvider);
      }

      if (context.getAttribute("http.request-config") == null) {
         context.setAttribute("http.request-config", this.defaultConfig);
      }

   }

   protected CloseableHttpResponse doExecute(HttpHost target, HttpRequest request, HttpContext context) throws IOException, ClientProtocolException {
      Args.notNull(request, "HTTP request");
      HttpExecutionAware execAware = null;
      if (request instanceof HttpExecutionAware) {
         execAware = (HttpExecutionAware)request;
      }

      try {
         HttpRequestWrapper wrapper = HttpRequestWrapper.wrap(request, target);
         HttpClientContext localcontext = HttpClientContext.adapt((HttpContext)(context != null ? context : new BasicHttpContext()));
         RequestConfig config = null;
         if (request instanceof Configurable) {
            config = ((Configurable)request).getConfig();
         }

         if (config == null) {
            HttpParams params = request.getParams();
            if (params instanceof HttpParamsNames) {
               if (!((HttpParamsNames)params).getNames().isEmpty()) {
                  config = HttpClientParamConfig.getRequestConfig(params, this.defaultConfig);
               }
            } else {
               config = HttpClientParamConfig.getRequestConfig(params, this.defaultConfig);
            }
         }

         if (config != null) {
            localcontext.setRequestConfig(config);
         }

         this.setupContext(localcontext);
         HttpRoute route = this.determineRoute(target, wrapper, localcontext);
         return this.execChain.execute(route, wrapper, localcontext, execAware);
      } catch (HttpException httpException) {
         throw new ClientProtocolException(httpException);
      }
   }

   public RequestConfig getConfig() {
      return this.defaultConfig;
   }

   public void close() {
      if (this.closeables != null) {
         for(Closeable closeable : this.closeables) {
            try {
               closeable.close();
            } catch (IOException ex) {
               this.log.error(ex.getMessage(), ex);
            }
         }
      }

   }

   public HttpParams getParams() {
      throw new UnsupportedOperationException();
   }

   public ClientConnectionManager getConnectionManager() {
      return new ClientConnectionManager() {
         public void shutdown() {
            InternalHttpClient.this.connManager.shutdown();
         }

         public ClientConnectionRequest requestConnection(HttpRoute route, Object state) {
            throw new UnsupportedOperationException();
         }

         public void releaseConnection(ManagedClientConnection conn, long validDuration, TimeUnit timeUnit) {
            throw new UnsupportedOperationException();
         }

         public SchemeRegistry getSchemeRegistry() {
            throw new UnsupportedOperationException();
         }

         public void closeIdleConnections(long idletime, TimeUnit timeUnit) {
            InternalHttpClient.this.connManager.closeIdleConnections(idletime, timeUnit);
         }

         public void closeExpiredConnections() {
            InternalHttpClient.this.connManager.closeExpiredConnections();
         }
      };
   }
}
