package com.alibaba.fastjson;

import com.alibaba.fastjson.annotation.JSONField;
import com.alibaba.fastjson.parser.ParserConfig;
import com.alibaba.fastjson.util.TypeUtils;
import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public class JSONObject extends JSON implements Map<String, Object>, Cloneable, Serializable, InvocationHandler {
   private static final long serialVersionUID = 1L;
   private static final int DEFAULT_INITIAL_CAPACITY = 16;
   private final Map<String, Object> map;

   public JSONObject() {
      this(16, false);
   }

   public JSONObject(Map<String, Object> map) {
      if (map == null) {
         throw new IllegalArgumentException("map is null.");
      } else {
         this.map = map;
      }
   }

   public JSONObject(boolean ordered) {
      this(16, ordered);
   }

   public JSONObject(int initialCapacity) {
      this(initialCapacity, false);
   }

   public JSONObject(int initialCapacity, boolean ordered) {
      if (ordered) {
         this.map = new LinkedHashMap(initialCapacity);
      } else {
         this.map = new HashMap(initialCapacity);
      }

   }

   public int size() {
      return this.map.size();
   }

   public boolean isEmpty() {
      return this.map.isEmpty();
   }

   public boolean containsKey(Object key) {
      boolean result = this.map.containsKey(key);
      if (!result && (key instanceof Number || key instanceof Character || key instanceof Boolean || key instanceof UUID)) {
         result = this.map.containsKey(key.toString());
      }

      return result;
   }

   public boolean containsValue(Object value) {
      return this.map.containsValue(value);
   }

   public Object get(Object key) {
      Object val = this.map.get(key);
      if (val == null && (key instanceof Number || key instanceof Character || key instanceof Boolean || key instanceof UUID)) {
         val = this.map.get(key.toString());
      }

      return val;
   }

   public Object getOrDefault(Object key, Object defaultValue) {
      Object v;
      return (v = this.get(key)) != null ? v : defaultValue;
   }

   public JSONObject getJSONObject(String key) {
      Object value = this.map.get(key);
      if (value instanceof JSONObject) {
         return (JSONObject)value;
      } else if (value instanceof Map) {
         return new JSONObject((Map)value);
      } else {
         return value instanceof String ? JSON.parseObject((String)value) : (JSONObject)toJSON(value);
      }
   }

   public JSONArray getJSONArray(String key) {
      Object value = this.map.get(key);
      if (value instanceof JSONArray) {
         return (JSONArray)value;
      } else if (value instanceof List) {
         return new JSONArray((List)value);
      } else {
         return value instanceof String ? (JSONArray)JSON.parse((String)value) : (JSONArray)toJSON(value);
      }
   }

   public <T> T getObject(String key, Class<T> clazz) {
      Object obj = this.map.get(key);
      return (T)TypeUtils.castToJavaBean(obj, clazz);
   }

   public <T> T getObject(String key, Type type) {
      Object obj = this.map.get(key);
      return (T)TypeUtils.cast(obj, type, ParserConfig.getGlobalInstance());
   }

   public <T> T getObject(String key, TypeReference typeReference) {
      Object obj = this.map.get(key);
      return (T)(typeReference == null ? obj : TypeUtils.cast(obj, typeReference.getType(), ParserConfig.getGlobalInstance()));
   }

   public Boolean getBoolean(String key) {
      Object value = this.get(key);
      return value == null ? null : TypeUtils.castToBoolean(value);
   }

   public byte[] getBytes(String key) {
      Object value = this.get(key);
      return value == null ? null : TypeUtils.castToBytes(value);
   }

   public boolean getBooleanValue(String key) {
      Object value = this.get(key);
      Boolean booleanVal = TypeUtils.castToBoolean(value);
      return booleanVal == null ? false : booleanVal;
   }

   public Byte getByte(String key) {
      Object value = this.get(key);
      return TypeUtils.castToByte(value);
   }

   public byte getByteValue(String key) {
      Object value = this.get(key);
      Byte byteVal = TypeUtils.castToByte(value);
      return byteVal == null ? 0 : byteVal;
   }

   public Short getShort(String key) {
      Object value = this.get(key);
      return TypeUtils.castToShort(value);
   }

   public short getShortValue(String key) {
      Object value = this.get(key);
      Short shortVal = TypeUtils.castToShort(value);
      return shortVal == null ? 0 : shortVal;
   }

   public Integer getInteger(String key) {
      Object value = this.get(key);
      return TypeUtils.castToInt(value);
   }

   public int getIntValue(String key) {
      Object value = this.get(key);
      Integer intVal = TypeUtils.castToInt(value);
      return intVal == null ? 0 : intVal;
   }

   public Long getLong(String key) {
      Object value = this.get(key);
      return TypeUtils.castToLong(value);
   }

   public long getLongValue(String key) {
      Object value = this.get(key);
      Long longVal = TypeUtils.castToLong(value);
      return longVal == null ? 0L : longVal;
   }

   public Float getFloat(String key) {
      Object value = this.get(key);
      return TypeUtils.castToFloat(value);
   }

   public float getFloatValue(String key) {
      Object value = this.get(key);
      Float floatValue = TypeUtils.castToFloat(value);
      return floatValue == null ? 0.0F : floatValue;
   }

   public Double getDouble(String key) {
      Object value = this.get(key);
      return TypeUtils.castToDouble(value);
   }

   public double getDoubleValue(String key) {
      Object value = this.get(key);
      Double doubleValue = TypeUtils.castToDouble(value);
      return doubleValue == null ? (double)0.0F : doubleValue;
   }

   public BigDecimal getBigDecimal(String key) {
      Object value = this.get(key);
      return TypeUtils.castToBigDecimal(value);
   }

   public BigInteger getBigInteger(String key) {
      Object value = this.get(key);
      return TypeUtils.castToBigInteger(value);
   }

   public String getString(String key) {
      Object value = this.get(key);
      return value == null ? null : value.toString();
   }

   public Date getDate(String key) {
      Object value = this.get(key);
      return TypeUtils.castToDate(value);
   }

   public Object getSqlDate(String key) {
      Object value = this.get(key);
      return TypeUtils.castToSqlDate(value);
   }

   public Object getTimestamp(String key) {
      Object value = this.get(key);
      return TypeUtils.castToTimestamp(value);
   }

   public Object put(String key, Object value) {
      return this.map.put(key, value);
   }

   public JSONObject fluentPut(String key, Object value) {
      this.map.put(key, value);
      return this;
   }

   public void putAll(Map<? extends String, ?> m) {
      this.map.putAll(m);
   }

   public JSONObject fluentPutAll(Map<? extends String, ?> m) {
      this.map.putAll(m);
      return this;
   }

   public void clear() {
      this.map.clear();
   }

   public JSONObject fluentClear() {
      this.map.clear();
      return this;
   }

   public Object remove(Object key) {
      return this.map.remove(key);
   }

   public JSONObject fluentRemove(Object key) {
      this.map.remove(key);
      return this;
   }

   public Set<String> keySet() {
      return this.map.keySet();
   }

   public Collection<Object> values() {
      return this.map.values();
   }

   public Set<Entry<String, Object>> entrySet() {
      return this.map.entrySet();
   }

   public JSONObject clone() {
      return new JSONObject((Map)(this.map instanceof LinkedHashMap ? new LinkedHashMap(this.map) : new HashMap(this.map)));
   }

   public boolean equals(Object obj) {
      if (this == obj) {
         return true;
      } else {
         return obj instanceof JSONObject ? this.map.equals(((JSONObject)obj).map) : this.map.equals(obj);
      }
   }

   public int hashCode() {
      return this.map.hashCode();
   }

   public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
      Class<?>[] parameterTypes = method.getParameterTypes();
      if (parameterTypes.length == 1) {
         if (method.getName().equals("equals")) {
            return this.equals(args[0]);
         } else {
            Class<?> returnType = method.getReturnType();
            if (returnType != Void.TYPE) {
               throw new JSONException("illegal setter");
            } else {
               String name = null;
               JSONField annotation = (JSONField)TypeUtils.getAnnotation(method, JSONField.class);
               if (annotation != null && annotation.name().length() != 0) {
                  name = annotation.name();
               }

               if (name == null) {
                  name = method.getName();
                  if (!name.startsWith("set")) {
                     throw new JSONException("illegal setter");
                  }

                  name = name.substring(3);
                  if (name.length() == 0) {
                     throw new JSONException("illegal setter");
                  }

                  name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
               }

               this.map.put(name, args[0]);
               return null;
            }
         }
      } else if (parameterTypes.length == 0) {
         Class<?> returnType = method.getReturnType();
         if (returnType == Void.TYPE) {
            throw new JSONException("illegal getter");
         } else {
            String name = null;
            JSONField annotation = (JSONField)TypeUtils.getAnnotation(method, JSONField.class);
            if (annotation != null && annotation.name().length() != 0) {
               name = annotation.name();
            }

            if (name == null) {
               name = method.getName();
               if (name.startsWith("get")) {
                  name = name.substring(3);
                  if (name.length() == 0) {
                     throw new JSONException("illegal getter");
                  }

                  name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
               } else {
                  if (!name.startsWith("is")) {
                     if (name.startsWith("hashCode")) {
                        return this.hashCode();
                     }

                     if (name.startsWith("toString")) {
                        return this.toString();
                     }

                     throw new JSONException("illegal getter");
                  }

                  name = name.substring(2);
                  if (name.length() == 0) {
                     throw new JSONException("illegal getter");
                  }

                  name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
               }
            }

            Object value = this.map.get(name);
            return TypeUtils.cast(value, method.getGenericReturnType(), ParserConfig.getGlobalInstance());
         }
      } else {
         throw new UnsupportedOperationException(method.toGenericString());
      }
   }

   public Map<String, Object> getInnerMap() {
      return this.map;
   }

   public <T> T toJavaObject(Class<T> clazz) {
      if (clazz != Map.class && clazz != JSONObject.class && clazz != JSON.class) {
         return (T)(clazz == Object.class && !this.containsKey(JSON.DEFAULT_TYPE_KEY) ? this : TypeUtils.castToJavaBean(this, clazz, ParserConfig.getGlobalInstance()));
      } else {
         return (T)this;
      }
   }

   public <T> T toJavaObject(Class<T> clazz, ParserConfig config, int features) {
      if (clazz == Map.class) {
         return (T)this;
      } else {
         return (T)(clazz == Object.class && !this.containsKey(JSON.DEFAULT_TYPE_KEY) ? this : TypeUtils.castToJavaBean(this, clazz, config));
      }
   }
}
