package com.alibaba.fastjson.parser;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.util.IOUtils;
import java.io.Closeable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Locale;
import java.util.TimeZone;
import java.util.UUID;

public abstract class JSONLexerBase implements JSONLexer, Closeable {
   protected int token;
   protected int pos;
   protected int features;
   protected char ch;
   protected int bp;
   protected int eofPos;
   protected char[] sbuf;
   protected int sp;
   protected int np;
   protected boolean hasSpecial;
   protected Calendar calendar = null;
   protected TimeZone timeZone;
   protected Locale locale;
   public int matchStat;
   private static final ThreadLocal<char[]> SBUF_LOCAL = new ThreadLocal();
   protected String stringDefaultValue;
   protected int nanos;
   protected static final char[] typeFieldName;
   protected static final long MULTMIN_RADIX_TEN = -922337203685477580L;
   protected static final int INT_MULTMIN_RADIX_TEN = -214748364;
   protected static final int[] digits;

   protected void lexError(String key, Object... args) {
      this.token = 1;
   }

   public JSONLexerBase(int features) {
      this.timeZone = JSON.defaultTimeZone;
      this.locale = JSON.defaultLocale;
      this.matchStat = 0;
      this.stringDefaultValue = null;
      this.nanos = 0;
      this.features = features;
      if ((features & Feature.InitStringFieldAsEmpty.mask) != 0) {
         this.stringDefaultValue = "";
      }

      this.sbuf = (char[])SBUF_LOCAL.get();
      if (this.sbuf == null) {
         this.sbuf = new char[512];
      }

   }

   public final int matchStat() {
      return this.matchStat;
   }

   public void setToken(int token) {
      this.token = token;
   }

   public final void nextToken() {
      this.sp = 0;

      while(true) {
         this.pos = this.bp;
         if (this.ch != '/') {
            if (this.ch == '"') {
               this.scanString();
               return;
            }

            if (this.ch == ',') {
               this.next();
               this.token = 16;
               return;
            }

            if (this.ch >= '0' && this.ch <= '9') {
               this.scanNumber();
               return;
            }

            if (this.ch == '-') {
               this.scanNumber();
               return;
            }

            switch (this.ch) {
               case '\b':
               case '\t':
               case '\n':
               case '\f':
               case '\r':
               case ' ':
                  this.next();
                  break;
               case '\'':
                  if (!this.isEnabled(Feature.AllowSingleQuotes)) {
                     throw new JSONException("Feature.AllowSingleQuotes is false");
                  }

                  this.scanStringSingleQuote();
                  return;
               case '(':
                  this.next();
                  this.token = 10;
                  return;
               case ')':
                  this.next();
                  this.token = 11;
                  return;
               case '+':
                  this.next();
                  this.scanNumber();
                  return;
               case '.':
                  this.next();
                  this.token = 25;
                  return;
               case ':':
                  this.next();
                  this.token = 17;
                  return;
               case ';':
                  this.next();
                  this.token = 24;
                  return;
               case 'N':
               case 'S':
               case 'T':
               case 'u':
                  this.scanIdent();
                  return;
               case '[':
                  this.next();
                  this.token = 14;
                  return;
               case ']':
                  this.next();
                  this.token = 15;
                  return;
               case 'f':
                  this.scanFalse();
                  return;
               case 'n':
                  this.scanNullOrNew();
                  return;
               case 't':
                  this.scanTrue();
                  return;
               case 'x':
                  this.scanHex();
                  return;
               case '{':
                  this.next();
                  this.token = 12;
                  return;
               case '}':
                  this.next();
                  this.token = 13;
                  return;
               default:
                  if (this.isEOF()) {
                     if (this.token == 20) {
                        throw new JSONException("EOF error");
                     }

                     this.token = 20;
                     this.eofPos = this.pos = this.bp;
                  } else {
                     if (this.ch <= 31 || this.ch == 127) {
                        this.next();
                        continue;
                     }

                     this.lexError("illegal.char", String.valueOf(this.ch));
                     this.next();
                  }

                  return;
            }
         } else {
            this.skipComment();
         }
      }
   }

   public final void nextToken(int expect) {
      this.sp = 0;

      while(true) {
         switch (expect) {
            case 2:
               if (this.ch >= '0' && this.ch <= '9') {
                  this.pos = this.bp;
                  this.scanNumber();
                  return;
               }

               if (this.ch == '"') {
                  this.pos = this.bp;
                  this.scanString();
                  return;
               }

               if (this.ch == '[') {
                  this.token = 14;
                  this.next();
                  return;
               }

               if (this.ch == '{') {
                  this.token = 12;
                  this.next();
                  return;
               }
            case 3:
            case 5:
            case 6:
            case 7:
            case 8:
            case 9:
            case 10:
            case 11:
            case 13:
            case 17:
            case 19:
            default:
               break;
            case 4:
               if (this.ch == '"') {
                  this.pos = this.bp;
                  this.scanString();
                  return;
               }

               if (this.ch >= '0' && this.ch <= '9') {
                  this.pos = this.bp;
                  this.scanNumber();
                  return;
               }

               if (this.ch == '[') {
                  this.token = 14;
                  this.next();
                  return;
               }

               if (this.ch == '{') {
                  this.token = 12;
                  this.next();
                  return;
               }
               break;
            case 12:
               if (this.ch == '{') {
                  this.token = 12;
                  this.next();
                  return;
               }

               if (this.ch == '[') {
                  this.token = 14;
                  this.next();
                  return;
               }
               break;
            case 14:
               if (this.ch == '[') {
                  this.token = 14;
                  this.next();
                  return;
               }

               if (this.ch == '{') {
                  this.token = 12;
                  this.next();
                  return;
               }
               break;
            case 15:
               if (this.ch == ']') {
                  this.token = 15;
                  this.next();
                  return;
               }
            case 20:
               if (this.ch == 26) {
                  this.token = 20;
                  return;
               }
               break;
            case 16:
               if (this.ch == ',') {
                  this.token = 16;
                  this.next();
                  return;
               }

               if (this.ch == '}') {
                  this.token = 13;
                  this.next();
                  return;
               }

               if (this.ch == ']') {
                  this.token = 15;
                  this.next();
                  return;
               }

               if (this.ch == 26) {
                  this.token = 20;
                  return;
               }

               if (this.ch == 'n') {
                  this.scanNullOrNew(false);
                  return;
               }
               break;
            case 18:
               this.nextIdent();
               return;
         }

         if (this.ch != ' ' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != '\f' && this.ch != '\b') {
            this.nextToken();
            return;
         }

         this.next();
      }
   }

   public final void nextIdent() {
      while(isWhitespace(this.ch)) {
         this.next();
      }

      if (this.ch != '_' && this.ch != '$' && !Character.isLetter(this.ch)) {
         this.nextToken();
      } else {
         this.scanIdent();
      }

   }

   public final void nextTokenWithColon() {
      this.nextTokenWithChar(':');
   }

   public final void nextTokenWithChar(char expect) {
      this.sp = 0;

      while(this.ch != expect) {
         if (this.ch != ' ' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != '\f' && this.ch != '\b') {
            throw new JSONException("not match " + expect + " - " + this.ch + ", info : " + this.info());
         }

         this.next();
      }

      this.next();
      this.nextToken();
   }

   public final int token() {
      return this.token;
   }

   public final String tokenName() {
      return JSONToken.name(this.token);
   }

   public final int pos() {
      return this.pos;
   }

   public final String stringDefaultValue() {
      return this.stringDefaultValue;
   }

   public final Number integerValue() throws NumberFormatException {
      long result = 0L;
      boolean negative = false;
      if (this.np == -1) {
         this.np = 0;
      }

      int i = this.np;
      int max = this.np + this.sp;
      char type = ' ';
      switch (this.charAt(max - 1)) {
         case 'B':
            --max;
            type = 'B';
            break;
         case 'L':
            --max;
            type = 'L';
            break;
         case 'S':
            --max;
            type = 'S';
      }

      long limit;
      if (this.charAt(this.np) == '-') {
         negative = true;
         limit = Long.MIN_VALUE;
         ++i;
      } else {
         limit = -9223372036854775807L;
      }

      long multmin = -922337203685477580L;
      if (i < max) {
         int digit = this.charAt(i++) - 48;
         result = (long)(-digit);
      }

      while(i < max) {
         int digit = this.charAt(i++) - 48;
         if (result < multmin) {
            return new BigInteger(this.numberString(), 10);
         }

         result *= 10L;
         if (result < limit + (long)digit) {
            return new BigInteger(this.numberString(), 10);
         }

         result -= (long)digit;
      }

      if (negative) {
         if (i > this.np + 1) {
            if (result >= -2147483648L && type != 'L') {
               if (type == 'S') {
                  return (short)((int)result);
               } else if (type == 'B') {
                  return (byte)((int)result);
               } else {
                  return (int)result;
               }
            } else {
               return result;
            }
         } else {
            throw new JSONException("illegal number format : " + this.numberString());
         }
      } else {
         result = -result;
         if (result <= 2147483647L && type != 'L') {
            if (type == 'S') {
               return (short)((int)result);
            } else if (type == 'B') {
               return (byte)((int)result);
            } else {
               return (int)result;
            }
         } else {
            return result;
         }
      }
   }

   public final void nextTokenWithColon(int expect) {
      this.nextTokenWithChar(':');
   }

   public float floatValue() {
      String strVal = this.numberString();
      float floatValue = Float.parseFloat(strVal);
      if (floatValue == 0.0F || floatValue == Float.POSITIVE_INFINITY) {
         char c0 = strVal.charAt(0);
         if (c0 > '0' && c0 <= '9') {
            throw new JSONException("float overflow : " + strVal);
         }
      }

      return floatValue;
   }

   public double doubleValue() {
      return Double.parseDouble(this.numberString());
   }

   public void config(Feature feature, boolean state) {
      this.features = Feature.config(this.features, feature, state);
      if ((this.features & Feature.InitStringFieldAsEmpty.mask) != 0) {
         this.stringDefaultValue = "";
      }

   }

   public final boolean isEnabled(Feature feature) {
      return this.isEnabled(feature.mask);
   }

   public final boolean isEnabled(int feature) {
      return (this.features & feature) != 0;
   }

   public final boolean isEnabled(int features, int feature) {
      return (this.features & feature) != 0 || (features & feature) != 0;
   }

   public abstract String numberString();

   public abstract boolean isEOF();

   public final char getCurrent() {
      return this.ch;
   }

   public abstract char charAt(int var1);

   public abstract char next();

   protected void skipComment() {
      this.next();
      if (this.ch != '/') {
         if (this.ch == '*') {
            this.next();

            while(this.ch != 26) {
               if (this.ch == '*') {
                  this.next();
                  if (this.ch == '/') {
                     this.next();
                     return;
                  }
               } else {
                  this.next();
               }
            }

         } else {
            throw new JSONException("invalid comment");
         }
      } else {
         do {
            this.next();
            if (this.ch == '\n') {
               this.next();
               return;
            }
         } while(this.ch != 26);

      }
   }

   public final String scanSymbol(SymbolTable symbolTable) {
      this.skipWhitespace();
      if (this.ch == '"') {
         return this.scanSymbol(symbolTable, '"');
      } else if (this.ch == '\'') {
         if (!this.isEnabled(Feature.AllowSingleQuotes)) {
            throw new JSONException("syntax error");
         } else {
            return this.scanSymbol(symbolTable, '\'');
         }
      } else if (this.ch == '}') {
         this.next();
         this.token = 13;
         return null;
      } else if (this.ch == ',') {
         this.next();
         this.token = 16;
         return null;
      } else if (this.ch == 26) {
         this.token = 20;
         return null;
      } else if (!this.isEnabled(Feature.AllowUnQuotedFieldNames)) {
         throw new JSONException("syntax error");
      } else {
         return this.scanSymbolUnQuoted(symbolTable);
      }
   }

   protected abstract void arrayCopy(int var1, char[] var2, int var3, int var4);

   public final String scanSymbol(SymbolTable symbolTable, char quote) {
      int hash = 0;
      this.np = this.bp;
      this.sp = 0;
      boolean hasSpecial = false;

      while(true) {
         char chLocal = this.next();
         if (chLocal == quote) {
            this.token = 4;
            String value;
            if (!hasSpecial) {
               int offset;
               if (this.np == -1) {
                  offset = 0;
               } else {
                  offset = this.np + 1;
               }

               value = this.addSymbol(offset, this.sp, hash, symbolTable);
            } else {
               value = symbolTable.addSymbol((char[])this.sbuf, 0, this.sp, hash);
            }

            this.sp = 0;
            this.next();
            return value;
         }

         if (chLocal == 26) {
            throw new JSONException("unclosed.str");
         }

         if (chLocal == '\\') {
            if (!hasSpecial) {
               hasSpecial = true;
               if (this.sp >= this.sbuf.length) {
                  int newCapcity = this.sbuf.length * 2;
                  if (this.sp > newCapcity) {
                     newCapcity = this.sp;
                  }

                  char[] newsbuf = new char[newCapcity];
                  System.arraycopy(this.sbuf, 0, newsbuf, 0, this.sbuf.length);
                  this.sbuf = newsbuf;
               }

               this.arrayCopy(this.np + 1, this.sbuf, 0, this.sp);
            }

            chLocal = this.next();
            switch (chLocal) {
               case '"':
                  hash = 31 * hash + 34;
                  this.putChar('"');
                  break;
               case '#':
               case '$':
               case '%':
               case '&':
               case '(':
               case ')':
               case '*':
               case '+':
               case ',':
               case '-':
               case '.':
               case '8':
               case '9':
               case ':':
               case ';':
               case '<':
               case '=':
               case '>':
               case '?':
               case '@':
               case 'A':
               case 'B':
               case 'C':
               case 'D':
               case 'E':
               case 'G':
               case 'H':
               case 'I':
               case 'J':
               case 'K':
               case 'L':
               case 'M':
               case 'N':
               case 'O':
               case 'P':
               case 'Q':
               case 'R':
               case 'S':
               case 'T':
               case 'U':
               case 'V':
               case 'W':
               case 'X':
               case 'Y':
               case 'Z':
               case '[':
               case ']':
               case '^':
               case '_':
               case '`':
               case 'a':
               case 'c':
               case 'd':
               case 'e':
               case 'g':
               case 'h':
               case 'i':
               case 'j':
               case 'k':
               case 'l':
               case 'm':
               case 'o':
               case 'p':
               case 'q':
               case 's':
               case 'w':
               default:
                  this.ch = chLocal;
                  throw new JSONException("unclosed.str.lit");
               case '\'':
                  hash = 31 * hash + 39;
                  this.putChar('\'');
                  break;
               case '/':
                  hash = 31 * hash + 47;
                  this.putChar('/');
                  break;
               case '0':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0000');
                  break;
               case '1':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0001');
                  break;
               case '2':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0002');
                  break;
               case '3':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0003');
                  break;
               case '4':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0004');
                  break;
               case '5':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0005');
                  break;
               case '6':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0006');
                  break;
               case '7':
                  hash = 31 * hash + chLocal;
                  this.putChar('\u0007');
                  break;
               case 'F':
               case 'f':
                  hash = 31 * hash + 12;
                  this.putChar('\f');
                  break;
               case '\\':
                  hash = 31 * hash + 92;
                  this.putChar('\\');
                  break;
               case 'b':
                  hash = 31 * hash + 8;
                  this.putChar('\b');
                  break;
               case 'n':
                  hash = 31 * hash + 10;
                  this.putChar('\n');
                  break;
               case 'r':
                  hash = 31 * hash + 13;
                  this.putChar('\r');
                  break;
               case 't':
                  hash = 31 * hash + 9;
                  this.putChar('\t');
                  break;
               case 'u':
                  char c1 = this.next();
                  char c2 = this.next();
                  char c3 = this.next();
                  char c4 = this.next();
                  int val = Integer.parseInt(new String(new char[]{c1, c2, c3, c4}), 16);
                  hash = 31 * hash + val;
                  this.putChar((char)val);
                  break;
               case 'v':
                  hash = 31 * hash + 11;
                  this.putChar('\u000b');
                  break;
               case 'x':
                  char x1 = this.ch = this.next();
                  char x2 = this.ch = this.next();
                  int x_val = digits[x1] * 16 + digits[x2];
                  char x_char = (char)x_val;
                  hash = 31 * hash + x_char;
                  this.putChar(x_char);
            }
         } else {
            hash = 31 * hash + chLocal;
            if (!hasSpecial) {
               ++this.sp;
            } else if (this.sp == this.sbuf.length) {
               this.putChar(chLocal);
            } else {
               this.sbuf[this.sp++] = chLocal;
            }
         }
      }
   }

   public final void resetStringPosition() {
      this.sp = 0;
   }

   public String info() {
      return "";
   }

   public final String scanSymbolUnQuoted(SymbolTable symbolTable) {
      if (this.token == 1 && this.pos == 0 && this.bp == 1) {
         this.bp = 0;
      }

      boolean[] firstIdentifierFlags = IOUtils.firstIdentifierFlags;
      char first = this.ch;
      boolean firstFlag = this.ch >= firstIdentifierFlags.length || firstIdentifierFlags[first];
      if (!firstFlag) {
         throw new JSONException("illegal identifier : " + this.ch + this.info());
      } else {
         boolean[] identifierFlags = IOUtils.identifierFlags;
         int hash = first;
         this.np = this.bp;
         this.sp = 1;

         while(true) {
            char chLocal = this.next();
            if (chLocal < identifierFlags.length && !identifierFlags[chLocal]) {
               this.ch = this.charAt(this.bp);
               this.token = 18;
               int NULL_HASH = 3392903;
               if (this.sp == 4 && hash == 3392903 && this.charAt(this.np) == 'n' && this.charAt(this.np + 1) == 'u' && this.charAt(this.np + 2) == 'l' && this.charAt(this.np + 3) == 'l') {
                  return null;
               }

               if (symbolTable == null) {
                  return this.subString(this.np, this.sp);
               }

               return this.addSymbol(this.np, this.sp, hash, symbolTable);
            }

            hash = 31 * hash + chLocal;
            ++this.sp;
         }
      }
   }

   protected abstract void copyTo(int var1, int var2, char[] var3);

   public final void scanString() {
      this.np = this.bp;
      this.hasSpecial = false;

      while(true) {
         char ch = this.next();
         if (ch == '"') {
            this.token = 4;
            this.ch = this.next();
            return;
         }

         if (ch != 26) {
            if (ch == '\\') {
               if (!this.hasSpecial) {
                  this.hasSpecial = true;
                  if (this.sp >= this.sbuf.length) {
                     int newCapcity = this.sbuf.length * 2;
                     if (this.sp > newCapcity) {
                        newCapcity = this.sp;
                     }

                     char[] newsbuf = new char[newCapcity];
                     System.arraycopy(this.sbuf, 0, newsbuf, 0, this.sbuf.length);
                     this.sbuf = newsbuf;
                  }

                  this.copyTo(this.np + 1, this.sp, this.sbuf);
               }

               ch = this.next();
               switch (ch) {
                  case '"':
                     this.putChar('"');
                     continue;
                  case '#':
                  case '$':
                  case '%':
                  case '&':
                  case '(':
                  case ')':
                  case '*':
                  case '+':
                  case ',':
                  case '-':
                  case '.':
                  case '8':
                  case '9':
                  case ':':
                  case ';':
                  case '<':
                  case '=':
                  case '>':
                  case '?':
                  case '@':
                  case 'A':
                  case 'B':
                  case 'C':
                  case 'D':
                  case 'E':
                  case 'G':
                  case 'H':
                  case 'I':
                  case 'J':
                  case 'K':
                  case 'L':
                  case 'M':
                  case 'N':
                  case 'O':
                  case 'P':
                  case 'Q':
                  case 'R':
                  case 'S':
                  case 'T':
                  case 'U':
                  case 'V':
                  case 'W':
                  case 'X':
                  case 'Y':
                  case 'Z':
                  case '[':
                  case ']':
                  case '^':
                  case '_':
                  case '`':
                  case 'a':
                  case 'c':
                  case 'd':
                  case 'e':
                  case 'g':
                  case 'h':
                  case 'i':
                  case 'j':
                  case 'k':
                  case 'l':
                  case 'm':
                  case 'o':
                  case 'p':
                  case 'q':
                  case 's':
                  case 'w':
                  default:
                     this.ch = ch;
                     throw new JSONException("unclosed string : " + ch);
                  case '\'':
                     this.putChar('\'');
                     continue;
                  case '/':
                     this.putChar('/');
                     continue;
                  case '0':
                     this.putChar('\u0000');
                     continue;
                  case '1':
                     this.putChar('\u0001');
                     continue;
                  case '2':
                     this.putChar('\u0002');
                     continue;
                  case '3':
                     this.putChar('\u0003');
                     continue;
                  case '4':
                     this.putChar('\u0004');
                     continue;
                  case '5':
                     this.putChar('\u0005');
                     continue;
                  case '6':
                     this.putChar('\u0006');
                     continue;
                  case '7':
                     this.putChar('\u0007');
                     continue;
                  case 'F':
                  case 'f':
                     this.putChar('\f');
                     continue;
                  case '\\':
                     this.putChar('\\');
                     continue;
                  case 'b':
                     this.putChar('\b');
                     continue;
                  case 'n':
                     this.putChar('\n');
                     continue;
                  case 'r':
                     this.putChar('\r');
                     continue;
                  case 't':
                     this.putChar('\t');
                     continue;
                  case 'u':
                     char u1 = this.next();
                     char u2 = this.next();
                     char u3 = this.next();
                     char u4 = this.next();
                     int val = Integer.parseInt(new String(new char[]{u1, u2, u3, u4}), 16);
                     this.putChar((char)val);
                     continue;
                  case 'v':
                     this.putChar('\u000b');
                     continue;
                  case 'x':
               }

               char x1 = this.next();
               char x2 = this.next();
               boolean hex1 = x1 >= '0' && x1 <= '9' || x1 >= 'a' && x1 <= 'f' || x1 >= 'A' && x1 <= 'F';
               boolean hex2 = x2 >= '0' && x2 <= '9' || x2 >= 'a' && x2 <= 'f' || x2 >= 'A' && x2 <= 'F';
               if (!hex1 || !hex2) {
                  throw new JSONException("invalid escape character \\x" + x1 + x2);
               }

               char x_char = (char)(digits[x1] * 16 + digits[x2]);
               this.putChar(x_char);
            } else if (!this.hasSpecial) {
               ++this.sp;
            } else if (this.sp == this.sbuf.length) {
               this.putChar(ch);
            } else {
               this.sbuf[this.sp++] = ch;
            }
         } else {
            if (this.isEOF()) {
               throw new JSONException("unclosed string : " + ch);
            }

            this.putChar('\u001a');
         }
      }
   }

   public Calendar getCalendar() {
      return this.calendar;
   }

   public TimeZone getTimeZone() {
      return this.timeZone;
   }

   public void setTimeZone(TimeZone timeZone) {
      this.timeZone = timeZone;
   }

   public Locale getLocale() {
      return this.locale;
   }

   public void setLocale(Locale locale) {
      this.locale = locale;
   }

   public final int intValue() {
      if (this.np == -1) {
         this.np = 0;
      }

      int result = 0;
      boolean negative = false;
      int i = this.np;
      int max = this.np + this.sp;
      int limit;
      if (this.charAt(this.np) == '-') {
         negative = true;
         limit = Integer.MIN_VALUE;
         ++i;
      } else {
         limit = -2147483647;
      }

      long multmin = -214748364L;
      if (i < max) {
         int digit = this.charAt(i++) - 48;
         result = -digit;
      }

      while(i < max) {
         char chLocal = this.charAt(i++);
         if (chLocal == 'L' || chLocal == 'S' || chLocal == 'B') {
            break;
         }

         int digit = chLocal - 48;
         if ((long)result < multmin) {
            throw new NumberFormatException(this.numberString());
         }

         result *= 10;
         if (result < limit + digit) {
            throw new NumberFormatException(this.numberString());
         }

         result -= digit;
      }

      if (negative) {
         if (i > this.np + 1) {
            return result;
         } else {
            throw new NumberFormatException(this.numberString());
         }
      } else {
         return -result;
      }
   }

   public abstract byte[] bytesValue();

   public void close() {
      if (this.sbuf.length <= 8192) {
         SBUF_LOCAL.set(this.sbuf);
      }

      this.sbuf = null;
   }

   public final boolean isRef() {
      if (this.sp != 4) {
         return false;
      } else {
         return this.charAt(this.np + 1) == '$' && this.charAt(this.np + 2) == 'r' && this.charAt(this.np + 3) == 'e' && this.charAt(this.np + 4) == 'f';
      }
   }

   public String scanTypeName(SymbolTable symbolTable) {
      return null;
   }

   public final int scanType(String type) {
      this.matchStat = 0;
      if (!this.charArrayCompare(typeFieldName)) {
         return -2;
      } else {
         int bpLocal = this.bp + typeFieldName.length;
         int typeLength = type.length();

         for(int i = 0; i < typeLength; ++i) {
            if (type.charAt(i) != this.charAt(bpLocal + i)) {
               return -1;
            }
         }

         bpLocal += typeLength;
         if (this.charAt(bpLocal) != '"') {
            return -1;
         } else {
            ++bpLocal;
            this.ch = this.charAt(bpLocal);
            if (this.ch == ',') {
               ++bpLocal;
               this.ch = this.charAt(bpLocal);
               this.bp = bpLocal;
               this.token = 16;
               return 3;
            } else {
               if (this.ch == '}') {
                  ++bpLocal;
                  this.ch = this.charAt(bpLocal);
                  if (this.ch == ',') {
                     this.token = 16;
                     ++bpLocal;
                     this.ch = this.charAt(bpLocal);
                  } else if (this.ch == ']') {
                     this.token = 15;
                     ++bpLocal;
                     this.ch = this.charAt(bpLocal);
                  } else if (this.ch == '}') {
                     this.token = 13;
                     ++bpLocal;
                     this.ch = this.charAt(bpLocal);
                  } else {
                     if (this.ch != 26) {
                        return -1;
                     }

                     this.token = 20;
                  }

                  this.matchStat = 4;
               }

               this.bp = bpLocal;
               return this.matchStat;
            }
         }
      }
   }

   public final boolean matchField(char[] fieldName) {
      while(true) {
         if (!this.charArrayCompare(fieldName)) {
            if (isWhitespace(this.ch)) {
               this.next();
               continue;
            }

            return false;
         }

         this.bp += fieldName.length;
         this.ch = this.charAt(this.bp);
         if (this.ch == '{') {
            this.next();
            this.token = 12;
         } else if (this.ch == '[') {
            this.next();
            this.token = 14;
         } else if (this.ch == 'S' && this.charAt(this.bp + 1) == 'e' && this.charAt(this.bp + 2) == 't' && this.charAt(this.bp + 3) == '[') {
            this.bp += 3;
            this.ch = this.charAt(this.bp);
            this.token = 21;
         } else {
            this.nextToken();
         }

         return true;
      }
   }

   public int matchField(long fieldNameHash) {
      throw new UnsupportedOperationException();
   }

   public boolean seekArrayToItem(int index) {
      throw new UnsupportedOperationException();
   }

   public int seekObjectToField(long fieldNameHash, boolean deepScan) {
      throw new UnsupportedOperationException();
   }

   public int seekObjectToField(long[] fieldNameHash) {
      throw new UnsupportedOperationException();
   }

   public int seekObjectToFieldDeepScan(long fieldNameHash) {
      throw new UnsupportedOperationException();
   }

   public void skipObject() {
      throw new UnsupportedOperationException();
   }

   public void skipObject(boolean valid) {
      throw new UnsupportedOperationException();
   }

   public void skipArray() {
      throw new UnsupportedOperationException();
   }

   public abstract int indexOf(char var1, int var2);

   public abstract String addSymbol(int var1, int var2, int var3, SymbolTable var4);

   public String scanFieldString(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return this.stringDefaultValue();
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '"') {
            this.matchStat = -1;
            return this.stringDefaultValue();
         } else {
            int startIndex = this.bp + fieldName.length + 1;
            int endIndex = this.indexOf('"', startIndex);
            if (endIndex == -1) {
               throw new JSONException("unclosed str");
            } else {
               int startIndex2 = this.bp + fieldName.length + 1;
               String stringVal = this.subString(startIndex2, endIndex - startIndex2);
               if (stringVal.indexOf(92) != -1) {
                  while(true) {
                     int slashCount = 0;

                     for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                        ++slashCount;
                     }

                     if (slashCount % 2 == 0) {
                        slashCount = endIndex - (this.bp + fieldName.length + 1);
                        char[] chars = this.sub_chars(this.bp + fieldName.length + 1, slashCount);
                        stringVal = readString(chars, slashCount);
                        break;
                     }

                     endIndex = this.indexOf('"', endIndex + 1);
                  }
               }

               offset += endIndex - (this.bp + fieldName.length + 1) + 1;
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 3;
                  return stringVal;
               } else if (chLocal == '}') {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == ',') {
                     this.token = 16;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else if (chLocal == ']') {
                     this.token = 15;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else if (chLocal == '}') {
                     this.token = 13;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else {
                     if (chLocal != 26) {
                        this.matchStat = -1;
                        return this.stringDefaultValue();
                     }

                     this.token = 20;
                     this.bp += offset - 1;
                     this.ch = 26;
                  }

                  this.matchStat = 4;
                  return stringVal;
               } else {
                  this.matchStat = -1;
                  return this.stringDefaultValue();
               }
            }
         }
      }
   }

   public String scanString(char expectNextChar) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      if (chLocal == 'n') {
         if (this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == expectNextChar) {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               return null;
            } else {
               this.matchStat = -1;
               return null;
            }
         } else {
            this.matchStat = -1;
            return null;
         }
      } else {
         while(chLocal != '"') {
            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return this.stringDefaultValue();
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         int startIndex = this.bp + offset;
         int endIndex = this.indexOf('"', startIndex);
         if (endIndex == -1) {
            throw new JSONException("unclosed str");
         } else {
            String stringVal = this.subString(this.bp + offset, endIndex - startIndex);
            if (stringVal.indexOf(92) != -1) {
               while(true) {
                  int slashCount = 0;

                  for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                     ++slashCount;
                  }

                  if (slashCount % 2 == 0) {
                     slashCount = endIndex - startIndex;
                     char[] chars = this.sub_chars(this.bp + 1, slashCount);
                     stringVal = readString(chars, slashCount);
                     break;
                  }

                  endIndex = this.indexOf('"', endIndex + 1);
               }
            }

            offset += endIndex - startIndex + 1;

            for(char var15 = this.charAt(this.bp + offset++); var15 != expectNextChar; var15 = this.charAt(this.bp + offset++)) {
               if (!isWhitespace(var15)) {
                  if (var15 == ']') {
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = -1;
                  }

                  return stringVal;
               }
            }

            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 3;
            this.token = 16;
            return stringVal;
         }
      }
   }

   public long scanFieldSymbol(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return 0L;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '"') {
            this.matchStat = -1;
            return 0L;
         } else {
            long hash = -3750763034362895579L;

            do {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == '"') {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == ',') {
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = 3;
                     return hash;
                  } else if (chLocal == '}') {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == ',') {
                        this.token = 16;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == ']') {
                        this.token = 15;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == '}') {
                        this.token = 13;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else {
                        if (chLocal != 26) {
                           this.matchStat = -1;
                           return 0L;
                        }

                        this.token = 20;
                        this.bp += offset - 1;
                        this.ch = 26;
                     }

                     this.matchStat = 4;
                     return hash;
                  } else {
                     this.matchStat = -1;
                     return 0L;
                  }
               }

               hash ^= (long)chLocal;
               hash *= 1099511628211L;
            } while(chLocal != '\\');

            this.matchStat = -1;
            return 0L;
         }
      }
   }

   public long scanEnumSymbol(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return 0L;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '"') {
            this.matchStat = -1;
            return 0L;
         } else {
            long hash = -3750763034362895579L;

            do {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == '"') {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == ',') {
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = 3;
                     return hash;
                  }

                  if (chLocal == '}') {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == ',') {
                        this.token = 16;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == ']') {
                        this.token = 15;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == '}') {
                        this.token = 13;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else {
                        if (chLocal != 26) {
                           this.matchStat = -1;
                           return 0L;
                        }

                        this.token = 20;
                        this.bp += offset - 1;
                        this.ch = 26;
                     }

                     this.matchStat = 4;
                     return hash;
                  }

                  this.matchStat = -1;
                  return 0L;
               }

               hash ^= (long)(chLocal >= 'A' && chLocal <= 'Z' ? chLocal + 32 : chLocal);
               hash *= 1099511628211L;
            } while(chLocal != '\\');

            this.matchStat = -1;
            return 0L;
         }
      }
   }

   public Enum<?> scanEnum(Class<?> enumClass, SymbolTable symbolTable, char serperator) {
      String name = this.scanSymbolWithSeperator(symbolTable, serperator);
      return name == null ? null : (Enum<?>) Enum.valueOf((Class<? extends Enum>) enumClass, name);
   }

   public String scanSymbolWithSeperator(SymbolTable symbolTable, char serperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      if (chLocal == 'n') {
         if (this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == serperator) {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               return null;
            } else {
               this.matchStat = -1;
               return null;
            }
         } else {
            this.matchStat = -1;
            return null;
         }
      } else if (chLocal != '"') {
         this.matchStat = -1;
         return null;
      } else {
         int hash = 0;

         do {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == '"') {
               int start = this.bp + 0 + 1;
               int len = this.bp + offset - start - 1;
               String strVal = this.addSymbol(start, len, hash, symbolTable);

               for(char var14 = this.charAt(this.bp + offset++); var14 != serperator; var14 = this.charAt(this.bp + offset++)) {
                  if (!isWhitespace(var14)) {
                     this.matchStat = -1;
                     return strVal;
                  }
               }

               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               return strVal;
            }

            hash = 31 * hash + chLocal;
         } while(chLocal != '\\');

         this.matchStat = -1;
         return null;
      }
   }

   public Collection<String> newCollectionByType(Class<?> type) {
      if (type.isAssignableFrom(HashSet.class)) {
         return new HashSet();
      } else if (type.isAssignableFrom(ArrayList.class)) {
         return new ArrayList();
      } else if (type.isAssignableFrom(LinkedList.class)) {
         return new LinkedList();
      } else {
         try {
            return (Collection)type.newInstance();
         } catch (Exception e) {
            throw new JSONException(e.getMessage(), e);
         }
      }
   }

   public Collection<String> scanFieldStringArray(char[] fieldName, Class<?> type) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         Collection<String> list = this.newCollectionByType(type);
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '[') {
            this.matchStat = -1;
            return null;
         } else {
            chLocal = this.charAt(this.bp + offset++);

            while(true) {
               if (chLocal == '"') {
                  int startIndex = this.bp + offset;
                  int endIndex = this.indexOf('"', startIndex);
                  if (endIndex == -1) {
                     throw new JSONException("unclosed str");
                  }

                  int startIndex2 = this.bp + offset;
                  String stringVal = this.subString(startIndex2, endIndex - startIndex2);
                  if (stringVal.indexOf(92) != -1) {
                     while(true) {
                        int slashCount = 0;

                        for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                           ++slashCount;
                        }

                        if (slashCount % 2 == 0) {
                           slashCount = endIndex - (this.bp + offset);
                           char[] chars = this.sub_chars(this.bp + offset, slashCount);
                           stringVal = readString(chars, slashCount);
                           break;
                        }

                        endIndex = this.indexOf('"', endIndex + 1);
                     }
                  }

                  offset += endIndex - (this.bp + offset) + 1;
                  chLocal = this.charAt(this.bp + offset++);
                  list.add(stringVal);
               } else {
                  if (chLocal != 'n' || this.charAt(this.bp + offset) != 'u' || this.charAt(this.bp + offset + 1) != 'l' || this.charAt(this.bp + offset + 2) != 'l') {
                     if (chLocal != ']' || list.size() != 0) {
                        throw new JSONException("illega str");
                     }

                     chLocal = this.charAt(this.bp + offset++);
                     break;
                  }

                  offset += 3;
                  chLocal = this.charAt(this.bp + offset++);
                  list.add(null);
               }

               if (chLocal != ',') {
                  if (chLocal != ']') {
                     this.matchStat = -1;
                     return null;
                  }

                  chLocal = this.charAt(this.bp + offset++);
                  break;
               }

               chLocal = this.charAt(this.bp + offset++);
            }

            if (chLocal == ',') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               return list;
            } else if (chLocal == '}') {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.token = 16;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == ']') {
                  this.token = 15;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == '}') {
                  this.token = 13;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else {
                  if (chLocal != 26) {
                     this.matchStat = -1;
                     return null;
                  }

                  this.bp += offset - 1;
                  this.token = 20;
                  this.ch = 26;
               }

               this.matchStat = 4;
               return list;
            } else {
               this.matchStat = -1;
               return null;
            }
         }
      }
   }

   public void scanStringArray(Collection<String> list, char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l' && this.charAt(this.bp + offset + 3) == seperator) {
         this.bp += 5;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
      } else if (chLocal != '[') {
         this.matchStat = -1;
      } else {
         chLocal = this.charAt(this.bp + offset++);

         while(true) {
            if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
               offset += 3;
               chLocal = this.charAt(this.bp + offset++);
               list.add(null);
            } else {
               if (chLocal == ']' && list.size() == 0) {
                  chLocal = this.charAt(this.bp + offset++);
                  break;
               }

               if (chLocal != '"') {
                  this.matchStat = -1;
                  return;
               }

               int startIndex = this.bp + offset;
               int endIndex = this.indexOf('"', startIndex);
               if (endIndex == -1) {
                  throw new JSONException("unclosed str");
               }

               String stringVal = this.subString(this.bp + offset, endIndex - startIndex);
               if (stringVal.indexOf(92) != -1) {
                  while(true) {
                     int slashCount = 0;

                     for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                        ++slashCount;
                     }

                     if (slashCount % 2 == 0) {
                        slashCount = endIndex - startIndex;
                        char[] chars = this.sub_chars(this.bp + offset, slashCount);
                        stringVal = readString(chars, slashCount);
                        break;
                     }

                     endIndex = this.indexOf('"', endIndex + 1);
                  }
               }

               offset += endIndex - (this.bp + offset) + 1;
               chLocal = this.charAt(this.bp + offset++);
               list.add(stringVal);
            }

            if (chLocal != ',') {
               if (chLocal != ']') {
                  this.matchStat = -1;
                  return;
               }

               chLocal = this.charAt(this.bp + offset++);
               break;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal == seperator) {
            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 3;
         } else {
            this.matchStat = -1;
         }
      }
   }

   public int scanFieldInt(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return 0;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean negative = chLocal == '-';
         if (negative) {
            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal >= '0' && chLocal <= '9') {
            int value = chLocal - 48;

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  if (chLocal == '.') {
                     this.matchStat = -1;
                     return 0;
                  } else if (value >= 0 && offset <= 14 + fieldName.length || value == Integer.MIN_VALUE && offset == 17 && negative) {
                     if (chLocal == ',') {
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                        this.matchStat = 3;
                        this.token = 16;
                        return negative ? -value : value;
                     } else if (chLocal == '}') {
                        chLocal = this.charAt(this.bp + offset++);
                        if (chLocal == ',') {
                           this.token = 16;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else if (chLocal == ']') {
                           this.token = 15;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else if (chLocal == '}') {
                           this.token = 13;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else {
                           if (chLocal != 26) {
                              this.matchStat = -1;
                              return 0;
                           }

                           this.token = 20;
                           this.bp += offset - 1;
                           this.ch = 26;
                        }

                        this.matchStat = 4;
                        return negative ? -value : value;
                     } else {
                        this.matchStat = -1;
                        return 0;
                     }
                  } else {
                     this.matchStat = -1;
                     return 0;
                  }
               }

               value = value * 10 + (chLocal - 48);
            }
         } else {
            this.matchStat = -1;
            return 0;
         }
      }
   }

   public final int[] scanFieldIntArray(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '[') {
            this.matchStat = -2;
            return null;
         } else {
            chLocal = this.charAt(this.bp + offset++);
            int[] array = new int[16];
            int arrayIndex = 0;
            if (chLocal == ']') {
               chLocal = this.charAt(this.bp + offset++);
            } else {
               label81:
               while(true) {
                  boolean nagative = false;
                  if (chLocal == '-') {
                     chLocal = this.charAt(this.bp + offset++);
                     nagative = true;
                  }

                  if (chLocal < '0' || chLocal > '9') {
                     this.matchStat = -1;
                     return null;
                  }

                  int value = chLocal - 48;

                  while(true) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        if (arrayIndex >= array.length) {
                           int[] tmp = new int[array.length * 3 / 2];
                           System.arraycopy(array, 0, tmp, 0, arrayIndex);
                           array = tmp;
                        }

                        array[arrayIndex++] = nagative ? -value : value;
                        if (chLocal == ',') {
                           chLocal = this.charAt(this.bp + offset++);
                        } else if (chLocal == ']') {
                           chLocal = this.charAt(this.bp + offset++);
                           break label81;
                        }
                        break;
                     }

                     value = value * 10 + (chLocal - 48);
                  }
               }
            }

            if (arrayIndex != array.length) {
               int[] tmp = new int[arrayIndex];
               System.arraycopy(array, 0, tmp, 0, arrayIndex);
               array = tmp;
            }

            if (chLocal == ',') {
               this.bp += offset - 1;
               this.next();
               this.matchStat = 3;
               this.token = 16;
               return array;
            } else if (chLocal == '}') {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.token = 16;
                  this.bp += offset - 1;
                  this.next();
               } else if (chLocal == ']') {
                  this.token = 15;
                  this.bp += offset - 1;
                  this.next();
               } else if (chLocal == '}') {
                  this.token = 13;
                  this.bp += offset - 1;
                  this.next();
               } else {
                  if (chLocal != 26) {
                     this.matchStat = -1;
                     return null;
                  }

                  this.bp += offset - 1;
                  this.token = 20;
                  this.ch = 26;
               }

               this.matchStat = 4;
               return array;
            } else {
               this.matchStat = -1;
               return null;
            }
         }
      }
   }

   public boolean scanBoolean(char expectNext) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean value = false;
      if (chLocal == 't') {
         if (this.charAt(this.bp + offset) != 'r' || this.charAt(this.bp + offset + 1) != 'u' || this.charAt(this.bp + offset + 2) != 'e') {
            this.matchStat = -1;
            return false;
         }

         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         value = true;
      } else if (chLocal == 'f') {
         if (this.charAt(this.bp + offset) != 'a' || this.charAt(this.bp + offset + 1) != 'l' || this.charAt(this.bp + offset + 2) != 's' || this.charAt(this.bp + offset + 3) != 'e') {
            this.matchStat = -1;
            return false;
         }

         offset += 4;
         chLocal = this.charAt(this.bp + offset++);
         value = false;
      } else if (chLocal == '1') {
         chLocal = this.charAt(this.bp + offset++);
         value = true;
      } else if (chLocal == '0') {
         chLocal = this.charAt(this.bp + offset++);
         value = false;
      }

      while(chLocal != expectNext) {
         if (!isWhitespace(chLocal)) {
            this.matchStat = -1;
            return value;
         }

         chLocal = this.charAt(this.bp + offset++);
      }

      this.bp += offset;
      this.ch = this.charAt(this.bp);
      this.matchStat = 3;
      return value;
   }

   public int scanInt(char expectNext) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean quote = chLocal == '"';
      if (quote) {
         chLocal = this.charAt(this.bp + offset++);
      }

      boolean negative = chLocal == '-';
      if (negative) {
         chLocal = this.charAt(this.bp + offset++);
      }

      if (chLocal >= '0' && chLocal <= '9') {
         int value = chLocal - 48;

         while(true) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal < '0' || chLocal > '9') {
               if (chLocal == '.') {
                  this.matchStat = -1;
                  return 0;
               } else if (value < 0) {
                  this.matchStat = -1;
                  return 0;
               } else {
                  while(chLocal != expectNext) {
                     if (!isWhitespace(chLocal)) {
                        this.matchStat = -1;
                        return negative ? -value : value;
                     }

                     chLocal = this.charAt(this.bp + offset++);
                  }

                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 3;
                  this.token = 16;
                  return negative ? -value : value;
               }
            }

            value = value * 10 + (chLocal - 48);
         }
      } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
         this.matchStat = 5;
         int value = 0;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         if (quote && chLocal == '"') {
            chLocal = this.charAt(this.bp + offset++);
         }

         while(chLocal != ',') {
            if (chLocal == ']') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 5;
               this.token = 15;
               return value;
            }

            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return 0;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
         this.token = 16;
         return value;
      } else {
         this.matchStat = -1;
         return 0;
      }
   }

   public boolean scanFieldBoolean(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return false;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean value;
         if (chLocal == 't') {
            if (this.charAt(this.bp + offset++) != 'r') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 'u') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 'e') {
               this.matchStat = -1;
               return false;
            }

            value = true;
         } else {
            if (chLocal != 'f') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 'a') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 'l') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 's') {
               this.matchStat = -1;
               return false;
            }

            if (this.charAt(this.bp + offset++) != 'e') {
               this.matchStat = -1;
               return false;
            }

            value = false;
         }

         chLocal = this.charAt(this.bp + offset++);
         if (chLocal == ',') {
            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 3;
            this.token = 16;
            return value;
         } else if (chLocal == '}') {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == ',') {
               this.token = 16;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == ']') {
               this.token = 15;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == '}') {
               this.token = 13;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else {
               if (chLocal != 26) {
                  this.matchStat = -1;
                  return false;
               }

               this.token = 20;
               this.bp += offset - 1;
               this.ch = 26;
            }

            this.matchStat = 4;
            return value;
         } else {
            this.matchStat = -1;
            return false;
         }
      }
   }

   public long scanFieldLong(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return 0L;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean negative = false;
         if (chLocal == '-') {
            chLocal = this.charAt(this.bp + offset++);
            negative = true;
         }

         if (chLocal >= '0' && chLocal <= '9') {
            long value = (long)(chLocal - 48);

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  if (chLocal == '.') {
                     this.matchStat = -1;
                     return 0L;
                  } else {
                     boolean valid = offset - fieldName.length < 21 && (value >= 0L || value == Long.MIN_VALUE && negative);
                     if (!valid) {
                        this.matchStat = -1;
                        return 0L;
                     } else if (chLocal == ',') {
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                        this.matchStat = 3;
                        this.token = 16;
                        return negative ? -value : value;
                     } else if (chLocal == '}') {
                        chLocal = this.charAt(this.bp + offset++);
                        if (chLocal == ',') {
                           this.token = 16;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else if (chLocal == ']') {
                           this.token = 15;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else if (chLocal == '}') {
                           this.token = 13;
                           this.bp += offset;
                           this.ch = this.charAt(this.bp);
                        } else {
                           if (chLocal != 26) {
                              this.matchStat = -1;
                              return 0L;
                           }

                           this.token = 20;
                           this.bp += offset - 1;
                           this.ch = 26;
                        }

                        this.matchStat = 4;
                        return negative ? -value : value;
                     } else {
                        this.matchStat = -1;
                        return 0L;
                     }
                  }
               }

               value = value * 10L + (long)(chLocal - 48);
            }
         } else {
            this.matchStat = -1;
            return 0L;
         }
      }
   }

   public long scanLong(char expectNextChar) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean quote = chLocal == '"';
      if (quote) {
         chLocal = this.charAt(this.bp + offset++);
      }

      boolean negative = chLocal == '-';
      if (negative) {
         chLocal = this.charAt(this.bp + offset++);
      }

      if (chLocal >= '0' && chLocal <= '9') {
         long value = (long)(chLocal - 48);

         while(true) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal < '0' || chLocal > '9') {
               if (chLocal == '.') {
                  this.matchStat = -1;
                  return 0L;
               } else {
                  boolean valid = value >= 0L || value == Long.MIN_VALUE && negative;
                  if (!valid) {
                     String val = this.subString(this.bp, offset - 1);
                     throw new NumberFormatException(val);
                  } else {
                     if (quote) {
                        if (chLocal != '"') {
                           this.matchStat = -1;
                           return 0L;
                        }

                        chLocal = this.charAt(this.bp + offset++);
                     }

                     while(chLocal != expectNextChar) {
                        if (!isWhitespace(chLocal)) {
                           this.matchStat = -1;
                           return value;
                        }

                        chLocal = this.charAt(this.bp + offset++);
                     }

                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = 3;
                     this.token = 16;
                     return negative ? -value : value;
                  }
               }
            }

            value = value * 10L + (long)(chLocal - 48);
         }
      } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
         this.matchStat = 5;
         long value = 0L;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         if (quote && chLocal == '"') {
            chLocal = this.charAt(this.bp + offset++);
         }

         while(chLocal != ',') {
            if (chLocal == ']') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 5;
               this.token = 15;
               return value;
            }

            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return 0L;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
         this.token = 16;
         return value;
      } else {
         this.matchStat = -1;
         return 0L;
      }
   }

   public final float scanFieldFloat(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return 0.0F;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean quote = chLocal == '"';
         if (quote) {
            chLocal = this.charAt(this.bp + offset++);
         }

         boolean negative = chLocal == '-';
         if (negative) {
            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal >= '0' && chLocal <= '9') {
            long intVal = (long)(chLocal - 48);

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  long power = 1L;
                  boolean small = chLocal == '.';
                  if (small) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        this.matchStat = -1;
                        return 0.0F;
                     }

                     intVal = intVal * 10L + (long)(chLocal - 48);
                     power = 10L;

                     while(true) {
                        chLocal = this.charAt(this.bp + offset++);
                        if (chLocal < '0' || chLocal > '9') {
                           break;
                        }

                        intVal = intVal * 10L + (long)(chLocal - 48);
                        power *= 10L;
                     }
                  }

                  boolean exp = chLocal == 'e' || chLocal == 'E';
                  if (exp) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == '+' || chLocal == '-') {
                        chLocal = this.charAt(this.bp + offset++);
                     }

                     while(chLocal >= '0' && chLocal <= '9') {
                        chLocal = this.charAt(this.bp + offset++);
                     }
                  }

                  int start;
                  int count;
                  if (quote) {
                     if (chLocal != '"') {
                        this.matchStat = -1;
                        return 0.0F;
                     }

                     chLocal = this.charAt(this.bp + offset++);
                     start = this.bp + fieldName.length + 1;
                     count = this.bp + offset - start - 2;
                  } else {
                     start = this.bp + fieldName.length;
                     count = this.bp + offset - start - 1;
                  }

                  float value;
                  if (!exp && count < 17) {
                     value = (float)((double)intVal / (double)power);
                     if (negative) {
                        value = -value;
                     }
                  } else {
                     String text = this.subString(start, count);
                     value = Float.parseFloat(text);
                  }

                  if (chLocal == ',') {
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = 3;
                     this.token = 16;
                     return value;
                  }

                  if (chLocal == '}') {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == ',') {
                        this.token = 16;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == ']') {
                        this.token = 15;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == '}') {
                        this.token = 13;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else {
                        if (chLocal != 26) {
                           this.matchStat = -1;
                           return 0.0F;
                        }

                        this.bp += offset - 1;
                        this.token = 20;
                        this.ch = 26;
                     }

                     this.matchStat = 4;
                     return value;
                  }

                  this.matchStat = -1;
                  return 0.0F;
               }

               intVal = intVal * 10L + (long)(chLocal - 48);
            }
         } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            this.matchStat = 5;
            float value = 0.0F;
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (quote && chLocal == '"') {
               chLocal = this.charAt(this.bp + offset++);
            }

            while(chLocal != ',') {
               if (chLocal == '}') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 5;
                  this.token = 13;
                  return value;
               }

               if (!isWhitespace(chLocal)) {
                  this.matchStat = -1;
                  return 0.0F;
               }

               chLocal = this.charAt(this.bp + offset++);
            }

            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 5;
            this.token = 16;
            return value;
         } else {
            this.matchStat = -1;
            return 0.0F;
         }
      }
   }

   public final float scanFloat(char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean quote = chLocal == '"';
      if (quote) {
         chLocal = this.charAt(this.bp + offset++);
      }

      boolean negative = chLocal == '-';
      if (negative) {
         chLocal = this.charAt(this.bp + offset++);
      }

      if (chLocal >= '0' && chLocal <= '9') {
         long intVal = (long)(chLocal - 48);

         while(true) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal < '0' || chLocal > '9') {
               long power = 1L;
               boolean small = chLocal == '.';
               if (small) {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal < '0' || chLocal > '9') {
                     this.matchStat = -1;
                     return 0.0F;
                  }

                  intVal = intVal * 10L + (long)(chLocal - 48);
                  power = 10L;

                  while(true) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        break;
                     }

                     intVal = intVal * 10L + (long)(chLocal - 48);
                     power *= 10L;
                  }
               }

               boolean exp = chLocal == 'e' || chLocal == 'E';
               if (exp) {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == '+' || chLocal == '-') {
                     chLocal = this.charAt(this.bp + offset++);
                  }

                  while(chLocal >= '0' && chLocal <= '9') {
                     chLocal = this.charAt(this.bp + offset++);
                  }
               }

               int start;
               int count;
               if (quote) {
                  if (chLocal != '"') {
                     this.matchStat = -1;
                     return 0.0F;
                  }

                  chLocal = this.charAt(this.bp + offset++);
                  start = this.bp + 1;
                  count = this.bp + offset - start - 2;
               } else {
                  start = this.bp;
                  count = this.bp + offset - start - 1;
               }

               float value;
               if (!exp && count < 17) {
                  value = (float)((double)intVal / (double)power);
                  if (negative) {
                     value = -value;
                  }
               } else {
                  String text = this.subString(start, count);
                  value = Float.parseFloat(text);
               }

               if (chLocal == seperator) {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 3;
                  this.token = 16;
                  return value;
               }

               this.matchStat = -1;
               return value;
            }

            intVal = intVal * 10L + (long)(chLocal - 48);
         }
      } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
         this.matchStat = 5;
         float value = 0.0F;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         if (quote && chLocal == '"') {
            chLocal = this.charAt(this.bp + offset++);
         }

         while(chLocal != ',') {
            if (chLocal == ']') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 5;
               this.token = 15;
               return value;
            }

            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return 0.0F;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
         this.token = 16;
         return value;
      } else {
         this.matchStat = -1;
         return 0.0F;
      }
   }

   public double scanDouble(char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean quote = chLocal == '"';
      if (quote) {
         chLocal = this.charAt(this.bp + offset++);
      }

      boolean negative = chLocal == '-';
      if (negative) {
         chLocal = this.charAt(this.bp + offset++);
      }

      if (chLocal >= '0' && chLocal <= '9') {
         long intVal = (long)(chLocal - 48);

         while(true) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal < '0' || chLocal > '9') {
               long power = 1L;
               boolean small = chLocal == '.';
               if (small) {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal < '0' || chLocal > '9') {
                     this.matchStat = -1;
                     return (double)0.0F;
                  }

                  intVal = intVal * 10L + (long)(chLocal - 48);
                  power = 10L;

                  while(true) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        break;
                     }

                     intVal = intVal * 10L + (long)(chLocal - 48);
                     power *= 10L;
                  }
               }

               boolean exp = chLocal == 'e' || chLocal == 'E';
               if (exp) {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == '+' || chLocal == '-') {
                     chLocal = this.charAt(this.bp + offset++);
                  }

                  while(chLocal >= '0' && chLocal <= '9') {
                     chLocal = this.charAt(this.bp + offset++);
                  }
               }

               int start;
               int count;
               if (quote) {
                  if (chLocal != '"') {
                     this.matchStat = -1;
                     return (double)0.0F;
                  }

                  chLocal = this.charAt(this.bp + offset++);
                  start = this.bp + 1;
                  count = this.bp + offset - start - 2;
               } else {
                  start = this.bp;
                  count = this.bp + offset - start - 1;
               }

               double value;
               if (!exp && count < 17) {
                  value = (double)intVal / (double)power;
                  if (negative) {
                     value = -value;
                  }
               } else {
                  String text = this.subString(start, count);
                  value = Double.parseDouble(text);
               }

               if (chLocal == seperator) {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 3;
                  this.token = 16;
                  return value;
               }

               this.matchStat = -1;
               return value;
            }

            intVal = intVal * 10L + (long)(chLocal - 48);
         }
      } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
         this.matchStat = 5;
         double value = (double)0.0F;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         if (quote && chLocal == '"') {
            chLocal = this.charAt(this.bp + offset++);
         }

         while(chLocal != ',') {
            if (chLocal == ']') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 5;
               this.token = 15;
               return value;
            }

            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return (double)0.0F;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
         this.token = 16;
         return value;
      } else {
         this.matchStat = -1;
         return (double)0.0F;
      }
   }

   public BigDecimal scanDecimal(char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      boolean quote = chLocal == '"';
      if (quote) {
         chLocal = this.charAt(this.bp + offset++);
      }

      boolean negative = chLocal == '-';
      if (negative) {
         chLocal = this.charAt(this.bp + offset++);
      }

      if (chLocal >= '0' && chLocal <= '9') {
         do {
            chLocal = this.charAt(this.bp + offset++);
         } while(chLocal >= '0' && chLocal <= '9');

         boolean small = chLocal == '.';
         if (small) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal < '0' || chLocal > '9') {
               this.matchStat = -1;
               return null;
            }

            do {
               chLocal = this.charAt(this.bp + offset++);
            } while(chLocal >= '0' && chLocal <= '9');
         }

         boolean exp = chLocal == 'e' || chLocal == 'E';
         if (exp) {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == '+' || chLocal == '-') {
               chLocal = this.charAt(this.bp + offset++);
            }

            while(chLocal >= '0' && chLocal <= '9') {
               chLocal = this.charAt(this.bp + offset++);
            }
         }

         int start;
         int count;
         if (quote) {
            if (chLocal != '"') {
               this.matchStat = -1;
               return null;
            }

            chLocal = this.charAt(this.bp + offset++);
            start = this.bp + 1;
            count = this.bp + offset - start - 2;
         } else {
            start = this.bp;
            count = this.bp + offset - start - 1;
         }

         if (count > 65535) {
            throw new JSONException("decimal overflow");
         } else {
            char[] chars = this.sub_chars(start, count);
            BigDecimal value = new BigDecimal(chars, 0, chars.length, MathContext.UNLIMITED);
            if (chLocal == ',') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               this.token = 16;
               return value;
            } else if (chLocal == ']') {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.token = 16;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == ']') {
                  this.token = 15;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == '}') {
                  this.token = 13;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else {
                  if (chLocal != 26) {
                     this.matchStat = -1;
                     return null;
                  }

                  this.token = 20;
                  this.bp += offset - 1;
                  this.ch = 26;
               }

               this.matchStat = 4;
               return value;
            } else {
               this.matchStat = -1;
               return null;
            }
         }
      } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
         this.matchStat = 5;
         BigDecimal value = null;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
         if (quote && chLocal == '"') {
            chLocal = this.charAt(this.bp + offset++);
         }

         while(chLocal != ',') {
            if (chLocal == '}') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 5;
               this.token = 13;
               return value;
            }

            if (!isWhitespace(chLocal)) {
               this.matchStat = -1;
               return null;
            }

            chLocal = this.charAt(this.bp + offset++);
         }

         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 5;
         this.token = 16;
         return value;
      } else {
         this.matchStat = -1;
         return null;
      }
   }

   public final float[] scanFieldFloatArray(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '[') {
            this.matchStat = -2;
            return null;
         } else {
            chLocal = this.charAt(this.bp + offset++);
            float[] array = new float[16];
            int arrayIndex = 0;

            label150:
            while(true) {
               int start = this.bp + offset - 1;
               boolean negative = chLocal == '-';
               if (negative) {
                  chLocal = this.charAt(this.bp + offset++);
               }

               if (chLocal >= '0' && chLocal <= '9') {
                  int intVal = chLocal - 48;

                  while(true) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        int power = 1;
                        boolean small = chLocal == '.';
                        if (small) {
                           chLocal = this.charAt(this.bp + offset++);
                           power = 10;
                           if (chLocal < '0' || chLocal > '9') {
                              this.matchStat = -1;
                              return null;
                           }

                           intVal = intVal * 10 + (chLocal - 48);

                           while(true) {
                              chLocal = this.charAt(this.bp + offset++);
                              if (chLocal < '0' || chLocal > '9') {
                                 break;
                              }

                              intVal = intVal * 10 + (chLocal - 48);
                              power *= 10;
                           }
                        }

                        boolean exp = chLocal == 'e' || chLocal == 'E';
                        if (exp) {
                           chLocal = this.charAt(this.bp + offset++);
                           if (chLocal == '+' || chLocal == '-') {
                              chLocal = this.charAt(this.bp + offset++);
                           }

                           while(chLocal >= '0' && chLocal <= '9') {
                              chLocal = this.charAt(this.bp + offset++);
                           }
                        }

                        int count = this.bp + offset - start - 1;
                        float value;
                        if (!exp && count < 10) {
                           value = (float)intVal / (float)power;
                           if (negative) {
                              value = -value;
                           }
                        } else {
                           String text = this.subString(start, count);
                           value = Float.parseFloat(text);
                        }

                        if (arrayIndex >= array.length) {
                           float[] tmp = new float[array.length * 3 / 2];
                           System.arraycopy(array, 0, tmp, 0, arrayIndex);
                           array = tmp;
                        }

                        array[arrayIndex++] = value;
                        if (chLocal == ',') {
                           chLocal = this.charAt(this.bp + offset++);
                        } else if (chLocal == ']') {
                           chLocal = this.charAt(this.bp + offset++);
                           if (arrayIndex != array.length) {
                              float[] tmp = new float[arrayIndex];
                              System.arraycopy(array, 0, tmp, 0, arrayIndex);
                              array = tmp;
                           }

                           if (chLocal == ',') {
                              this.bp += offset - 1;
                              this.next();
                              this.matchStat = 3;
                              this.token = 16;
                              return array;
                           }

                           if (chLocal == '}') {
                              chLocal = this.charAt(this.bp + offset++);
                              if (chLocal == ',') {
                                 this.token = 16;
                                 this.bp += offset - 1;
                                 this.next();
                              } else if (chLocal == ']') {
                                 this.token = 15;
                                 this.bp += offset - 1;
                                 this.next();
                              } else if (chLocal == '}') {
                                 this.token = 13;
                                 this.bp += offset - 1;
                                 this.next();
                              } else {
                                 if (chLocal != 26) {
                                    this.matchStat = -1;
                                    return null;
                                 }

                                 this.bp += offset - 1;
                                 this.token = 20;
                                 this.ch = 26;
                              }

                              this.matchStat = 4;
                              return array;
                           }

                           this.matchStat = -1;
                           return null;
                        }
                        continue label150;
                     }

                     intVal = intVal * 10 + (chLocal - 48);
                  }
               }

               this.matchStat = -1;
               return null;
            }
         }
      }
   }

   public final float[][] scanFieldFloatArray2(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return (float[][])null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         if (chLocal != '[') {
            this.matchStat = -2;
            return (float[][])null;
         } else {
            chLocal = this.charAt(this.bp + offset++);
            float[][] arrayarray = new float[16][];
            int arrayarrayIndex = 0;

            label167:
            while(chLocal == '[') {
               chLocal = this.charAt(this.bp + offset++);
               float[] array = new float[16];
               int arrayIndex = 0;

               while(true) {
                  int start = this.bp + offset - 1;
                  boolean negative = chLocal == '-';
                  if (negative) {
                     chLocal = this.charAt(this.bp + offset++);
                  }

                  if (chLocal < '0' || chLocal > '9') {
                     this.matchStat = -1;
                     return (float[][])null;
                  }

                  int intVal = chLocal - 48;

                  while(true) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        int power = 1;
                        if (chLocal == '.') {
                           chLocal = this.charAt(this.bp + offset++);
                           if (chLocal < '0' || chLocal > '9') {
                              this.matchStat = -1;
                              return (float[][])null;
                           }

                           intVal = intVal * 10 + (chLocal - 48);
                           power = 10;

                           while(true) {
                              chLocal = this.charAt(this.bp + offset++);
                              if (chLocal < '0' || chLocal > '9') {
                                 break;
                              }

                              intVal = intVal * 10 + (chLocal - 48);
                              power *= 10;
                           }
                        }

                        boolean exp = chLocal == 'e' || chLocal == 'E';
                        if (exp) {
                           chLocal = this.charAt(this.bp + offset++);
                           if (chLocal == '+' || chLocal == '-') {
                              chLocal = this.charAt(this.bp + offset++);
                           }

                           while(chLocal >= '0' && chLocal <= '9') {
                              chLocal = this.charAt(this.bp + offset++);
                           }
                        }

                        int count = this.bp + offset - start - 1;
                        float value;
                        if (!exp && count < 10) {
                           value = (float)intVal / (float)power;
                           if (negative) {
                              value = -value;
                           }
                        } else {
                           String text = this.subString(start, count);
                           value = Float.parseFloat(text);
                        }

                        if (arrayIndex >= array.length) {
                           float[] tmp = new float[array.length * 3 / 2];
                           System.arraycopy(array, 0, tmp, 0, arrayIndex);
                           array = tmp;
                        }

                        array[arrayIndex++] = value;
                        if (chLocal == ',') {
                           chLocal = this.charAt(this.bp + offset++);
                        } else if (chLocal == ']') {
                           chLocal = this.charAt(this.bp + offset++);
                           if (arrayIndex != array.length) {
                              float[] tmp = new float[arrayIndex];
                              System.arraycopy(array, 0, tmp, 0, arrayIndex);
                              array = tmp;
                           }

                           if (arrayarrayIndex >= arrayarray.length) {
                              float[][] tmp = new float[arrayarray.length * 3 / 2][];
                              System.arraycopy(array, 0, tmp, 0, arrayIndex);
                              arrayarray = tmp;
                           }

                           arrayarray[arrayarrayIndex++] = array;
                           if (chLocal == ',') {
                              chLocal = this.charAt(this.bp + offset++);
                              continue label167;
                           }

                           if (chLocal != ']') {
                              continue label167;
                           }

                           chLocal = this.charAt(this.bp + offset++);
                           break label167;
                        }
                        break;
                     }

                     intVal = intVal * 10 + (chLocal - 48);
                  }
               }
            }

            if (arrayarrayIndex != arrayarray.length) {
               float[][] tmp = new float[arrayarrayIndex][];
               System.arraycopy(arrayarray, 0, tmp, 0, arrayarrayIndex);
               arrayarray = tmp;
            }

            if (chLocal == ',') {
               this.bp += offset - 1;
               this.next();
               this.matchStat = 3;
               this.token = 16;
               return arrayarray;
            } else if (chLocal == '}') {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.token = 16;
                  this.bp += offset - 1;
                  this.next();
               } else if (chLocal == ']') {
                  this.token = 15;
                  this.bp += offset - 1;
                  this.next();
               } else if (chLocal == '}') {
                  this.token = 13;
                  this.bp += offset - 1;
                  this.next();
               } else {
                  if (chLocal != 26) {
                     this.matchStat = -1;
                     return (float[][])null;
                  }

                  this.bp += offset - 1;
                  this.token = 20;
                  this.ch = 26;
               }

               this.matchStat = 4;
               return arrayarray;
            } else {
               this.matchStat = -1;
               return (float[][])null;
            }
         }
      }
   }

   public final double scanFieldDouble(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return (double)0.0F;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean quote = chLocal == '"';
         if (quote) {
            chLocal = this.charAt(this.bp + offset++);
         }

         boolean negative = chLocal == '-';
         if (negative) {
            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal >= '0' && chLocal <= '9') {
            long intVal = (long)(chLocal - 48);

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  long power = 1L;
                  boolean small = chLocal == '.';
                  if (small) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal < '0' || chLocal > '9') {
                        this.matchStat = -1;
                        return (double)0.0F;
                     }

                     intVal = intVal * 10L + (long)(chLocal - 48);
                     power = 10L;

                     while(true) {
                        chLocal = this.charAt(this.bp + offset++);
                        if (chLocal < '0' || chLocal > '9') {
                           break;
                        }

                        intVal = intVal * 10L + (long)(chLocal - 48);
                        power *= 10L;
                     }
                  }

                  boolean exp = chLocal == 'e' || chLocal == 'E';
                  if (exp) {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == '+' || chLocal == '-') {
                        chLocal = this.charAt(this.bp + offset++);
                     }

                     while(chLocal >= '0' && chLocal <= '9') {
                        chLocal = this.charAt(this.bp + offset++);
                     }
                  }

                  int start;
                  int count;
                  if (quote) {
                     if (chLocal != '"') {
                        this.matchStat = -1;
                        return (double)0.0F;
                     }

                     chLocal = this.charAt(this.bp + offset++);
                     start = this.bp + fieldName.length + 1;
                     count = this.bp + offset - start - 2;
                  } else {
                     start = this.bp + fieldName.length;
                     count = this.bp + offset - start - 1;
                  }

                  double value;
                  if (!exp && count < 17) {
                     value = (double)intVal / (double)power;
                     if (negative) {
                        value = -value;
                     }
                  } else {
                     String text = this.subString(start, count);
                     value = Double.parseDouble(text);
                  }

                  if (chLocal == ',') {
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                     this.matchStat = 3;
                     this.token = 16;
                     return value;
                  }

                  if (chLocal == '}') {
                     chLocal = this.charAt(this.bp + offset++);
                     if (chLocal == ',') {
                        this.token = 16;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == ']') {
                        this.token = 15;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else if (chLocal == '}') {
                        this.token = 13;
                        this.bp += offset;
                        this.ch = this.charAt(this.bp);
                     } else {
                        if (chLocal != 26) {
                           this.matchStat = -1;
                           return (double)0.0F;
                        }

                        this.token = 20;
                        this.bp += offset - 1;
                        this.ch = 26;
                     }

                     this.matchStat = 4;
                     return value;
                  }

                  this.matchStat = -1;
                  return (double)0.0F;
               }

               intVal = intVal * 10L + (long)(chLocal - 48);
            }
         } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            this.matchStat = 5;
            double value = (double)0.0F;
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (quote && chLocal == '"') {
               chLocal = this.charAt(this.bp + offset++);
            }

            while(chLocal != ',') {
               if (chLocal == '}') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 5;
                  this.token = 13;
                  return value;
               }

               if (!isWhitespace(chLocal)) {
                  this.matchStat = -1;
                  return (double)0.0F;
               }

               chLocal = this.charAt(this.bp + offset++);
            }

            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 5;
            this.token = 16;
            return value;
         } else {
            this.matchStat = -1;
            return (double)0.0F;
         }
      }
   }

   public BigDecimal scanFieldDecimal(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean quote = chLocal == '"';
         if (quote) {
            chLocal = this.charAt(this.bp + offset++);
         }

         boolean negative = chLocal == '-';
         if (negative) {
            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal >= '0' && chLocal <= '9') {
            do {
               chLocal = this.charAt(this.bp + offset++);
            } while(chLocal >= '0' && chLocal <= '9');

            boolean small = chLocal == '.';
            if (small) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  this.matchStat = -1;
                  return null;
               }

               do {
                  chLocal = this.charAt(this.bp + offset++);
               } while(chLocal >= '0' && chLocal <= '9');
            }

            boolean exp = chLocal == 'e' || chLocal == 'E';
            if (exp) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == '+' || chLocal == '-') {
                  chLocal = this.charAt(this.bp + offset++);
               }

               while(chLocal >= '0' && chLocal <= '9') {
                  chLocal = this.charAt(this.bp + offset++);
               }
            }

            int start;
            int count;
            if (quote) {
               if (chLocal != '"') {
                  this.matchStat = -1;
                  return null;
               }

               chLocal = this.charAt(this.bp + offset++);
               start = this.bp + fieldName.length + 1;
               count = this.bp + offset - start - 2;
            } else {
               start = this.bp + fieldName.length;
               count = this.bp + offset - start - 1;
            }

            if (count > 65535) {
               throw new JSONException("scan decimal overflow");
            } else {
               char[] chars = this.sub_chars(start, count);
               BigDecimal value = new BigDecimal(chars, 0, chars.length, MathContext.UNLIMITED);
               if (chLocal == ',') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 3;
                  this.token = 16;
                  return value;
               } else if (chLocal == '}') {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal == ',') {
                     this.token = 16;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else if (chLocal == ']') {
                     this.token = 15;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else if (chLocal == '}') {
                     this.token = 13;
                     this.bp += offset;
                     this.ch = this.charAt(this.bp);
                  } else {
                     if (chLocal != 26) {
                        this.matchStat = -1;
                        return null;
                     }

                     this.token = 20;
                     this.bp += offset - 1;
                     this.ch = 26;
                  }

                  this.matchStat = 4;
                  return value;
               } else {
                  this.matchStat = -1;
                  return null;
               }
            }
         } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            this.matchStat = 5;
            BigDecimal value = null;
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (quote && chLocal == '"') {
               chLocal = this.charAt(this.bp + offset++);
            }

            while(chLocal != ',') {
               if (chLocal == '}') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 5;
                  this.token = 13;
                  return value;
               }

               if (!isWhitespace(chLocal)) {
                  this.matchStat = -1;
                  return null;
               }

               chLocal = this.charAt(this.bp + offset++);
            }

            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 5;
            this.token = 16;
            return value;
         } else {
            this.matchStat = -1;
            return null;
         }
      }
   }

   public BigInteger scanFieldBigInteger(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         boolean quote = chLocal == '"';
         if (quote) {
            chLocal = this.charAt(this.bp + offset++);
         }

         boolean negative = chLocal == '-';
         if (negative) {
            chLocal = this.charAt(this.bp + offset++);
         }

         if (chLocal >= '0' && chLocal <= '9') {
            long intVal = (long)(chLocal - 48);
            boolean overflow = false;

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  break;
               }

               long temp = intVal * 10L + (long)(chLocal - 48);
               if (temp < intVal) {
                  overflow = true;
                  break;
               }

               intVal = temp;
            }

            int start;
            int count;
            if (quote) {
               if (chLocal != '"') {
                  this.matchStat = -1;
                  return null;
               }

               chLocal = this.charAt(this.bp + offset++);
               start = this.bp + fieldName.length + 1;
               count = this.bp + offset - start - 2;
            } else {
               start = this.bp + fieldName.length;
               count = this.bp + offset - start - 1;
            }

            BigInteger value;
            if (!overflow && (count < 20 || negative && count < 21)) {
               value = BigInteger.valueOf(negative ? -intVal : intVal);
            } else {
               if (count > 65535) {
                  throw new JSONException("scanInteger overflow");
               }

               String strVal = this.subString(start, count);
               value = new BigInteger(strVal, 10);
            }

            if (chLocal == ',') {
               this.bp += offset;
               this.ch = this.charAt(this.bp);
               this.matchStat = 3;
               this.token = 16;
               return value;
            } else if (chLocal == '}') {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal == ',') {
                  this.token = 16;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == ']') {
                  this.token = 15;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else if (chLocal == '}') {
                  this.token = 13;
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
               } else {
                  if (chLocal != 26) {
                     this.matchStat = -1;
                     return null;
                  }

                  this.token = 20;
                  this.bp += offset - 1;
                  this.ch = 26;
               }

               this.matchStat = 4;
               return value;
            } else {
               this.matchStat = -1;
               return null;
            }
         } else if (chLocal == 'n' && this.charAt(this.bp + offset) == 'u' && this.charAt(this.bp + offset + 1) == 'l' && this.charAt(this.bp + offset + 2) == 'l') {
            this.matchStat = 5;
            BigInteger value = null;
            offset += 3;
            chLocal = this.charAt(this.bp + offset++);
            if (quote && chLocal == '"') {
               chLocal = this.charAt(this.bp + offset++);
            }

            while(chLocal != ',') {
               if (chLocal == '}') {
                  this.bp += offset;
                  this.ch = this.charAt(this.bp);
                  this.matchStat = 5;
                  this.token = 13;
                  return value;
               }

               if (!isWhitespace(chLocal)) {
                  this.matchStat = -1;
                  return null;
               }

               chLocal = this.charAt(this.bp + offset++);
            }

            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 5;
            this.token = 16;
            return value;
         } else {
            this.matchStat = -1;
            return null;
         }
      }
   }

   public Date scanFieldDate(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         Date dateVal;
         if (chLocal == '"') {
            int startIndex = this.bp + fieldName.length + 1;
            int endIndex = this.indexOf('"', startIndex);
            if (endIndex == -1) {
               throw new JSONException("unclosed str");
            }

            int startIndex2 = this.bp + fieldName.length + 1;
            String stringVal = this.subString(startIndex2, endIndex - startIndex2);
            if (stringVal.indexOf(92) != -1) {
               while(true) {
                  int slashCount = 0;

                  for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                     ++slashCount;
                  }

                  if (slashCount % 2 == 0) {
                     slashCount = endIndex - (this.bp + fieldName.length + 1);
                     char[] chars = this.sub_chars(this.bp + fieldName.length + 1, slashCount);
                     stringVal = readString(chars, slashCount);
                     break;
                  }

                  endIndex = this.indexOf('"', endIndex + 1);
               }
            }

            offset += endIndex - (this.bp + fieldName.length + 1) + 1;
            chLocal = this.charAt(this.bp + offset++);
            JSONScanner dateLexer = new JSONScanner(stringVal);

            try {
               if (!dateLexer.scanISO8601DateIfMatch(false)) {
                  this.matchStat = -1;
                  Object var24 = null;
                  return (Date)var24;
               }

               Calendar calendar = dateLexer.getCalendar();
               dateVal = calendar.getTime();
            } finally {
               dateLexer.close();
            }
         } else {
            if (chLocal != '-' && (chLocal < '0' || chLocal > '9')) {
               this.matchStat = -1;
               return null;
            }

            long millis = 0L;
            boolean negative = false;
            if (chLocal == '-') {
               chLocal = this.charAt(this.bp + offset++);
               negative = true;
            }

            if (chLocal >= '0' && chLocal <= '9') {
               millis = (long)(chLocal - 48);

               while(true) {
                  chLocal = this.charAt(this.bp + offset++);
                  if (chLocal < '0' || chLocal > '9') {
                     break;
                  }

                  millis = millis * 10L + (long)(chLocal - 48);
               }
            }

            if (millis < 0L) {
               this.matchStat = -1;
               return null;
            }

            if (negative) {
               millis = -millis;
            }

            dateVal = new Date(millis);
         }

         if (chLocal == ',') {
            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 3;
            return dateVal;
         } else if (chLocal == '}') {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == ',') {
               this.token = 16;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == ']') {
               this.token = 15;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == '}') {
               this.token = 13;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else {
               if (chLocal != 26) {
                  this.matchStat = -1;
                  return null;
               }

               this.token = 20;
               this.bp += offset - 1;
               this.ch = 26;
            }

            this.matchStat = 4;
            return dateVal;
         } else {
            this.matchStat = -1;
            return null;
         }
      }
   }

   public Date scanDate(char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      Date dateVal;
      if (chLocal == '"') {
         int startIndex = this.bp + 1;
         int endIndex = this.indexOf('"', startIndex);
         if (endIndex == -1) {
            throw new JSONException("unclosed str");
         }

         int startIndex2 = this.bp + 1;
         String stringVal = this.subString(startIndex2, endIndex - startIndex2);
         if (stringVal.indexOf(92) != -1) {
            while(true) {
               int slashCount = 0;

               for(int i = endIndex - 1; i >= 0 && this.charAt(i) == '\\'; --i) {
                  ++slashCount;
               }

               if (slashCount % 2 == 0) {
                  slashCount = endIndex - (this.bp + 1);
                  char[] chars = this.sub_chars(this.bp + 1, slashCount);
                  stringVal = readString(chars, slashCount);
                  break;
               }

               endIndex = this.indexOf('"', endIndex + 1);
            }
         }

         offset += endIndex - (this.bp + 1) + 1;
         chLocal = this.charAt(this.bp + offset++);
         JSONScanner dateLexer = new JSONScanner(stringVal);

         try {
            if (!dateLexer.scanISO8601DateIfMatch(false)) {
               this.matchStat = -1;
               Object var25 = null;
               return (Date)var25;
            }

            Calendar calendar = dateLexer.getCalendar();
            dateVal = calendar.getTime();
         } finally {
            dateLexer.close();
         }
      } else if (chLocal != '-' && (chLocal < '0' || chLocal > '9')) {
         if (chLocal != 'n' || this.charAt(this.bp + offset) != 'u' || this.charAt(this.bp + offset + 1) != 'l' || this.charAt(this.bp + offset + 2) != 'l') {
            this.matchStat = -1;
            return null;
         }

         this.matchStat = 5;
         dateVal = null;
         offset += 3;
         chLocal = this.charAt(this.bp + offset++);
      } else {
         long millis = 0L;
         boolean negative = false;
         if (chLocal == '-') {
            chLocal = this.charAt(this.bp + offset++);
            negative = true;
         }

         if (chLocal >= '0' && chLocal <= '9') {
            millis = (long)(chLocal - 48);

            while(true) {
               chLocal = this.charAt(this.bp + offset++);
               if (chLocal < '0' || chLocal > '9') {
                  break;
               }

               millis = millis * 10L + (long)(chLocal - 48);
            }
         }

         if (millis < 0L) {
            this.matchStat = -1;
            return null;
         }

         if (negative) {
            millis = -millis;
         }

         dateVal = new Date(millis);
      }

      if (chLocal == ',') {
         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 3;
         this.token = 16;
         return dateVal;
      } else if (chLocal == ']') {
         chLocal = this.charAt(this.bp + offset++);
         if (chLocal == ',') {
            this.token = 16;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else if (chLocal == ']') {
            this.token = 15;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else if (chLocal == '}') {
            this.token = 13;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else {
            if (chLocal != 26) {
               this.matchStat = -1;
               return null;
            }

            this.token = 20;
            this.bp += offset - 1;
            this.ch = 26;
         }

         this.matchStat = 4;
         return dateVal;
      } else {
         this.matchStat = -1;
         return null;
      }
   }

   public UUID scanFieldUUID(char[] fieldName) {
      this.matchStat = 0;
      if (!this.charArrayCompare(fieldName)) {
         this.matchStat = -2;
         return null;
      } else {
         int offset = fieldName.length;
         char chLocal = this.charAt(this.bp + offset++);
         UUID uuid;
         if (chLocal != '"') {
            if (chLocal != 'n' || this.charAt(this.bp + offset++) != 'u' || this.charAt(this.bp + offset++) != 'l' || this.charAt(this.bp + offset++) != 'l') {
               this.matchStat = -1;
               return null;
            }

            uuid = null;
            chLocal = this.charAt(this.bp + offset++);
         } else {
            int startIndex = this.bp + fieldName.length + 1;
            int endIndex = this.indexOf('"', startIndex);
            if (endIndex == -1) {
               throw new JSONException("unclosed str");
            }

            int startIndex2 = this.bp + fieldName.length + 1;
            int len = endIndex - startIndex2;
            if (len == 36) {
               long mostSigBits = 0L;
               long leastSigBits = 0L;

               for(int i = 0; i < 8; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  mostSigBits <<= 4;
                  mostSigBits |= (long)num;
               }

               for(int i = 9; i < 13; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  mostSigBits <<= 4;
                  mostSigBits |= (long)num;
               }

               for(int i = 14; i < 18; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  mostSigBits <<= 4;
                  mostSigBits |= (long)num;
               }

               for(int i = 19; i < 23; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  leastSigBits <<= 4;
                  leastSigBits |= (long)num;
               }

               for(int i = 24; i < 36; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  leastSigBits <<= 4;
                  leastSigBits |= (long)num;
               }

               uuid = new UUID(mostSigBits, leastSigBits);
               offset += endIndex - (this.bp + fieldName.length + 1) + 1;
               chLocal = this.charAt(this.bp + offset++);
            } else {
               if (len != 32) {
                  this.matchStat = -1;
                  return null;
               }

               long mostSigBits = 0L;
               long leastSigBits = 0L;

               for(int i = 0; i < 16; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  mostSigBits <<= 4;
                  mostSigBits |= (long)num;
               }

               for(int i = 16; i < 32; ++i) {
                  char ch = this.charAt(startIndex2 + i);
                  int num;
                  if (ch >= '0' && ch <= '9') {
                     num = ch - 48;
                  } else if (ch >= 'a' && ch <= 'f') {
                     num = 10 + (ch - 97);
                  } else {
                     if (ch < 'A' || ch > 'F') {
                        this.matchStat = -2;
                        return null;
                     }

                     num = 10 + (ch - 65);
                  }

                  leastSigBits <<= 4;
                  leastSigBits |= (long)num;
               }

               uuid = new UUID(mostSigBits, leastSigBits);
               offset += endIndex - (this.bp + fieldName.length + 1) + 1;
               chLocal = this.charAt(this.bp + offset++);
            }
         }

         if (chLocal == ',') {
            this.bp += offset;
            this.ch = this.charAt(this.bp);
            this.matchStat = 3;
            return uuid;
         } else if (chLocal == '}') {
            chLocal = this.charAt(this.bp + offset++);
            if (chLocal == ',') {
               this.token = 16;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == ']') {
               this.token = 15;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else if (chLocal == '}') {
               this.token = 13;
               this.bp += offset;
               this.ch = this.charAt(this.bp);
            } else {
               if (chLocal != 26) {
                  this.matchStat = -1;
                  return null;
               }

               this.token = 20;
               this.bp += offset - 1;
               this.ch = 26;
            }

            this.matchStat = 4;
            return uuid;
         } else {
            this.matchStat = -1;
            return null;
         }
      }
   }

   public UUID scanUUID(char seperator) {
      this.matchStat = 0;
      int offset = 0;
      char chLocal = this.charAt(this.bp + offset++);
      UUID uuid;
      if (chLocal != '"') {
         if (chLocal != 'n' || this.charAt(this.bp + offset++) != 'u' || this.charAt(this.bp + offset++) != 'l' || this.charAt(this.bp + offset++) != 'l') {
            this.matchStat = -1;
            return null;
         }

         uuid = null;
         chLocal = this.charAt(this.bp + offset++);
      } else {
         int startIndex = this.bp + 1;
         int endIndex = this.indexOf('"', startIndex);
         if (endIndex == -1) {
            throw new JSONException("unclosed str");
         }

         int startIndex2 = this.bp + 1;
         int len = endIndex - startIndex2;
         if (len == 36) {
            long mostSigBits = 0L;
            long leastSigBits = 0L;

            for(int i = 0; i < 8; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               mostSigBits <<= 4;
               mostSigBits |= (long)num;
            }

            for(int i = 9; i < 13; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               mostSigBits <<= 4;
               mostSigBits |= (long)num;
            }

            for(int i = 14; i < 18; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               mostSigBits <<= 4;
               mostSigBits |= (long)num;
            }

            for(int i = 19; i < 23; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               leastSigBits <<= 4;
               leastSigBits |= (long)num;
            }

            for(int i = 24; i < 36; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               leastSigBits <<= 4;
               leastSigBits |= (long)num;
            }

            uuid = new UUID(mostSigBits, leastSigBits);
            offset += endIndex - (this.bp + 1) + 1;
            chLocal = this.charAt(this.bp + offset++);
         } else {
            if (len != 32) {
               this.matchStat = -1;
               return null;
            }

            long mostSigBits = 0L;
            long leastSigBits = 0L;

            for(int i = 0; i < 16; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               mostSigBits <<= 4;
               mostSigBits |= (long)num;
            }

            for(int i = 16; i < 32; ++i) {
               char ch = this.charAt(startIndex2 + i);
               int num;
               if (ch >= '0' && ch <= '9') {
                  num = ch - 48;
               } else if (ch >= 'a' && ch <= 'f') {
                  num = 10 + (ch - 97);
               } else {
                  if (ch < 'A' || ch > 'F') {
                     this.matchStat = -2;
                     return null;
                  }

                  num = 10 + (ch - 65);
               }

               leastSigBits <<= 4;
               leastSigBits |= (long)num;
            }

            uuid = new UUID(mostSigBits, leastSigBits);
            offset += endIndex - (this.bp + 1) + 1;
            chLocal = this.charAt(this.bp + offset++);
         }
      }

      if (chLocal == ',') {
         this.bp += offset;
         this.ch = this.charAt(this.bp);
         this.matchStat = 3;
         return uuid;
      } else if (chLocal == ']') {
         chLocal = this.charAt(this.bp + offset++);
         if (chLocal == ',') {
            this.token = 16;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else if (chLocal == ']') {
            this.token = 15;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else if (chLocal == '}') {
            this.token = 13;
            this.bp += offset;
            this.ch = this.charAt(this.bp);
         } else {
            if (chLocal != 26) {
               this.matchStat = -1;
               return null;
            }

            this.token = 20;
            this.bp += offset - 1;
            this.ch = 26;
         }

         this.matchStat = 4;
         return uuid;
      } else {
         this.matchStat = -1;
         return null;
      }
   }

   public final void scanTrue() {
      if (this.ch != 't') {
         throw new JSONException("error parse true");
      } else {
         this.next();
         if (this.ch != 'r') {
            throw new JSONException("error parse true");
         } else {
            this.next();
            if (this.ch != 'u') {
               throw new JSONException("error parse true");
            } else {
               this.next();
               if (this.ch != 'e') {
                  throw new JSONException("error parse true");
               } else {
                  this.next();
                  if (this.ch != ' ' && this.ch != ',' && this.ch != '}' && this.ch != ']' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != 26 && this.ch != '\f' && this.ch != '\b' && this.ch != ':' && this.ch != '/') {
                     throw new JSONException("scan true error");
                  } else {
                     this.token = 6;
                  }
               }
            }
         }
      }
   }

   public final void scanNullOrNew() {
      this.scanNullOrNew(true);
   }

   public final void scanNullOrNew(boolean acceptColon) {
      if (this.ch != 'n') {
         throw new JSONException("error parse null or new");
      } else {
         this.next();
         if (this.ch != 'u') {
            if (this.ch != 'e') {
               throw new JSONException("error parse new");
            } else {
               this.next();
               if (this.ch != 'w') {
                  throw new JSONException("error parse new");
               } else {
                  this.next();
                  if (this.ch != ' ' && this.ch != ',' && this.ch != '}' && this.ch != ']' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != 26 && this.ch != '\f' && this.ch != '\b') {
                     throw new JSONException("scan new error");
                  } else {
                     this.token = 9;
                  }
               }
            }
         } else {
            this.next();
            if (this.ch != 'l') {
               throw new JSONException("error parse null");
            } else {
               this.next();
               if (this.ch != 'l') {
                  throw new JSONException("error parse null");
               } else {
                  this.next();
                  if (this.ch != ' ' && this.ch != ',' && this.ch != '}' && this.ch != ']' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != 26 && (this.ch != ':' || !acceptColon) && this.ch != '\f' && this.ch != '\b') {
                     throw new JSONException("scan null error");
                  } else {
                     this.token = 8;
                  }
               }
            }
         }
      }
   }

   public final void scanFalse() {
      if (this.ch != 'f') {
         throw new JSONException("error parse false");
      } else {
         this.next();
         if (this.ch != 'a') {
            throw new JSONException("error parse false");
         } else {
            this.next();
            if (this.ch != 'l') {
               throw new JSONException("error parse false");
            } else {
               this.next();
               if (this.ch != 's') {
                  throw new JSONException("error parse false");
               } else {
                  this.next();
                  if (this.ch != 'e') {
                     throw new JSONException("error parse false");
                  } else {
                     this.next();
                     if (this.ch != ' ' && this.ch != ',' && this.ch != '}' && this.ch != ']' && this.ch != '\n' && this.ch != '\r' && this.ch != '\t' && this.ch != 26 && this.ch != '\f' && this.ch != '\b' && this.ch != ':' && this.ch != '/') {
                        throw new JSONException("scan false error");
                     } else {
                        this.token = 7;
                     }
                  }
               }
            }
         }
      }
   }

   public final void scanIdent() {
      this.np = this.bp - 1;
      this.hasSpecial = false;

      do {
         ++this.sp;
         this.next();
      } while(Character.isLetterOrDigit(this.ch));

      String ident = this.stringVal();
      if ("null".equalsIgnoreCase(ident)) {
         this.token = 8;
      } else if ("new".equals(ident)) {
         this.token = 9;
      } else if ("true".equals(ident)) {
         this.token = 6;
      } else if ("false".equals(ident)) {
         this.token = 7;
      } else if ("undefined".equals(ident)) {
         this.token = 23;
      } else if ("Set".equals(ident)) {
         this.token = 21;
      } else if ("TreeSet".equals(ident)) {
         this.token = 22;
      } else {
         this.token = 18;
      }

   }

   public abstract String stringVal();

   public abstract String subString(int var1, int var2);

   protected abstract char[] sub_chars(int var1, int var2);

   public static String readString(char[] chars, int chars_len) {
      char[] sbuf = new char[chars_len];
      int len = 0;

      for(int i = 0; i < chars_len; ++i) {
         char ch = chars[i];
         if (ch != '\\') {
            sbuf[len++] = ch;
         } else {
            ++i;
            ch = chars[i];
            switch (ch) {
               case '"':
                  sbuf[len++] = '"';
                  break;
               case '#':
               case '$':
               case '%':
               case '&':
               case '(':
               case ')':
               case '*':
               case '+':
               case ',':
               case '-':
               case '.':
               case '8':
               case '9':
               case ':':
               case ';':
               case '<':
               case '=':
               case '>':
               case '?':
               case '@':
               case 'A':
               case 'B':
               case 'C':
               case 'D':
               case 'E':
               case 'G':
               case 'H':
               case 'I':
               case 'J':
               case 'K':
               case 'L':
               case 'M':
               case 'N':
               case 'O':
               case 'P':
               case 'Q':
               case 'R':
               case 'S':
               case 'T':
               case 'U':
               case 'V':
               case 'W':
               case 'X':
               case 'Y':
               case 'Z':
               case '[':
               case ']':
               case '^':
               case '_':
               case '`':
               case 'a':
               case 'c':
               case 'd':
               case 'e':
               case 'g':
               case 'h':
               case 'i':
               case 'j':
               case 'k':
               case 'l':
               case 'm':
               case 'o':
               case 'p':
               case 'q':
               case 's':
               case 'w':
               default:
                  throw new JSONException("unclosed.str.lit");
               case '\'':
                  sbuf[len++] = '\'';
                  break;
               case '/':
                  sbuf[len++] = '/';
                  break;
               case '0':
                  sbuf[len++] = 0;
                  break;
               case '1':
                  sbuf[len++] = 1;
                  break;
               case '2':
                  sbuf[len++] = 2;
                  break;
               case '3':
                  sbuf[len++] = 3;
                  break;
               case '4':
                  sbuf[len++] = 4;
                  break;
               case '5':
                  sbuf[len++] = 5;
                  break;
               case '6':
                  sbuf[len++] = 6;
                  break;
               case '7':
                  sbuf[len++] = 7;
                  break;
               case 'F':
               case 'f':
                  sbuf[len++] = '\f';
                  break;
               case '\\':
                  sbuf[len++] = '\\';
                  break;
               case 'b':
                  sbuf[len++] = '\b';
                  break;
               case 'n':
                  sbuf[len++] = '\n';
                  break;
               case 'r':
                  sbuf[len++] = '\r';
                  break;
               case 't':
                  sbuf[len++] = '\t';
                  break;
               case 'u':
                  int var13 = len++;
                  char[] var10004 = new char[4];
                  ++i;
                  var10004[0] = chars[i];
                  ++i;
                  var10004[1] = chars[i];
                  ++i;
                  var10004[2] = chars[i];
                  ++i;
                  var10004[3] = chars[i];
                  sbuf[var13] = (char)Integer.parseInt(new String(var10004), 16);
                  break;
               case 'v':
                  sbuf[len++] = 11;
                  break;
               case 'x':
                  int var11 = len++;
                  ++i;
                  int var10002 = digits[chars[i]] * 16;
                  ++i;
                  sbuf[var11] = (char)(var10002 + digits[chars[i]]);
            }
         }
      }

      return new String(sbuf, 0, len);
   }

   protected abstract boolean charArrayCompare(char[] var1);

   public boolean isBlankInput() {
      int i = 0;

      while(true) {
         char chLocal = this.charAt(i);
         if (chLocal == 26) {
            this.token = 20;
            return true;
         }

         if (!isWhitespace(chLocal)) {
            return false;
         }

         ++i;
      }
   }

   public final void skipWhitespace() {
      while(true) {
         if (this.ch <= '/') {
            if (this.ch == ' ' || this.ch == '\r' || this.ch == '\n' || this.ch == '\t' || this.ch == '\f' || this.ch == '\b') {
               this.next();
               continue;
            }

            if (this.ch == '/') {
               this.skipComment();
               continue;
            }
         }

         return;
      }
   }

   private void scanStringSingleQuote() {
      this.np = this.bp;
      this.hasSpecial = false;

      while(true) {
         char chLocal = this.next();
         if (chLocal == '\'') {
            this.token = 4;
            this.next();
            return;
         }

         if (chLocal != 26) {
            if (chLocal == '\\') {
               if (!this.hasSpecial) {
                  this.hasSpecial = true;
                  if (this.sp > this.sbuf.length) {
                     char[] newsbuf = new char[this.sp * 2];
                     System.arraycopy(this.sbuf, 0, newsbuf, 0, this.sbuf.length);
                     this.sbuf = newsbuf;
                  }

                  this.copyTo(this.np + 1, this.sp, this.sbuf);
               }

               chLocal = this.next();
               switch (chLocal) {
                  case '"':
                     this.putChar('"');
                     continue;
                  case '#':
                  case '$':
                  case '%':
                  case '&':
                  case '(':
                  case ')':
                  case '*':
                  case '+':
                  case ',':
                  case '-':
                  case '.':
                  case '8':
                  case '9':
                  case ':':
                  case ';':
                  case '<':
                  case '=':
                  case '>':
                  case '?':
                  case '@':
                  case 'A':
                  case 'B':
                  case 'C':
                  case 'D':
                  case 'E':
                  case 'G':
                  case 'H':
                  case 'I':
                  case 'J':
                  case 'K':
                  case 'L':
                  case 'M':
                  case 'N':
                  case 'O':
                  case 'P':
                  case 'Q':
                  case 'R':
                  case 'S':
                  case 'T':
                  case 'U':
                  case 'V':
                  case 'W':
                  case 'X':
                  case 'Y':
                  case 'Z':
                  case '[':
                  case ']':
                  case '^':
                  case '_':
                  case '`':
                  case 'a':
                  case 'c':
                  case 'd':
                  case 'e':
                  case 'g':
                  case 'h':
                  case 'i':
                  case 'j':
                  case 'k':
                  case 'l':
                  case 'm':
                  case 'o':
                  case 'p':
                  case 'q':
                  case 's':
                  case 'w':
                  default:
                     this.ch = chLocal;
                     throw new JSONException("unclosed single-quote string");
                  case '\'':
                     this.putChar('\'');
                     continue;
                  case '/':
                     this.putChar('/');
                     continue;
                  case '0':
                     this.putChar('\u0000');
                     continue;
                  case '1':
                     this.putChar('\u0001');
                     continue;
                  case '2':
                     this.putChar('\u0002');
                     continue;
                  case '3':
                     this.putChar('\u0003');
                     continue;
                  case '4':
                     this.putChar('\u0004');
                     continue;
                  case '5':
                     this.putChar('\u0005');
                     continue;
                  case '6':
                     this.putChar('\u0006');
                     continue;
                  case '7':
                     this.putChar('\u0007');
                     continue;
                  case 'F':
                  case 'f':
                     this.putChar('\f');
                     continue;
                  case '\\':
                     this.putChar('\\');
                     continue;
                  case 'b':
                     this.putChar('\b');
                     continue;
                  case 'n':
                     this.putChar('\n');
                     continue;
                  case 'r':
                     this.putChar('\r');
                     continue;
                  case 't':
                     this.putChar('\t');
                     continue;
                  case 'u':
                     this.putChar((char)Integer.parseInt(new String(new char[]{this.next(), this.next(), this.next(), this.next()}), 16));
                     continue;
                  case 'v':
                     this.putChar('\u000b');
                     continue;
                  case 'x':
               }

               char x1 = this.next();
               char x2 = this.next();
               boolean hex1 = x1 >= '0' && x1 <= '9' || x1 >= 'a' && x1 <= 'f' || x1 >= 'A' && x1 <= 'F';
               boolean hex2 = x2 >= '0' && x2 <= '9' || x2 >= 'a' && x2 <= 'f' || x2 >= 'A' && x2 <= 'F';
               if (!hex1 || !hex2) {
                  throw new JSONException("invalid escape character \\x" + x1 + x2);
               }

               this.putChar((char)(digits[x1] * 16 + digits[x2]));
            } else if (!this.hasSpecial) {
               ++this.sp;
            } else if (this.sp == this.sbuf.length) {
               this.putChar(chLocal);
            } else {
               this.sbuf[this.sp++] = chLocal;
            }
         } else {
            if (this.isEOF()) {
               throw new JSONException("unclosed single-quote string");
            }

            this.putChar('\u001a');
         }
      }
   }

   protected final void putChar(char ch) {
      if (this.sp >= this.sbuf.length) {
         int len = this.sbuf.length * 2;
         if (len < this.sp) {
            len = this.sp + 1;
         }

         char[] newsbuf = new char[len];
         System.arraycopy(this.sbuf, 0, newsbuf, 0, this.sbuf.length);
         this.sbuf = newsbuf;
      }

      this.sbuf[this.sp++] = ch;
   }

   public final void scanHex() {
      if (this.ch != 'x') {
         throw new JSONException("illegal state. " + this.ch);
      } else {
         this.next();
         if (this.ch != '\'') {
            throw new JSONException("illegal state. " + this.ch);
         } else {
            this.np = this.bp;
            this.next();
            if (this.ch == '\'') {
               this.next();
               this.token = 26;
            } else {
               int i = 0;

               while(true) {
                  char ch = this.next();
                  if ((ch < '0' || ch > '9') && (ch < 'A' || ch > 'F')) {
                     if (ch == '\'') {
                        ++this.sp;
                        this.next();
                        this.token = 26;
                        return;
                     } else {
                        throw new JSONException("illegal state. " + ch);
                     }
                  }

                  ++this.sp;
                  ++i;
               }
            }
         }
      }
   }

   public final void scanNumber() {
      this.np = this.bp;
      if (this.ch == '-') {
         ++this.sp;
         this.next();
      }

      while(this.ch >= '0' && this.ch <= '9') {
         ++this.sp;
         this.next();
      }

      boolean isDouble = false;
      if (this.ch == '.') {
         ++this.sp;
         this.next();
         isDouble = true;

         while(this.ch >= '0' && this.ch <= '9') {
            ++this.sp;
            this.next();
         }
      }

      if (this.sp > 65535) {
         throw new JSONException("scanNumber overflow");
      } else {
         if (this.ch == 'L') {
            ++this.sp;
            this.next();
         } else if (this.ch == 'S') {
            ++this.sp;
            this.next();
         } else if (this.ch == 'B') {
            ++this.sp;
            this.next();
         } else if (this.ch == 'F') {
            ++this.sp;
            this.next();
            isDouble = true;
         } else if (this.ch == 'D') {
            ++this.sp;
            this.next();
            isDouble = true;
         } else if (this.ch == 'e' || this.ch == 'E') {
            ++this.sp;
            this.next();
            if (this.ch == '+' || this.ch == '-') {
               ++this.sp;
               this.next();
            }

            while(this.ch >= '0' && this.ch <= '9') {
               ++this.sp;
               this.next();
            }

            if (this.ch == 'D' || this.ch == 'F') {
               ++this.sp;
               this.next();
            }

            isDouble = true;
         }

         if (isDouble) {
            this.token = 3;
         } else {
            this.token = 2;
         }

      }
   }

   public final long longValue() throws NumberFormatException {
      long result = 0L;
      boolean negative = false;
      if (this.np == -1) {
         this.np = 0;
      }

      int i = this.np;
      int max = this.np + this.sp;
      long limit;
      if (this.charAt(this.np) == '-') {
         negative = true;
         limit = Long.MIN_VALUE;
         ++i;
      } else {
         limit = -9223372036854775807L;
      }

      long multmin = -922337203685477580L;
      if (i < max) {
         int digit = this.charAt(i++) - 48;
         result = (long)(-digit);
      }

      while(i < max) {
         char chLocal = this.charAt(i++);
         if (chLocal == 'L' || chLocal == 'S' || chLocal == 'B') {
            break;
         }

         int digit = chLocal - 48;
         if (result < multmin) {
            throw new NumberFormatException(this.numberString());
         }

         result *= 10L;
         if (result < limit + (long)digit) {
            throw new NumberFormatException(this.numberString());
         }

         result -= (long)digit;
      }

      if (negative) {
         if (i > this.np + 1) {
            return result;
         } else {
            throw new NumberFormatException(this.numberString());
         }
      } else {
         return -result;
      }
   }

   public final Number decimalValue(boolean decimal) {
      char chLocal = this.charAt(this.np + this.sp - 1);

      try {
         if (chLocal == 'F') {
            return Float.parseFloat(this.numberString());
         } else if (chLocal == 'D') {
            return Double.parseDouble(this.numberString());
         } else {
            return (Number)(decimal ? this.decimalValue() : this.doubleValue());
         }
      } catch (NumberFormatException ex) {
         throw new JSONException(ex.getMessage() + ", " + this.info());
      }
   }

   public abstract BigDecimal decimalValue();

   public static boolean isWhitespace(char ch) {
      return ch <= ' ' && (ch == ' ' || ch == '\n' || ch == '\r' || ch == '\t' || ch == '\f' || ch == '\b');
   }

   public String[] scanFieldStringArray(char[] fieldName, int argTypesCount, SymbolTable typeSymbolTable) {
      throw new UnsupportedOperationException();
   }

   public boolean matchField2(char[] fieldName) {
      throw new UnsupportedOperationException();
   }

   public int getFeatures() {
      return this.features;
   }

   public void setFeatures(int features) {
      this.features = features;
   }

   static {
      typeFieldName = ("\"" + JSON.DEFAULT_TYPE_KEY + "\":\"").toCharArray();
      digits = new int[103];

      for(int i = 48; i <= 57; ++i) {
         digits[i] = i - 48;
      }

      for(int i = 97; i <= 102; ++i) {
         digits[i] = i - 97 + 10;
      }

      for(int i = 65; i <= 70; ++i) {
         digits[i] = i - 65 + 10;
      }

   }
}
