package com.alibaba.fastjson.serializer;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.annotation.JSONType;
import com.alibaba.fastjson.util.FieldInfo;
import com.alibaba.fastjson.util.TypeUtils;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class JavaBeanSerializer extends SerializeFilterable implements ObjectSerializer {
   protected final FieldSerializer[] getters;
   protected final FieldSerializer[] sortedGetters;
   protected final SerializeBeanInfo beanInfo;
   private transient volatile long[] hashArray;
   private transient volatile short[] hashArrayMapping;

   public JavaBeanSerializer(Class<?> beanType) {
      this(beanType, (Map)null);
   }

   public JavaBeanSerializer(Class<?> beanType, String... aliasList) {
      this(beanType, createAliasMap(aliasList));
   }

   static Map<String, String> createAliasMap(String... aliasList) {
      Map<String, String> aliasMap = new HashMap();

      for(String alias : aliasList) {
         aliasMap.put(alias, alias);
      }

      return aliasMap;
   }

   public JSONType getJSONType() {
      return this.beanInfo.jsonType;
   }

   public Class<?> getType() {
      return this.beanInfo.beanType;
   }

   public JavaBeanSerializer(Class<?> beanType, Map<String, String> aliasMap) {
      this(TypeUtils.buildBeanInfo(beanType, aliasMap, (PropertyNamingStrategy)null));
   }

   public JavaBeanSerializer(SerializeBeanInfo beanInfo) {
      this.beanInfo = beanInfo;
      this.sortedGetters = new FieldSerializer[beanInfo.sortedFields.length];

      for(int i = 0; i < this.sortedGetters.length; ++i) {
         this.sortedGetters[i] = new FieldSerializer(beanInfo.beanType, beanInfo.sortedFields[i]);
      }

      if (beanInfo.fields == beanInfo.sortedFields) {
         this.getters = this.sortedGetters;
      } else {
         this.getters = new FieldSerializer[beanInfo.fields.length];
         boolean hashNotMatch = false;

         for(int i = 0; i < this.getters.length; ++i) {
            FieldSerializer fieldSerializer = this.getFieldSerializer(beanInfo.fields[i].name);
            if (fieldSerializer == null) {
               hashNotMatch = true;
               break;
            }

            this.getters[i] = fieldSerializer;
         }

         if (hashNotMatch) {
            System.arraycopy(this.sortedGetters, 0, this.getters, 0, this.sortedGetters.length);
         }
      }

      if (beanInfo.jsonType != null) {
         for(Class<? extends SerializeFilter> filterClass : beanInfo.jsonType.serialzeFilters()) {
            try {
               SerializeFilter filter = (SerializeFilter)filterClass.getConstructor().newInstance();
               this.addFilter(filter);
            } catch (Exception var7) {
            }
         }
      }

   }

   public void writeDirectNonContext(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features) throws IOException {
      this.write(serializer, object, fieldName, fieldType, features);
   }

   public void writeAsArray(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features) throws IOException {
      this.write(serializer, object, fieldName, fieldType, features);
   }

   public void writeAsArrayNonContext(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features) throws IOException {
      this.write(serializer, object, fieldName, fieldType, features);
   }

   public void write(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features) throws IOException {
      this.write(serializer, object, fieldName, fieldType, features, false);
   }

   public void writeNoneASM(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features) throws IOException {
      this.write(serializer, object, fieldName, fieldType, features, false);
   }

   protected void write(JSONSerializer serializer, Object object, Object fieldName, Type fieldType, int features, boolean unwrapped) throws IOException {
      // $FF: Couldn't be decompiled
   }

   protected void writeClassName(JSONSerializer serializer, String typeKey, Object object) {
      if (typeKey == null) {
         typeKey = serializer.config.typeKey;
      }

      serializer.out.writeFieldName(typeKey, false);
      String typeName = this.beanInfo.typeName;
      if (typeName == null) {
         Class<?> clazz = object.getClass();
         if (TypeUtils.isProxy(clazz)) {
            clazz = clazz.getSuperclass();
         }

         typeName = clazz.getName();
      }

      serializer.write(typeName);
   }

   public boolean writeReference(JSONSerializer serializer, Object object, int fieldFeatures) {
      SerialContext context = serializer.context;
      int mask = SerializerFeature.DisableCircularReferenceDetect.mask;
      if (context != null && (context.features & mask) == 0 && (fieldFeatures & mask) == 0) {
         if (serializer.references != null && serializer.references.containsKey(object)) {
            serializer.writeReference(object);
            return true;
         } else {
            return false;
         }
      } else {
         return false;
      }
   }

   protected boolean isWriteAsArray(JSONSerializer serializer) {
      return this.isWriteAsArray(serializer, 0);
   }

   protected boolean isWriteAsArray(JSONSerializer serializer, int fieldFeatrues) {
      int mask = SerializerFeature.BeanToArray.mask;
      return (this.beanInfo.features & mask) != 0 || serializer.out.beanToArray || (fieldFeatrues & mask) != 0;
   }

   public Object getFieldValue(Object object, String key) {
      FieldSerializer fieldDeser = this.getFieldSerializer(key);
      if (fieldDeser == null) {
         throw new JSONException("field not found. " + key);
      } else {
         try {
            return fieldDeser.getPropertyValue(object);
         } catch (InvocationTargetException ex) {
            throw new JSONException("getFieldValue error." + key, ex);
         } catch (IllegalAccessException ex) {
            throw new JSONException("getFieldValue error." + key, ex);
         }
      }
   }

   public Object getFieldValue(Object object, String key, long keyHash, boolean throwFieldNotFoundException) {
      FieldSerializer fieldDeser = this.getFieldSerializer(keyHash);
      if (fieldDeser == null) {
         if (throwFieldNotFoundException) {
            throw new JSONException("field not found. " + key);
         } else {
            return null;
         }
      } else {
         try {
            return fieldDeser.getPropertyValue(object);
         } catch (InvocationTargetException ex) {
            throw new JSONException("getFieldValue error." + key, ex);
         } catch (IllegalAccessException ex) {
            throw new JSONException("getFieldValue error." + key, ex);
         }
      }
   }

   public FieldSerializer getFieldSerializer(String key) {
      if (key == null) {
         return null;
      } else {
         int low = 0;
         int high = this.sortedGetters.length - 1;

         while(low <= high) {
            int mid = low + high >>> 1;
            String fieldName = this.sortedGetters[mid].fieldInfo.name;
            int cmp = fieldName.compareTo(key);
            if (cmp < 0) {
               low = mid + 1;
            } else {
               if (cmp <= 0) {
                  return this.sortedGetters[mid];
               }

               high = mid - 1;
            }
         }

         return null;
      }
   }

   public FieldSerializer getFieldSerializer(long hash) {
      PropertyNamingStrategy[] namingStrategies = null;
      if (this.hashArray == null) {
         namingStrategies = PropertyNamingStrategy.values();
         long[] hashArray = new long[this.sortedGetters.length * namingStrategies.length];
         int index = 0;

         for(int i = 0; i < this.sortedGetters.length; ++i) {
            String name = this.sortedGetters[i].fieldInfo.name;
            hashArray[index++] = TypeUtils.fnv1a_64(name);

            for(int j = 0; j < namingStrategies.length; ++j) {
               String name_t = namingStrategies[j].translate(name);
               if (!name.equals(name_t)) {
                  hashArray[index++] = TypeUtils.fnv1a_64(name_t);
               }
            }
         }

         Arrays.sort(hashArray, 0, index);
         this.hashArray = new long[index];
         System.arraycopy(hashArray, 0, this.hashArray, 0, index);
      }

      int pos = Arrays.binarySearch(this.hashArray, hash);
      if (pos < 0) {
         return null;
      } else {
         if (this.hashArrayMapping == null) {
            if (namingStrategies == null) {
               namingStrategies = PropertyNamingStrategy.values();
            }

            short[] mapping = new short[this.hashArray.length];
            Arrays.fill(mapping, (short)-1);

            for(int i = 0; i < this.sortedGetters.length; ++i) {
               String name = this.sortedGetters[i].fieldInfo.name;
               int p = Arrays.binarySearch(this.hashArray, TypeUtils.fnv1a_64(name));
               if (p >= 0) {
                  mapping[p] = (short)i;
               }

               for(int j = 0; j < namingStrategies.length; ++j) {
                  String name_t = namingStrategies[j].translate(name);
                  if (!name.equals(name_t)) {
                     int p_t = Arrays.binarySearch(this.hashArray, TypeUtils.fnv1a_64(name_t));
                     if (p_t >= 0) {
                        mapping[p_t] = (short)i;
                     }
                  }
               }
            }

            this.hashArrayMapping = mapping;
         }

         int getterIndex = this.hashArrayMapping[pos];
         return getterIndex != -1 ? this.sortedGetters[getterIndex] : null;
      }
   }

   public List<Object> getFieldValues(Object object) throws Exception {
      List<Object> fieldValues = new ArrayList(this.sortedGetters.length);

      for(FieldSerializer getter : this.sortedGetters) {
         fieldValues.add(getter.getPropertyValue(object));
      }

      return fieldValues;
   }

   public List<Object> getObjectFieldValues(Object object) throws Exception {
      List<Object> fieldValues = new ArrayList(this.sortedGetters.length);

      for(FieldSerializer getter : this.sortedGetters) {
         Class fieldClass = getter.fieldInfo.fieldClass;
         if (!fieldClass.isPrimitive() && !fieldClass.getName().startsWith("java.lang.")) {
            fieldValues.add(getter.getPropertyValue(object));
         }
      }

      return fieldValues;
   }

   public int getSize(Object object) throws Exception {
      int size = 0;

      for(FieldSerializer getter : this.sortedGetters) {
         Object value = getter.getPropertyValueDirect(object);
         if (value != null) {
            ++size;
         }
      }

      return size;
   }

   public Set<String> getFieldNames(Object object) throws Exception {
      Set<String> fieldNames = new HashSet();

      for(FieldSerializer getter : this.sortedGetters) {
         Object value = getter.getPropertyValueDirect(object);
         if (value != null) {
            fieldNames.add(getter.fieldInfo.name);
         }
      }

      return fieldNames;
   }

   public Map<String, Object> getFieldValuesMap(Object object) throws Exception {
      Map<String, Object> map = new LinkedHashMap(this.sortedGetters.length);
      boolean skipTransient = true;
      FieldInfo fieldInfo = null;

      for(FieldSerializer getter : this.sortedGetters) {
         skipTransient = SerializerFeature.isEnabled(getter.features, SerializerFeature.SkipTransientField);
         fieldInfo = getter.fieldInfo;
         if (!skipTransient || fieldInfo == null || !fieldInfo.fieldTransient) {
            if (getter.fieldInfo.unwrapped) {
               Object unwrappedValue = getter.getPropertyValue(object);
               Object map1 = JSON.toJSON(unwrappedValue);
               if (map1 instanceof Map) {
                  map.putAll((Map)map1);
               } else {
                  map.put(getter.fieldInfo.name, getter.getPropertyValue(object));
               }
            } else {
               map.put(getter.fieldInfo.name, getter.getPropertyValue(object));
            }
         }
      }

      return map;
   }

   protected BeanContext getBeanContext(int orinal) {
      return this.sortedGetters[orinal].fieldContext;
   }

   protected Type getFieldType(int ordinal) {
      return this.sortedGetters[ordinal].fieldInfo.fieldType;
   }

   protected char writeBefore(JSONSerializer jsonBeanDeser, Object object, char seperator) {
      if (jsonBeanDeser.beforeFilters != null) {
         for(BeforeFilter beforeFilter : jsonBeanDeser.beforeFilters) {
            seperator = beforeFilter.writeBefore(jsonBeanDeser, object, seperator);
         }
      }

      if (this.beforeFilters != null) {
         for(BeforeFilter beforeFilter : this.beforeFilters) {
            seperator = beforeFilter.writeBefore(jsonBeanDeser, object, seperator);
         }
      }

      return seperator;
   }

   protected char writeAfter(JSONSerializer jsonBeanDeser, Object object, char seperator) {
      if (jsonBeanDeser.afterFilters != null) {
         for(AfterFilter afterFilter : jsonBeanDeser.afterFilters) {
            seperator = afterFilter.writeAfter(jsonBeanDeser, object, seperator);
         }
      }

      if (this.afterFilters != null) {
         for(AfterFilter afterFilter : this.afterFilters) {
            seperator = afterFilter.writeAfter(jsonBeanDeser, object, seperator);
         }
      }

      return seperator;
   }

   protected boolean applyLabel(JSONSerializer jsonBeanDeser, String label) {
      if (jsonBeanDeser.labelFilters != null) {
         for(LabelFilter propertyFilter : jsonBeanDeser.labelFilters) {
            if (!propertyFilter.apply(label)) {
               return false;
            }
         }
      }

      if (this.labelFilters != null) {
         for(LabelFilter propertyFilter : this.labelFilters) {
            if (!propertyFilter.apply(label)) {
               return false;
            }
         }
      }

      return true;
   }
}
