package org.apache.commons.jexl3;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.jexl3.internal.IntegerRange;
import org.apache.commons.jexl3.internal.LongRange;
import org.apache.commons.jexl3.introspection.JexlMethod;

public class JexlArithmetic {
   protected static final BigDecimal BIGD_DOUBLE_MAX_VALUE = BigDecimal.valueOf(Double.MAX_VALUE);
   protected static final BigDecimal BIGD_DOUBLE_MIN_VALUE = BigDecimal.valueOf(Double.MIN_VALUE);
   protected static final BigInteger BIGI_LONG_MAX_VALUE = BigInteger.valueOf(Long.MAX_VALUE);
   protected static final BigInteger BIGI_LONG_MIN_VALUE = BigInteger.valueOf(Long.MIN_VALUE);
   protected static final int BIGD_SCALE = -1;
   private final boolean strict;
   private final MathContext mathContext;
   private final int mathScale;
   public static final Pattern FLOAT_PATTERN = Pattern.compile("^[+-]?\\d*(\\.\\d*)?([eE][+-]?\\d+)?$");

   public JexlArithmetic(boolean astrict) {
      this(astrict, (MathContext)null, Integer.MIN_VALUE);
   }

   public JexlArithmetic(boolean astrict, MathContext bigdContext, int bigdScale) {
      this.strict = astrict;
      this.mathContext = bigdContext == null ? MathContext.DECIMAL128 : bigdContext;
      this.mathScale = bigdScale == Integer.MIN_VALUE ? -1 : bigdScale;
   }

   public JexlArithmetic options(JexlEngine.Options options) {
      Boolean ostrict = options.isStrictArithmetic();
      if (ostrict == null) {
         ostrict = this.isStrict();
      }

      MathContext bigdContext = options.getArithmeticMathContext();
      if (bigdContext == null) {
         bigdContext = this.getMathContext();
      }

      int bigdScale = options.getArithmeticMathScale();
      if (bigdScale == Integer.MIN_VALUE) {
         bigdScale = this.getMathScale();
      }

      return ostrict == this.isStrict() && bigdScale == this.getMathScale() && bigdContext == this.getMathContext() ? this : this.createWithOptions(ostrict, bigdContext, bigdScale);
   }

   public JexlArithmetic options(JexlContext context) {
      return context instanceof JexlEngine.Options ? this.options((JexlEngine.Options)context) : this;
   }

   protected JexlArithmetic createWithOptions(boolean astrict, MathContext bigdContext, int bigdScale) {
      return new JexlArithmetic(astrict, bigdContext, bigdScale);
   }

   public ArrayBuilder arrayBuilder(int size) {
      return new org.apache.commons.jexl3.internal.ArrayBuilder(size);
   }

   public SetBuilder setBuilder(int size) {
      return new org.apache.commons.jexl3.internal.SetBuilder(size);
   }

   public MapBuilder mapBuilder(int size) {
      return new org.apache.commons.jexl3.internal.MapBuilder(size);
   }

   public Iterable<?> createRange(Object from, Object to) throws ArithmeticException {
      long lfrom = this.toLong(from);
      long lto = this.toLong(to);
      return (Iterable<?>)(lfrom >= -2147483648L && lfrom <= 2147483647L && lto >= -2147483648L && lto <= 2147483647L ? IntegerRange.create((int)lfrom, (int)lto) : LongRange.create(lfrom, lto));
   }

   public boolean isStrict() {
      return this.strict;
   }

   public MathContext getMathContext() {
      return this.mathContext;
   }

   public int getMathScale() {
      return this.mathScale;
   }

   protected BigDecimal roundBigDecimal(BigDecimal number) {
      int mscale = this.getMathScale();
      return mscale >= 0 ? number.setScale(mscale, this.getMathContext().getRoundingMode()) : number;
   }

   protected Object controlNullNullOperands() {
      if (this.isStrict()) {
         throw new NullOperand();
      } else {
         return 0;
      }
   }

   protected void controlNullOperand() {
      if (this.isStrict()) {
         throw new NullOperand();
      }
   }

   protected boolean isFloatingPointNumber(Object val) {
      if (!(val instanceof Float) && !(val instanceof Double)) {
         if (!(val instanceof CharSequence)) {
            return false;
         } else {
            Matcher m = FLOAT_PATTERN.matcher((CharSequence)val);
            return m.matches() && (m.start(1) >= 0 || m.start(2) >= 0);
         }
      } else {
         return true;
      }
   }

   protected boolean isFloatingPoint(Object o) {
      return o instanceof Float || o instanceof Double;
   }

   protected boolean isNumberable(Object o) {
      return o instanceof Integer || o instanceof Long || o instanceof Byte || o instanceof Short || o instanceof Character;
   }

   public Number narrow(Number original) {
      return this.narrowNumber(original, (Class)null);
   }

   protected boolean narrowAccept(Class<?> narrow, Class<?> source) {
      return narrow == null || narrow.equals(source);
   }

   public Number narrowNumber(Number original, Class<?> narrow) {
      if (original == null) {
         return null;
      } else {
         Number result = original;
         if (original instanceof BigDecimal) {
            BigDecimal bigd = (BigDecimal)original;
            if (bigd.compareTo(BIGD_DOUBLE_MAX_VALUE) > 0 || bigd.compareTo(BIGD_DOUBLE_MIN_VALUE) < 0) {
               return original;
            }

            try {
               long l = bigd.longValueExact();
               if (this.narrowAccept(narrow, Integer.class) && l <= 2147483647L && l >= -2147483648L) {
                  return (int)l;
               }

               if (this.narrowAccept(narrow, Long.class)) {
                  return l;
               }
            } catch (ArithmeticException var7) {
            }
         }

         if (!(original instanceof Double) && !(original instanceof Float)) {
            if (original instanceof BigInteger) {
               BigInteger bigi = (BigInteger)original;
               if (bigi.compareTo(BIGI_LONG_MAX_VALUE) > 0 || bigi.compareTo(BIGI_LONG_MIN_VALUE) < 0) {
                  return original;
               }
            }

            long value = original.longValue();
            if (this.narrowAccept(narrow, Byte.class) && value <= 127L && value >= -128L) {
               result = (byte)((int)value);
            } else if (this.narrowAccept(narrow, Short.class) && value <= 32767L && value >= -32768L) {
               result = (short)((int)value);
            } else if (this.narrowAccept(narrow, Integer.class) && value <= 2147483647L && value >= -2147483648L) {
               result = (int)value;
            }
         } else {
            double value = original.doubleValue();
            if (this.narrowAccept(narrow, Float.class) && value <= (double)Float.MAX_VALUE && value >= (double)Float.MIN_VALUE) {
               result = original.floatValue();
            }
         }

         return result;
      }
   }

   protected Number narrowBigInteger(Object lhs, Object rhs, BigInteger bigi) {
      if (!(lhs instanceof BigInteger) && !(rhs instanceof BigInteger) && bigi.compareTo(BIGI_LONG_MAX_VALUE) <= 0 && bigi.compareTo(BIGI_LONG_MIN_VALUE) >= 0) {
         long l = bigi.longValue();
         return (Number)(!(lhs instanceof Long) && !(rhs instanceof Long) && l <= 2147483647L && l >= -2147483648L ? (int)l : l);
      } else {
         return bigi;
      }
   }

   protected Number narrowBigDecimal(Object lhs, Object rhs, BigDecimal bigd) {
      if (this.isNumberable(lhs) || this.isNumberable(rhs)) {
         try {
            long l = bigd.longValueExact();
            if (l <= 2147483647L && l >= -2147483648L) {
               return (int)l;
            }

            return l;
         } catch (ArithmeticException var6) {
         }
      }

      return bigd;
   }

   public boolean narrowArguments(Object[] args) {
      boolean narrowed = false;

      for(int a = 0; a < args.length; ++a) {
         Object arg = args[a];
         if (arg instanceof Number) {
            Number narg = (Number)arg;
            Number narrow = this.narrow(narg);
            if (!narg.equals(narrow)) {
               args[a] = narrow;
               narrowed = true;
            }
         }
      }

      return narrowed;
   }

   public Object add(Object left, Object right) {
      if (left == null && right == null) {
         return this.controlNullNullOperands();
      } else {
         boolean strconcat = this.strict ? left instanceof String || right instanceof String : left instanceof String && right instanceof String;
         if (!strconcat) {
            try {
               if (!(left instanceof BigDecimal) && !(right instanceof BigDecimal)) {
                  if (!this.isFloatingPointNumber(left) && !this.isFloatingPointNumber(right)) {
                     BigInteger l = this.toBigInteger(left);
                     BigInteger r = this.toBigInteger(right);
                     BigInteger result = l.add(r);
                     return this.narrowBigInteger(left, right, result);
                  }

                  double l = this.toDouble(left);
                  double r = this.toDouble(right);
                  return l + r;
               }

               BigDecimal l = this.toBigDecimal(left);
               BigDecimal r = this.toBigDecimal(right);
               BigDecimal result = l.add(r, this.getMathContext());
               return this.narrowBigDecimal(left, right, result);
            } catch (NumberFormatException var8) {
               if (left == null || right == null) {
                  this.controlNullOperand();
               }
            }
         }

         return this.toString(left).concat(this.toString(right));
      }
   }

   public Object divide(Object left, Object right) {
      if (left == null && right == null) {
         return this.controlNullNullOperands();
      } else if (!(left instanceof BigDecimal) && !(right instanceof BigDecimal)) {
         if (!this.isFloatingPointNumber(left) && !this.isFloatingPointNumber(right)) {
            BigInteger l = this.toBigInteger(left);
            BigInteger r = this.toBigInteger(right);
            if (BigInteger.ZERO.equals(r)) {
               throw new ArithmeticException("/");
            } else {
               BigInteger result = l.divide(r);
               return this.narrowBigInteger(left, right, result);
            }
         } else {
            double l = this.toDouble(left);
            double r = this.toDouble(right);
            if (r == (double)0.0F) {
               throw new ArithmeticException("/");
            } else {
               return l / r;
            }
         }
      } else {
         BigDecimal l = this.toBigDecimal(left);
         BigDecimal r = this.toBigDecimal(right);
         if (BigDecimal.ZERO.equals(r)) {
            throw new ArithmeticException("/");
         } else {
            BigDecimal result = l.divide(r, this.getMathContext());
            return this.narrowBigDecimal(left, right, result);
         }
      }
   }

   public Object mod(Object left, Object right) {
      if (left == null && right == null) {
         return this.controlNullNullOperands();
      } else if (!(left instanceof BigDecimal) && !(right instanceof BigDecimal)) {
         if (!this.isFloatingPointNumber(left) && !this.isFloatingPointNumber(right)) {
            BigInteger l = this.toBigInteger(left);
            BigInteger r = this.toBigInteger(right);
            if (BigInteger.ZERO.equals(r)) {
               throw new ArithmeticException("%");
            } else {
               BigInteger result = l.mod(r);
               return this.narrowBigInteger(left, right, result);
            }
         } else {
            double l = this.toDouble(left);
            double r = this.toDouble(right);
            if (r == (double)0.0F) {
               throw new ArithmeticException("%");
            } else {
               return l % r;
            }
         }
      } else {
         BigDecimal l = this.toBigDecimal(left);
         BigDecimal r = this.toBigDecimal(right);
         if (BigDecimal.ZERO.equals(r)) {
            throw new ArithmeticException("%");
         } else {
            BigDecimal remainder = l.remainder(r, this.getMathContext());
            return this.narrowBigDecimal(left, right, remainder);
         }
      }
   }

   public Object multiply(Object left, Object right) {
      if (left == null && right == null) {
         return this.controlNullNullOperands();
      } else if (!(left instanceof BigDecimal) && !(right instanceof BigDecimal)) {
         if (!this.isFloatingPointNumber(left) && !this.isFloatingPointNumber(right)) {
            BigInteger l = this.toBigInteger(left);
            BigInteger r = this.toBigInteger(right);
            BigInteger result = l.multiply(r);
            return this.narrowBigInteger(left, right, result);
         } else {
            double l = this.toDouble(left);
            double r = this.toDouble(right);
            return l * r;
         }
      } else {
         BigDecimal l = this.toBigDecimal(left);
         BigDecimal r = this.toBigDecimal(right);
         BigDecimal result = l.multiply(r, this.getMathContext());
         return this.narrowBigDecimal(left, right, result);
      }
   }

   public Object subtract(Object left, Object right) {
      if (left == null && right == null) {
         return this.controlNullNullOperands();
      } else if (!(left instanceof BigDecimal) && !(right instanceof BigDecimal)) {
         if (!this.isFloatingPointNumber(left) && !this.isFloatingPointNumber(right)) {
            BigInteger l = this.toBigInteger(left);
            BigInteger r = this.toBigInteger(right);
            BigInteger result = l.subtract(r);
            return this.narrowBigInteger(left, right, result);
         } else {
            double l = this.toDouble(left);
            double r = this.toDouble(right);
            return l - r;
         }
      } else {
         BigDecimal l = this.toBigDecimal(left);
         BigDecimal r = this.toBigDecimal(right);
         BigDecimal result = l.subtract(r, this.getMathContext());
         return this.narrowBigDecimal(left, right, result);
      }
   }

   public Object negate(Object val) {
      if (val instanceof Integer) {
         return -(Integer)val;
      } else if (val instanceof Double) {
         return -(Double)val;
      } else if (val instanceof Long) {
         return -(Long)val;
      } else if (val instanceof BigDecimal) {
         return ((BigDecimal)val).negate();
      } else if (val instanceof BigInteger) {
         return ((BigInteger)val).negate();
      } else if (val instanceof Float) {
         return -(Float)val;
      } else if (val instanceof Short) {
         return (short)(-(Short)val);
      } else if (val instanceof Byte) {
         return (byte)(-(Byte)val);
      } else if (val instanceof Boolean) {
         return (Boolean)val ? Boolean.FALSE : Boolean.TRUE;
      } else if (val instanceof AtomicBoolean) {
         return ((AtomicBoolean)val).get() ? Boolean.FALSE : Boolean.TRUE;
      } else {
         throw new ArithmeticException("Object negation:(" + val + ")");
      }
   }

   public Boolean contains(Object container, Object value) {
      if (value == null && container == null) {
         return true;
      } else if (value != null && container != null) {
         if (container instanceof Pattern) {
            return ((Pattern)container).matcher(value.toString()).matches();
         } else if (container instanceof String) {
            return value.toString().matches(container.toString());
         } else if (container instanceof Map) {
            return value instanceof Map ? ((Map)container).keySet().containsAll(((Map)value).keySet()) : ((Map)container).containsKey(value);
         } else if (container instanceof Collection) {
            return value instanceof Collection ? ((Collection)container).containsAll((Collection)value) : ((Collection)container).contains(value);
         } else {
            return null;
         }
      } else {
         return false;
      }
   }

   public Boolean endsWith(Object left, Object right) {
      if (left == null && right == null) {
         return true;
      } else if (left != null && right != null) {
         return left instanceof String ? ((String)left).endsWith(this.toString(right)) : null;
      } else {
         return false;
      }
   }

   public Boolean startsWith(Object left, Object right) {
      if (left == null && right == null) {
         return true;
      } else if (left != null && right != null) {
         return left instanceof String ? ((String)left).startsWith(this.toString(right)) : null;
      } else {
         return false;
      }
   }

   public Boolean isEmpty(Object object) {
      if (!(object instanceof Number)) {
         if (object instanceof String) {
            return "".equals(object) ? Boolean.TRUE : Boolean.FALSE;
         } else if (object.getClass().isArray()) {
            return Array.getLength(object) == 0 ? Boolean.TRUE : Boolean.FALSE;
         } else if (object instanceof Collection) {
            return ((Collection)object).isEmpty() ? Boolean.TRUE : Boolean.FALSE;
         } else if (object instanceof Map) {
            return ((Map)object).isEmpty() ? Boolean.TRUE : Boolean.FALSE;
         } else {
            return null;
         }
      } else {
         double d = ((Number)object).doubleValue();
         return !Double.isNaN(d) && d != (double)0.0F ? Boolean.FALSE : Boolean.TRUE;
      }
   }

   public Integer size(Object object) {
      if (object instanceof String) {
         return ((String)object).length();
      } else if (object.getClass().isArray()) {
         return Array.getLength(object);
      } else if (object instanceof Collection) {
         return ((Collection)object).size();
      } else {
         return object instanceof Map ? ((Map)object).size() : null;
      }
   }

   public Object and(Object left, Object right) {
      long l = this.toLong(left);
      long r = this.toLong(right);
      return l & r;
   }

   public Object or(Object left, Object right) {
      long l = this.toLong(left);
      long r = this.toLong(right);
      return l | r;
   }

   public Object xor(Object left, Object right) {
      long l = this.toLong(left);
      long r = this.toLong(right);
      return l ^ r;
   }

   public Object complement(Object val) {
      long l = this.toLong(val);
      return ~l;
   }

   public Object not(Object val) {
      return this.toBoolean(val) ? Boolean.FALSE : Boolean.TRUE;
   }

   protected int compare(Object left, Object right, String operator) {
      if (left != null && right != null) {
         if (left instanceof BigDecimal || right instanceof BigDecimal) {
            BigDecimal l = this.toBigDecimal(left);
            BigDecimal r = this.toBigDecimal(right);
            return l.compareTo(r);
         } else if (left instanceof BigInteger || right instanceof BigInteger) {
            BigInteger l = this.toBigInteger(left);
            BigInteger r = this.toBigInteger(right);
            return l.compareTo(r);
         } else if (!this.isFloatingPoint(left) && !this.isFloatingPoint(right)) {
            if (!this.isNumberable(left) && !this.isNumberable(right)) {
               if (left instanceof String || right instanceof String) {
                  return this.toString(left).compareTo(this.toString(right));
               } else if ("==".equals(operator)) {
                  return left.equals(right) ? 0 : -1;
               } else if (left instanceof Comparable) {
                  Comparable<Object> comparable = (Comparable)left;
                  return comparable.compareTo(right);
               } else if (right instanceof Comparable) {
                  Comparable<Object> comparable = (Comparable)right;
                  return comparable.compareTo(left);
               } else {
                  throw new ArithmeticException("Object comparison:(" + left + " " + operator + " " + right + ")");
               }
            } else {
               long lhs = this.toLong(left);
               long rhs = this.toLong(right);
               if (lhs < rhs) {
                  return -1;
               } else if (lhs > rhs) {
                  return 1;
               } else {
                  return 0;
               }
            }
         } else {
            double lhs = this.toDouble(left);
            double rhs = this.toDouble(right);
            if (Double.isNaN(lhs)) {
               if (Double.isNaN(rhs)) {
                  return 0;
               } else {
                  return -1;
               }
            } else if (Double.isNaN(rhs)) {
               return 1;
            } else if (lhs < rhs) {
               return -1;
            } else if (lhs > rhs) {
               return 1;
            } else {
               return 0;
            }
         }
      } else {
         throw new ArithmeticException("Object comparison:(" + left + " " + operator + " " + right + ")");
      }
   }

   public boolean equals(Object left, Object right) {
      if (left == right) {
         return true;
      } else if (left != null && right != null) {
         if (!(left instanceof Boolean) && !(right instanceof Boolean)) {
            return this.compare(left, right, "==") == 0;
         } else {
            return this.toBoolean(left) == this.toBoolean(right);
         }
      } else {
         return false;
      }
   }

   public boolean lessThan(Object left, Object right) {
      if (left != right && left != null && right != null) {
         return this.compare(left, right, "<") < 0;
      } else {
         return false;
      }
   }

   public boolean greaterThan(Object left, Object right) {
      if (left != right && left != null && right != null) {
         return this.compare(left, right, ">") > 0;
      } else {
         return false;
      }
   }

   public boolean lessThanOrEqual(Object left, Object right) {
      if (left == right) {
         return true;
      } else if (left != null && right != null) {
         return this.compare(left, right, "<=") <= 0;
      } else {
         return false;
      }
   }

   public boolean greaterThanOrEqual(Object left, Object right) {
      if (left == right) {
         return true;
      } else if (left != null && right != null) {
         return this.compare(left, right, ">=") >= 0;
      } else {
         return false;
      }
   }

   public boolean toBoolean(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return false;
      } else if (val instanceof Boolean) {
         return (Boolean)val;
      } else if (val instanceof Number) {
         double number = this.toDouble(val);
         return !Double.isNaN(number) && number != (double)0.0F;
      } else if (val instanceof AtomicBoolean) {
         return ((AtomicBoolean)val).get();
      } else if (!(val instanceof String)) {
         return true;
      } else {
         String strval = val.toString();
         return strval.length() > 0 && !"false".equals(strval);
      }
   }

   public int toInteger(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return 0;
      } else if (val instanceof Double) {
         Double dval = (Double)val;
         return Double.isNaN(dval) ? 0 : dval.intValue();
      } else if (val instanceof Number) {
         return ((Number)val).intValue();
      } else if (val instanceof String) {
         return "".equals(val) ? 0 : Integer.parseInt((String)val);
      } else if (val instanceof Boolean) {
         return (Boolean)val ? 1 : 0;
      } else if (val instanceof AtomicBoolean) {
         return ((AtomicBoolean)val).get() ? 1 : 0;
      } else if (val instanceof Character) {
         return (Character)val;
      } else {
         throw new ArithmeticException("Integer coercion: " + val.getClass().getName() + ":(" + val + ")");
      }
   }

   public long toLong(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return 0L;
      } else if (val instanceof Double) {
         Double dval = (Double)val;
         return Double.isNaN(dval) ? 0L : dval.longValue();
      } else if (val instanceof Number) {
         return ((Number)val).longValue();
      } else if (val instanceof String) {
         return "".equals(val) ? 0L : Long.parseLong((String)val);
      } else if (val instanceof Boolean) {
         return (Boolean)val ? 1L : 0L;
      } else if (val instanceof AtomicBoolean) {
         return ((AtomicBoolean)val).get() ? 1L : 0L;
      } else if (val instanceof Character) {
         return (long)(Character)val;
      } else {
         throw new ArithmeticException("Long coercion: " + val.getClass().getName() + ":(" + val + ")");
      }
   }

   public BigInteger toBigInteger(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return BigInteger.ZERO;
      } else if (val instanceof BigInteger) {
         return (BigInteger)val;
      } else if (val instanceof Double) {
         Double dval = (Double)val;
         return Double.isNaN(dval) ? BigInteger.ZERO : BigInteger.valueOf(dval.longValue());
      } else if (val instanceof BigDecimal) {
         return ((BigDecimal)val).toBigInteger();
      } else if (val instanceof Number) {
         return BigInteger.valueOf(((Number)val).longValue());
      } else if (val instanceof Boolean) {
         return BigInteger.valueOf((Boolean)val ? 1L : 0L);
      } else if (val instanceof AtomicBoolean) {
         return BigInteger.valueOf(((AtomicBoolean)val).get() ? 1L : 0L);
      } else if (val instanceof String) {
         String string = (String)val;
         return "".equals(string) ? BigInteger.ZERO : new BigInteger(string);
      } else if (val instanceof Character) {
         int i = (Character)val;
         return BigInteger.valueOf((long)i);
      } else {
         throw new ArithmeticException("BigInteger coercion: " + val.getClass().getName() + ":(" + val + ")");
      }
   }

   public BigDecimal toBigDecimal(Object val) {
      if (val instanceof BigDecimal) {
         return this.roundBigDecimal((BigDecimal)val);
      } else if (val == null) {
         this.controlNullOperand();
         return BigDecimal.ZERO;
      } else if (val instanceof Double) {
         return Double.isNaN((Double)val) ? BigDecimal.ZERO : this.roundBigDecimal(new BigDecimal(val.toString(), this.getMathContext()));
      } else if (val instanceof Number) {
         return this.roundBigDecimal(new BigDecimal(val.toString(), this.getMathContext()));
      } else if (val instanceof Boolean) {
         return BigDecimal.valueOf((Boolean)val ? (double)1.0F : (double)0.0F);
      } else if (val instanceof AtomicBoolean) {
         return BigDecimal.valueOf(((AtomicBoolean)val).get() ? 1L : 0L);
      } else if (val instanceof String) {
         String string = (String)val;
         return "".equals(string) ? BigDecimal.ZERO : this.roundBigDecimal(new BigDecimal(string, this.getMathContext()));
      } else if (val instanceof Character) {
         int i = (Character)val;
         return new BigDecimal(i);
      } else {
         throw new ArithmeticException("BigDecimal coercion: " + val.getClass().getName() + ":(" + val + ")");
      }
   }

   public double toDouble(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return (double)0.0F;
      } else if (val instanceof Double) {
         return (Double)val;
      } else if (val instanceof Number) {
         return Double.parseDouble(String.valueOf(val));
      } else if (val instanceof Boolean) {
         return (Boolean)val ? (double)1.0F : (double)0.0F;
      } else if (val instanceof AtomicBoolean) {
         return ((AtomicBoolean)val).get() ? (double)1.0F : (double)0.0F;
      } else if (val instanceof String) {
         String string = (String)val;
         return "".equals(string) ? Double.NaN : Double.parseDouble(string);
      } else if (val instanceof Character) {
         int i = (Character)val;
         return (double)i;
      } else {
         throw new ArithmeticException("Double coercion: " + val.getClass().getName() + ":(" + val + ")");
      }
   }

   public String toString(Object val) {
      if (val == null) {
         this.controlNullOperand();
         return "";
      } else if (val instanceof Double) {
         Double dval = (Double)val;
         return Double.isNaN(dval) ? "" : dval.toString();
      } else {
         return val.toString();
      }
   }

   /** @deprecated */
   @Deprecated
   public final Object bitwiseAnd(Object lhs, Object rhs) {
      return this.and(lhs, rhs);
   }

   /** @deprecated */
   @Deprecated
   public final Object bitwiseOr(Object lhs, Object rhs) {
      return this.or(lhs, rhs);
   }

   /** @deprecated */
   @Deprecated
   public final Object bitwiseXor(Object lhs, Object rhs) {
      return this.xor(lhs, rhs);
   }

   /** @deprecated */
   @Deprecated
   public final Object logicalNot(Object arg) {
      return this.not(arg);
   }

   /** @deprecated */
   @Deprecated
   public final Object matches(Object lhs, Object rhs) {
      return this.contains(rhs, lhs);
   }

   public static class NullOperand extends ArithmeticException {
   }

   public interface ArrayBuilder {
      void add(Object var1);

      Object create(boolean var1);
   }

   public interface MapBuilder {
      void put(Object var1, Object var2);

      Object create();
   }

   public interface SetBuilder {
      void add(Object var1);

      Object create();
   }

   public interface Uberspect {
      boolean overloads(JexlOperator var1);

      JexlMethod getOperator(JexlOperator var1, Object... var2);
   }
}
