package org.apache.commons.jexl3;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.MathContext;
import java.net.URL;
import java.nio.charset.Charset;
import org.apache.commons.jexl3.introspection.JexlUberspect;

public abstract class JexlEngine {
   public static final Object TRY_FAILED = new Object() {
      public String toString() {
         return "tryExecute failed";
      }
   };
   protected static final ThreadLocal<JexlContext.ThreadLocal> CONTEXT = new ThreadLocal<JexlContext.ThreadLocal>() {
      protected JexlContext.ThreadLocal initialValue() {
         return null;
      }
   };
   public static final JexlContext EMPTY_CONTEXT = new JexlContext() {
      public Object get(String name) {
         return null;
      }

      public boolean has(String name) {
         return false;
      }

      public void set(String name, Object value) {
         throw new UnsupportedOperationException("Not supported in void context.");
      }
   };
   public static final JexlContext.NamespaceResolver EMPTY_NS = new JexlContext.NamespaceResolver() {
      public Object resolveNamespace(String name) {
         return null;
      }
   };
   private static final int JXLT_CACHE_SIZE = 256;

   public static JexlContext.ThreadLocal getThreadContext() {
      return (JexlContext.ThreadLocal)CONTEXT.get();
   }

   public static void setThreadContext(JexlContext.ThreadLocal tls) {
      CONTEXT.set(tls);
   }

   public abstract Charset getCharset();

   public abstract JexlUberspect getUberspect();

   public abstract JexlArithmetic getArithmetic();

   public abstract boolean isDebug();

   public abstract boolean isSilent();

   public abstract boolean isStrict();

   public abstract boolean isCancellable();

   public abstract void setClassLoader(ClassLoader var1);

   public JxltEngine createJxltEngine() {
      return this.createJxltEngine(true);
   }

   public JxltEngine createJxltEngine(boolean noScript) {
      return this.createJxltEngine(noScript, 256, '$', '#');
   }

   public abstract JxltEngine createJxltEngine(boolean var1, int var2, char var3, char var4);

   public abstract void clearCache();

   public abstract JexlExpression createExpression(JexlInfo var1, String var2);

   public abstract JexlScript createScript(JexlInfo var1, String var2, String[] var3);

   public final JexlExpression createExpression(String expression) {
      return this.createExpression((JexlInfo)null, expression);
   }

   public final JexlScript createScript(String scriptText) {
      return this.createScript((JexlInfo)null, (String)scriptText, (String[])null);
   }

   public final JexlScript createScript(String scriptText, String... names) {
      return this.createScript((JexlInfo)null, (String)scriptText, names);
   }

   public final JexlScript createScript(File scriptFile) {
      return this.createScript((JexlInfo)null, (String)this.readSource(scriptFile), (String[])null);
   }

   public final JexlScript createScript(File scriptFile, String... names) {
      return this.createScript((JexlInfo)null, (String)this.readSource(scriptFile), names);
   }

   public final JexlScript createScript(JexlInfo info, File scriptFile, String[] names) {
      return this.createScript(info, this.readSource(scriptFile), names);
   }

   public final JexlScript createScript(URL scriptUrl) {
      return this.createScript((JexlInfo)null, (String)this.readSource(scriptUrl), (String[])null);
   }

   public final JexlScript createScript(URL scriptUrl, String[] names) {
      return this.createScript((JexlInfo)null, (String)this.readSource(scriptUrl), names);
   }

   public final JexlScript createScript(JexlInfo info, URL scriptUrl, String[] names) {
      return this.createScript(info, this.readSource(scriptUrl), names);
   }

   public abstract Object getProperty(Object var1, String var2);

   public abstract Object getProperty(JexlContext var1, Object var2, String var3);

   public abstract void setProperty(Object var1, String var2, Object var3);

   public abstract void setProperty(JexlContext var1, Object var2, String var3, Object var4);

   public abstract Object invokeMethod(Object var1, String var2, Object... var3);

   public abstract <T> T newInstance(Class<? extends T> var1, Object... var2);

   public abstract Object newInstance(String var1, Object... var2);

   public JexlInfo createInfo(String fn, int l, int c) {
      return new JexlInfo(fn, l, c);
   }

   public JexlInfo createInfo() {
      JexlInfo info = null;
      StackTraceElement[] stack = (new Throwable()).getStackTrace();
      StackTraceElement se = null;
      String name = this.getClass().getName();

      for(int s = 1; s < stack.length; ++s) {
         se = stack[s];
         String className = se.getClassName();
         if (!className.equals(name)) {
            if (!className.startsWith("org.apache.commons.jexl3.internal.") && !className.startsWith("org.apache.commons.jexl3.J")) {
               break;
            }

            name = className;
         }
      }

      if (se != null) {
         info = this.createInfo(se.getClassName() + "." + se.getMethodName(), se.getLineNumber(), 0);
      }

      return info;
   }

   protected static String toString(BufferedReader reader) throws IOException {
      StringBuilder buffer = new StringBuilder();

      String line;
      while((line = reader.readLine()) != null) {
         buffer.append(line).append('\n');
      }

      return buffer.toString();
   }

   protected String readSource(File file) {
      if (file == null) {
         throw new NullPointerException("source file is null");
      } else {
         BufferedReader reader = null;

         String var3;
         try {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), this.getCharset()));
            var3 = toString(reader);
         } catch (IOException xio) {
            throw new JexlException(this.createInfo(file.toString(), 1, 1), "could not read source File", xio);
         } finally {
            if (reader != null) {
               try {
                  reader.close();
               } catch (IOException var11) {
               }
            }

         }

         return var3;
      }
   }

   protected String readSource(URL url) {
      if (url == null) {
         throw new NullPointerException("source URL is null");
      } else {
         BufferedReader reader = null;

         String var3;
         try {
            reader = new BufferedReader(new InputStreamReader(url.openStream(), this.getCharset()));
            var3 = toString(reader);
         } catch (IOException xio) {
            throw new JexlException(this.createInfo(url.toString(), 1, 1), "could not read source URL", xio);
         } finally {
            if (reader != null) {
               try {
                  reader.close();
               } catch (IOException var11) {
               }
            }

         }

         return var3;
      }
   }

   public interface Options {
      Charset getCharset();

      Boolean isSilent();

      Boolean isStrict();

      Boolean isStrictArithmetic();

      Boolean isCancellable();

      MathContext getArithmeticMathContext();

      int getArithmeticMathScale();
   }
}
