package org.apache.commons.jexl3.internal;

import java.io.StringReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.jexl3.JexlArithmetic;
import org.apache.commons.jexl3.JexlBuilder;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlEngine;
import org.apache.commons.jexl3.JexlException;
import org.apache.commons.jexl3.JexlInfo;
import org.apache.commons.jexl3.JexlScript;
import org.apache.commons.jexl3.internal.introspection.SandboxUberspect;
import org.apache.commons.jexl3.internal.introspection.Uberspect;
import org.apache.commons.jexl3.introspection.JexlMethod;
import org.apache.commons.jexl3.introspection.JexlSandbox;
import org.apache.commons.jexl3.introspection.JexlUberspect;
import org.apache.commons.jexl3.parser.ASTArrayAccess;
import org.apache.commons.jexl3.parser.ASTFunctionNode;
import org.apache.commons.jexl3.parser.ASTIdentifier;
import org.apache.commons.jexl3.parser.ASTIdentifierAccess;
import org.apache.commons.jexl3.parser.ASTJexlScript;
import org.apache.commons.jexl3.parser.ASTMethodNode;
import org.apache.commons.jexl3.parser.JexlNode;
import org.apache.commons.jexl3.parser.Parser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Engine extends JexlEngine {
   protected final JexlUberspect uberspect;
   protected final JexlArithmetic arithmetic;
   protected final Log logger;
   protected final AtomicBoolean parsing;
   protected final Parser parser;
   protected final boolean strict;
   protected final boolean silent;
   protected final boolean cancellable;
   protected final boolean debug;
   protected final Map<String, Object> functions;
   protected final SoftCache<String, ASTJexlScript> cache;
   protected final int cacheThreshold;
   protected final Charset charset;
   protected volatile TemplateEngine jxlt;

   public Engine() {
      this(new JexlBuilder());
   }

   public Engine(JexlBuilder conf) {
      this.parsing = new AtomicBoolean(false);
      this.parser = new Parser(new StringReader(";"));
      this.jxlt = null;
      JexlSandbox sandbox = conf.sandbox();
      JexlUberspect uber = (JexlUberspect)(conf.uberspect() == null ? getUberspect(conf.logger(), conf.strategy()) : conf.uberspect());
      ClassLoader loader = conf.loader();
      if (loader != null) {
         uber.setClassLoader(loader);
      }

      if (sandbox == null) {
         this.uberspect = uber;
      } else {
         this.uberspect = new SandboxUberspect(uber, sandbox);
      }

      this.logger = conf.logger() == null ? LogFactory.getLog(JexlEngine.class) : conf.logger();
      this.functions = conf.namespaces() == null ? Collections.emptyMap() : conf.namespaces();
      this.strict = conf.strict() == null ? true : conf.strict();
      this.silent = conf.silent() == null ? false : conf.silent();
      this.cancellable = conf.cancellable() == null ? !this.silent && this.strict : conf.cancellable();
      this.debug = conf.debug() == null ? true : conf.debug();
      this.arithmetic = conf.arithmetic() == null ? new JexlArithmetic(this.strict) : conf.arithmetic();
      this.cache = conf.cache() <= 0 ? null : new SoftCache(conf.cache());
      this.cacheThreshold = conf.cacheThreshold();
      this.charset = conf.charset();
      if (this.uberspect == null) {
         throw new IllegalArgumentException("uberspect can not be null");
      }
   }

   public static Uberspect getUberspect(Log logger, JexlUberspect.ResolverStrategy strategy) {
      return logger != null && !logger.equals(LogFactory.getLog(JexlEngine.class)) || strategy != null && strategy != JexlUberspect.JEXL_STRATEGY ? new Uberspect(logger, strategy) : UberspectHolder.UBERSPECT;
   }

   public JexlUberspect getUberspect() {
      return this.uberspect;
   }

   public JexlArithmetic getArithmetic() {
      return this.arithmetic;
   }

   public boolean isDebug() {
      return this.debug;
   }

   public boolean isSilent() {
      return this.silent;
   }

   public boolean isStrict() {
      return this.strict;
   }

   public boolean isCancellable() {
      return this.cancellable;
   }

   public void setClassLoader(ClassLoader loader) {
      this.uberspect.setClassLoader(loader);
   }

   public Charset getCharset() {
      return this.charset;
   }

   public TemplateEngine createJxltEngine(boolean noScript, int cacheSize, char immediate, char deferred) {
      return new TemplateEngine(this, noScript, cacheSize, immediate, deferred);
   }

   public void clearCache() {
      if (this.cache != null) {
         this.cache.clear();
      }

   }

   protected Interpreter createInterpreter(JexlContext context, Scope.Frame frame) {
      return new Interpreter(this, context, frame);
   }

   public Script createScript(JexlInfo info, String scriptText, String[] names) {
      if (scriptText == null) {
         throw new NullPointerException("source is null");
      } else {
         String source = this.trimSource(scriptText);
         Scope scope = names == null ? null : new Scope((Scope)null, names);
         ASTJexlScript tree = this.parse(info, source, scope, false, false);
         return new Script(this, source, tree);
      }
   }

   public Script createExpression(JexlInfo info, String expression) {
      if (expression == null) {
         throw new NullPointerException("source is null");
      } else {
         String source = this.trimSource(expression);
         ASTJexlScript tree = this.parse(info, source, (Scope)null, false, true);
         return new Script(this, source, tree);
      }
   }

   public Object getProperty(Object bean, String expr) {
      return this.getProperty((JexlContext)null, bean, expr);
   }

   public Object getProperty(JexlContext context, Object bean, String expr) {
      if (context == null) {
         context = EMPTY_CONTEXT;
      }

      String src = this.trimSource(expr);
      src = "#0" + (src.charAt(0) == '[' ? "" : ".") + src;

      try {
         Scope scope = new Scope((Scope)null, new String[]{"#0"});
         ASTJexlScript script = this.parse((JexlInfo)null, src, scope, true, true);
         JexlNode node = script.jjtGetChild(0);
         Scope.Frame frame = script.createFrame(bean);
         Interpreter interpreter = this.createInterpreter(context, frame);
         return node.jjtAccept(interpreter, null);
      } catch (JexlException xjexl) {
         if (this.silent) {
            this.logger.warn(xjexl.getMessage(), xjexl.getCause());
            return null;
         } else {
            throw xjexl.clean();
         }
      }
   }

   public void setProperty(Object bean, String expr, Object value) {
      this.setProperty((JexlContext)null, bean, expr, value);
   }

   public void setProperty(JexlContext context, Object bean, String expr, Object value) {
      if (context == null) {
         context = EMPTY_CONTEXT;
      }

      String src = this.trimSource(expr);
      src = "#0" + (src.charAt(0) == '[' ? "" : ".") + src + "=" + "#1";

      try {
         Scope scope = new Scope((Scope)null, new String[]{"#0", "#1"});
         ASTJexlScript script = this.parse((JexlInfo)null, src, scope, true, true);
         JexlNode node = script.jjtGetChild(0);
         Scope.Frame frame = script.createFrame(bean, value);
         Interpreter interpreter = this.createInterpreter(context, frame);
         node.jjtAccept(interpreter, null);
      } catch (JexlException xjexl) {
         if (this.silent) {
            this.logger.warn(xjexl.getMessage(), xjexl.getCause());
         } else {
            throw xjexl.clean();
         }
      }
   }

   public Object invokeMethod(Object obj, String meth, Object... args) {
      JexlException xjexl = null;
      Object result = null;
      JexlInfo info = this.debug ? this.createInfo() : null;

      try {
         JexlMethod method = this.uberspect.getMethod(obj, meth, args);
         if (method == null && this.arithmetic.narrowArguments(args)) {
            method = this.uberspect.getMethod(obj, meth, args);
         }

         if (method != null) {
            result = method.invoke(obj, args);
         } else {
            xjexl = new JexlException.Method(info, meth, (Throwable)null);
         }
      } catch (JexlException xany) {
         xjexl = xany;
      } catch (Exception xany) {
         xjexl = new JexlException.Method(info, meth, xany);
      }

      if (xjexl != null) {
         if (!this.silent) {
            throw xjexl.clean();
         }

         this.logger.warn(xjexl.getMessage(), xjexl.getCause());
         result = null;
      }

      return result;
   }

   public <T> T newInstance(Class<? extends T> clazz, Object... args) {
      return (T)clazz.cast(this.doCreateInstance(clazz, args));
   }

   public Object newInstance(String clazz, Object... args) {
      return this.doCreateInstance(clazz, args);
   }

   protected Object doCreateInstance(Object clazz, Object... args) {
      JexlException xjexl = null;
      Object result = null;
      JexlInfo info = this.debug ? this.createInfo() : null;

      try {
         JexlMethod ctor = this.uberspect.getConstructor(clazz, args);
         if (ctor == null && this.arithmetic.narrowArguments(args)) {
            ctor = this.uberspect.getConstructor(clazz, args);
         }

         if (ctor != null) {
            result = ctor.invoke(clazz, args);
         } else {
            xjexl = new JexlException.Method(info, clazz.toString(), (Throwable)null);
         }
      } catch (JexlException xany) {
         xjexl = xany;
      } catch (Exception xany) {
         xjexl = new JexlException.Method(info, clazz.toString(), xany);
      }

      if (xjexl != null) {
         if (this.silent) {
            this.logger.warn(xjexl.getMessage(), xjexl.getCause());
            return null;
         } else {
            throw xjexl.clean();
         }
      } else {
         return result;
      }
   }

   protected JexlContext.ThreadLocal putThreadLocal(JexlContext.ThreadLocal tls) {
      JexlContext.ThreadLocal local = (JexlContext.ThreadLocal)CONTEXT.get();
      CONTEXT.set(tls);
      return local;
   }

   protected Set<List<String>> getVariables(ASTJexlScript script) {
      VarCollector collector = new VarCollector();
      this.getVariables(script, script, collector);
      return collector.collected();
   }

   protected void getVariables(ASTJexlScript script, JexlNode node, VarCollector collector) {
      if (node instanceof ASTIdentifier) {
         JexlNode parent = node.jjtGetParent();
         if (parent instanceof ASTMethodNode || parent instanceof ASTFunctionNode) {
            collector.collect((JexlNode)null);
            return;
         }

         ASTIdentifier identifier = (ASTIdentifier)node;
         int symbol = identifier.getSymbol();
         if (symbol >= 0 && script != null && !script.isHoistedSymbol(symbol)) {
            collector.collect((JexlNode)null);
         } else {
            collector.collect(identifier);
            collector.add(identifier.getName());
         }
      } else if (node instanceof ASTIdentifierAccess) {
         JexlNode parent = node.jjtGetParent();
         if (parent instanceof ASTMethodNode || parent instanceof ASTFunctionNode) {
            collector.collect((JexlNode)null);
            return;
         }

         if (collector.isCollecting()) {
            collector.add(((ASTIdentifierAccess)node).getName());
         }
      } else if (node instanceof ASTArrayAccess) {
         int num = node.jjtGetNumChildren();
         boolean collecting = collector.isCollecting();

         for(int i = 0; i < num; ++i) {
            JexlNode child = node.jjtGetChild(i);
            if (collecting && child.isConstant()) {
               String image = child.toString();
               collector.add(image);
            } else {
               collecting = false;
               collector.collect((JexlNode)null);
               this.getVariables(script, child, collector);
            }
         }
      } else {
         int num = node.jjtGetNumChildren();

         for(int i = 0; i < num; ++i) {
            this.getVariables(script, node.jjtGetChild(i), collector);
         }

         collector.collect((JexlNode)null);
      }

   }

   protected String[] getParameters(JexlScript script) {
      return script.getParameters();
   }

   protected String[] getLocalVariables(JexlScript script) {
      return script.getLocalVariables();
   }

   protected ASTJexlScript parse(JexlInfo info, String src, Scope scope, boolean registers, boolean expression) {
      boolean cached = src.length() < this.cacheThreshold && this.cache != null;
      ASTJexlScript script = null;
      if (cached) {
         script = this.cache.get(src);
         if (script != null) {
            Scope f = script.getScope();
            if (f == null && scope == null || f != null && f.equals(scope)) {
               return script;
            }
         }
      }

      JexlInfo ninfo = info == null && this.debug ? this.createInfo() : info;
      if (this.parsing.compareAndSet(false, true)) {
         try {
            script = this.parser.parse(ninfo, src, scope, registers, expression);
         } finally {
            this.parsing.set(false);
         }
      } else {
         Parser lparser = new Parser(new StringReader(";"));
         script = lparser.parse(ninfo, src, scope, registers, expression);
      }

      if (cached) {
         this.cache.put(src, script);
      }

      return script;
   }

   protected String trimSource(CharSequence str) {
      if (str == null) {
         return null;
      } else {
         int start = 0;
         int end = str.length();
         if (end <= 0) {
            return "";
         } else {
            while(start < end && Character.isSpaceChar(str.charAt(start))) {
               ++start;
            }

            while(end > 0 && Character.isSpaceChar(str.charAt(end - 1))) {
               --end;
            }

            return str.subSequence(start, end).toString();
         }
      }
   }

   protected TemplateEngine jxlt() {
      TemplateEngine e = this.jxlt;
      if (e == null) {
         synchronized(this) {
            if (this.jxlt == null) {
               e = new TemplateEngine(this, true, 0, '$', '#');
               this.jxlt = e;
            }
         }
      }

      return e;
   }

   private static final class UberspectHolder {
      private static final Uberspect UBERSPECT;

      static {
         UBERSPECT = new Uberspect(LogFactory.getLog(JexlEngine.class), JexlUberspect.JEXL_STRATEGY);
      }
   }

   protected static class VarCollector {
      private final Set<List<String>> refs = new LinkedHashSet();
      private List<String> ref = new ArrayList();
      private JexlNode root = null;

      public void collect(JexlNode node) {
         if (!this.ref.isEmpty()) {
            this.refs.add(this.ref);
            this.ref = new ArrayList();
         }

         this.root = node;
      }

      public boolean isCollecting() {
         return this.root instanceof ASTIdentifier;
      }

      public void add(String name) {
         this.ref.add(name);
      }

      public Set<List<String>> collected() {
         return this.refs;
      }
   }
}
