package org.apache.commons.jexl3.internal;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;

public abstract class IntegerRange implements Collection<Integer> {
   protected final int min;
   protected final int max;

   public static IntegerRange create(int from, int to) {
      return (IntegerRange)(from <= to ? new Ascending(from, to) : new Descending(to, from));
   }

   public IntegerRange(int from, int to) {
      this.min = from;
      this.max = to;
   }

   public int getMin() {
      return this.min;
   }

   public int getMax() {
      return this.max;
   }

   public int hashCode() {
      int hash = this.getClass().hashCode();
      hash = 13 * hash + this.min;
      hash = 13 * hash + this.max;
      return hash;
   }

   public boolean equals(Object obj) {
      if (obj == null) {
         return false;
      } else if (this.getClass() != obj.getClass()) {
         return false;
      } else {
         IntegerRange other = (IntegerRange)obj;
         if (this.min != other.min) {
            return false;
         } else {
            return this.max == other.max;
         }
      }
   }

   public abstract Iterator<Integer> iterator();

   public int size() {
      return this.max - this.min + 1;
   }

   public boolean isEmpty() {
      return false;
   }

   public boolean contains(Object o) {
      if (!(o instanceof Number)) {
         return false;
      } else {
         long v = (long)((Number)o).intValue();
         return (long)this.min <= v && v <= (long)this.max;
      }
   }

   public Object[] toArray() {
      int size = this.size();
      Object[] array = new Object[size];

      for(int a = 0; a < size; ++a) {
         array[a] = this.min + a;
      }

      return array;
   }

   public <T> T[] toArray(T[] array) {
      Class<?> ct = array.getClass().getComponentType();
      int length = this.size();
      T[] copy = array;
      if (!ct.isAssignableFrom(Integer.class)) {
         throw new UnsupportedOperationException();
      } else {
         if (array.length < length) {
            copy = (T[])(ct == Object.class ? (Object[])(new Object[length]) : (Object[])((Object[])Array.newInstance(ct, length)));
         }

         for(int a = 0; a < length; ++a) {
            Array.set(copy, a, this.min + a);
         }

         if (length < array.length) {
            array[length] = null;
         }

         return copy;
      }
   }

   public boolean containsAll(Collection<?> c) {
      for(Object cc : c) {
         if (!this.contains(cc)) {
            return false;
         }
      }

      return true;
   }

   public boolean add(Integer e) {
      throw new UnsupportedOperationException();
   }

   public boolean remove(Object o) {
      throw new UnsupportedOperationException();
   }

   public boolean addAll(Collection<? extends Integer> c) {
      throw new UnsupportedOperationException();
   }

   public boolean removeAll(Collection<?> c) {
      throw new UnsupportedOperationException();
   }

   public boolean retainAll(Collection<?> c) {
      throw new UnsupportedOperationException();
   }

   public void clear() {
      throw new UnsupportedOperationException();
   }

   public static class Ascending extends IntegerRange {
      protected Ascending(int from, int to) {
         super(from, to);
      }

      public Iterator<Integer> iterator() {
         return new AscIntegerIterator(this.min, this.max);
      }
   }

   public static class Descending extends IntegerRange {
      protected Descending(int from, int to) {
         super(from, to);
      }

      public Iterator<Integer> iterator() {
         return new DescIntegerIterator(this.min, this.max);
      }
   }
}
