package org.apache.commons.jexl3.internal;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import org.apache.commons.jexl3.JexlArithmetic;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlEngine;
import org.apache.commons.jexl3.JexlException;
import org.apache.commons.jexl3.JexlOperator;
import org.apache.commons.jexl3.JexlScript;
import org.apache.commons.jexl3.introspection.JexlMethod;
import org.apache.commons.jexl3.introspection.JexlPropertyGet;
import org.apache.commons.jexl3.introspection.JexlPropertySet;
import org.apache.commons.jexl3.introspection.JexlUberspect;
import org.apache.commons.jexl3.parser.ASTAddNode;
import org.apache.commons.jexl3.parser.ASTAndNode;
import org.apache.commons.jexl3.parser.ASTAnnotatedStatement;
import org.apache.commons.jexl3.parser.ASTAnnotation;
import org.apache.commons.jexl3.parser.ASTArguments;
import org.apache.commons.jexl3.parser.ASTArrayAccess;
import org.apache.commons.jexl3.parser.ASTArrayLiteral;
import org.apache.commons.jexl3.parser.ASTAssignment;
import org.apache.commons.jexl3.parser.ASTBitwiseAndNode;
import org.apache.commons.jexl3.parser.ASTBitwiseComplNode;
import org.apache.commons.jexl3.parser.ASTBitwiseOrNode;
import org.apache.commons.jexl3.parser.ASTBitwiseXorNode;
import org.apache.commons.jexl3.parser.ASTBlock;
import org.apache.commons.jexl3.parser.ASTBreak;
import org.apache.commons.jexl3.parser.ASTConstructorNode;
import org.apache.commons.jexl3.parser.ASTContinue;
import org.apache.commons.jexl3.parser.ASTDivNode;
import org.apache.commons.jexl3.parser.ASTEQNode;
import org.apache.commons.jexl3.parser.ASTERNode;
import org.apache.commons.jexl3.parser.ASTEWNode;
import org.apache.commons.jexl3.parser.ASTEmptyFunction;
import org.apache.commons.jexl3.parser.ASTEmptyMethod;
import org.apache.commons.jexl3.parser.ASTExtendedLiteral;
import org.apache.commons.jexl3.parser.ASTFalseNode;
import org.apache.commons.jexl3.parser.ASTForeachStatement;
import org.apache.commons.jexl3.parser.ASTFunctionNode;
import org.apache.commons.jexl3.parser.ASTGENode;
import org.apache.commons.jexl3.parser.ASTGTNode;
import org.apache.commons.jexl3.parser.ASTIdentifier;
import org.apache.commons.jexl3.parser.ASTIdentifierAccess;
import org.apache.commons.jexl3.parser.ASTIfStatement;
import org.apache.commons.jexl3.parser.ASTJexlLambda;
import org.apache.commons.jexl3.parser.ASTJexlScript;
import org.apache.commons.jexl3.parser.ASTJxltLiteral;
import org.apache.commons.jexl3.parser.ASTLENode;
import org.apache.commons.jexl3.parser.ASTLTNode;
import org.apache.commons.jexl3.parser.ASTMapEntry;
import org.apache.commons.jexl3.parser.ASTMapLiteral;
import org.apache.commons.jexl3.parser.ASTMethodNode;
import org.apache.commons.jexl3.parser.ASTModNode;
import org.apache.commons.jexl3.parser.ASTMulNode;
import org.apache.commons.jexl3.parser.ASTNENode;
import org.apache.commons.jexl3.parser.ASTNEWNode;
import org.apache.commons.jexl3.parser.ASTNRNode;
import org.apache.commons.jexl3.parser.ASTNSWNode;
import org.apache.commons.jexl3.parser.ASTNotNode;
import org.apache.commons.jexl3.parser.ASTNullLiteral;
import org.apache.commons.jexl3.parser.ASTNumberLiteral;
import org.apache.commons.jexl3.parser.ASTOrNode;
import org.apache.commons.jexl3.parser.ASTRangeNode;
import org.apache.commons.jexl3.parser.ASTReference;
import org.apache.commons.jexl3.parser.ASTReferenceExpression;
import org.apache.commons.jexl3.parser.ASTReturnStatement;
import org.apache.commons.jexl3.parser.ASTSWNode;
import org.apache.commons.jexl3.parser.ASTSetAddNode;
import org.apache.commons.jexl3.parser.ASTSetAndNode;
import org.apache.commons.jexl3.parser.ASTSetDivNode;
import org.apache.commons.jexl3.parser.ASTSetLiteral;
import org.apache.commons.jexl3.parser.ASTSetModNode;
import org.apache.commons.jexl3.parser.ASTSetMultNode;
import org.apache.commons.jexl3.parser.ASTSetOrNode;
import org.apache.commons.jexl3.parser.ASTSetSubNode;
import org.apache.commons.jexl3.parser.ASTSetXorNode;
import org.apache.commons.jexl3.parser.ASTSizeFunction;
import org.apache.commons.jexl3.parser.ASTSizeMethod;
import org.apache.commons.jexl3.parser.ASTStringLiteral;
import org.apache.commons.jexl3.parser.ASTSubNode;
import org.apache.commons.jexl3.parser.ASTTernaryNode;
import org.apache.commons.jexl3.parser.ASTTrueNode;
import org.apache.commons.jexl3.parser.ASTUnaryMinusNode;
import org.apache.commons.jexl3.parser.ASTVar;
import org.apache.commons.jexl3.parser.ASTWhileStatement;
import org.apache.commons.jexl3.parser.JexlNode;
import org.apache.commons.jexl3.parser.Node;

public class Interpreter extends InterpreterBase {
   protected final Operators operators;
   protected final boolean cache;
   protected final Scope.Frame frame;
   protected final JexlContext.NamespaceResolver ns;
   protected final Map<String, Object> functions;
   protected Map<String, Object> functors;

   protected Interpreter(Engine engine, JexlContext aContext, Scope.Frame eFrame) {
      super(engine, aContext);
      this.operators = new Operators(this);
      this.cache = this.jexl.cache != null;
      this.frame = eFrame;
      if (this.context instanceof JexlContext.NamespaceResolver) {
         this.ns = (JexlContext.NamespaceResolver)this.context;
      } else {
         this.ns = Engine.EMPTY_NS;
      }

      this.functions = this.jexl.functions;
      this.functors = null;
   }

   protected Interpreter(Interpreter ii, JexlArithmetic jexla) {
      super((InterpreterBase)ii, (JexlArithmetic)jexla);
      this.operators = ii.operators;
      this.cache = ii.cache;
      this.frame = ii.frame;
      this.ns = ii.ns;
      this.functions = ii.functions;
      this.functors = ii.functors;
   }

   public Object interpret(JexlNode node) {
      JexlContext.ThreadLocal local = null;

      Object var4;
      try {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         if (this.context instanceof JexlContext.ThreadLocal) {
            local = this.jexl.putThreadLocal((JexlContext.ThreadLocal)this.context);
         }

         Object var3 = node.jjtAccept(this, null);
         return var3;
      } catch (JexlException.Return xreturn) {
         var4 = xreturn.getValue();
      } catch (JexlException.Cancel xcancel) {
         this.cancelled |= Thread.interrupted();
         if (!this.isCancellable()) {
            return null;
         }

         throw xcancel.clean();
      } catch (JexlException xjexl) {
         if (!this.isSilent()) {
            throw xjexl.clean();
         }

         if (this.logger.isWarnEnabled()) {
            this.logger.warn(xjexl.getMessage(), xjexl.getCause());
         }

         return null;
      } finally {
         synchronized(this) {
            if (this.functors != null) {
               if (AUTOCLOSEABLE != null) {
                  for(Object functor : this.functors.values()) {
                     this.closeIfSupported(functor);
                  }
               }

               this.functors.clear();
               this.functors = null;
            }
         }

         if (this.context instanceof JexlContext.ThreadLocal) {
            this.jexl.putThreadLocal(local);
         }

      }

      return var4;
   }

   protected Object resolveNamespace(String prefix, JexlNode node) {
      synchronized(this) {
         if (this.functors != null) {
            Object namespace = this.functors.get(prefix);
            if (namespace != null) {
               return namespace;
            }
         }
      }

      Object namespace = this.ns.resolveNamespace(prefix);
      if (namespace == null) {
         namespace = this.functions.get(prefix);
         if (prefix != null && namespace == null) {
            throw new JexlException(node, "no such function namespace " + prefix, (Throwable)null);
         }
      }

      Object functor = null;
      if (namespace instanceof JexlContext.NamespaceFunctor) {
         functor = ((JexlContext.NamespaceFunctor)namespace).createFunctor(this.context);
      } else if (namespace instanceof Class) {
         Object[] args = new Object[]{this.context};
         JexlMethod ctor = this.uberspect.getConstructor(namespace, args);
         if (ctor != null) {
            try {
               functor = ctor.invoke(namespace, args);
            } catch (Exception xinst) {
               throw new JexlException(node, "unable to instantiate namespace " + prefix, xinst);
            }
         }
      }

      if (functor != null) {
         synchronized(this) {
            if (this.functors == null) {
               this.functors = new HashMap();
            }

            this.functors.put(prefix, functor);
            return functor;
         }
      } else {
         return namespace;
      }
   }

   protected Object visit(ASTAddNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.ADD, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.add(left, right);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "+ error", xrt);
      }
   }

   protected Object visit(ASTSubNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.SUBTRACT, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.subtract(left, right);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "- error", xrt);
      }
   }

   protected Object visit(ASTMulNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.MULTIPLY, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.multiply(left, right);
      } catch (ArithmeticException xrt) {
         JexlNode xnode = this.findNullOperand(xrt, node, left, right);
         throw new JexlException(xnode, "* error", xrt);
      }
   }

   protected Object visit(ASTDivNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.DIVIDE, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.divide(left, right);
      } catch (ArithmeticException xrt) {
         if (!this.arithmetic.isStrict()) {
            return (double)0.0F;
         } else {
            JexlNode xnode = this.findNullOperand(xrt, node, left, right);
            throw new JexlException(xnode, "/ error", xrt);
         }
      }
   }

   protected Object visit(ASTModNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.MOD, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.mod(left, right);
      } catch (ArithmeticException xrt) {
         if (!this.arithmetic.isStrict()) {
            return (double)0.0F;
         } else {
            JexlNode xnode = this.findNullOperand(xrt, node, left, right);
            throw new JexlException(xnode, "% error", xrt);
         }
      }
   }

   protected Object visit(ASTBitwiseAndNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.AND, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.and(left, right);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "& error", xrt);
      }
   }

   protected Object visit(ASTBitwiseOrNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.OR, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.or(left, right);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "| error", xrt);
      }
   }

   protected Object visit(ASTBitwiseXorNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.XOR, left, right);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.xor(left, right);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "^ error", xrt);
      }
   }

   protected Object visit(ASTEQNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.EQ, left, right);
         return result != JexlEngine.TRY_FAILED ? result : (this.arithmetic.equals(left, right) ? Boolean.TRUE : Boolean.FALSE);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "== error", xrt);
      }
   }

   protected Object visit(ASTNENode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.EQ, left, right);
         return result != JexlEngine.TRY_FAILED ? (this.arithmetic.toBoolean(result) ? Boolean.FALSE : Boolean.TRUE) : (this.arithmetic.equals(left, right) ? Boolean.FALSE : Boolean.TRUE);
      } catch (ArithmeticException xrt) {
         JexlNode xnode = this.findNullOperand(xrt, node, left, right);
         throw new JexlException(xnode, "!= error", xrt);
      }
   }

   protected Object visit(ASTGENode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.GTE, left, right);
         return result != JexlEngine.TRY_FAILED ? result : (this.arithmetic.greaterThanOrEqual(left, right) ? Boolean.TRUE : Boolean.FALSE);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, ">= error", xrt);
      }
   }

   protected Object visit(ASTGTNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.GT, left, right);
         return result != JexlEngine.TRY_FAILED ? result : (this.arithmetic.greaterThan(left, right) ? Boolean.TRUE : Boolean.FALSE);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "> error", xrt);
      }
   }

   protected Object visit(ASTLENode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.LTE, left, right);
         return result != JexlEngine.TRY_FAILED ? result : (this.arithmetic.lessThanOrEqual(left, right) ? Boolean.TRUE : Boolean.FALSE);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "<= error", xrt);
      }
   }

   protected Object visit(ASTLTNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.LT, left, right);
         return result != JexlEngine.TRY_FAILED ? result : (this.arithmetic.lessThan(left, right) ? Boolean.TRUE : Boolean.FALSE);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "< error", xrt);
      }
   }

   protected Object visit(ASTSWNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.startsWith(node, "^=", left, right) ? Boolean.TRUE : Boolean.FALSE;
   }

   protected Object visit(ASTNSWNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.startsWith(node, "^!", left, right) ? Boolean.FALSE : Boolean.TRUE;
   }

   protected Object visit(ASTEWNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.endsWith(node, "$=", left, right) ? Boolean.TRUE : Boolean.FALSE;
   }

   protected Object visit(ASTNEWNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.endsWith(node, "$!", left, right) ? Boolean.FALSE : Boolean.TRUE;
   }

   protected Object visit(ASTERNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.contains(node, "=~", right, left) ? Boolean.TRUE : Boolean.FALSE;
   }

   protected Object visit(ASTNRNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);
      return this.operators.contains(node, "!~", right, left) ? Boolean.FALSE : Boolean.TRUE;
   }

   protected Object visit(ASTRangeNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);
      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         return this.arithmetic.createRange(left, right);
      } catch (ArithmeticException xrt) {
         JexlNode xnode = this.findNullOperand(xrt, node, left, right);
         throw new JexlException(xnode, ".. error", xrt);
      }
   }

   protected Object visit(ASTUnaryMinusNode node, Object data) {
      JexlNode valNode = node.jjtGetChild(0);
      Object val = valNode.jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.NEGATE, val);
         if (result != JexlEngine.TRY_FAILED) {
            return result;
         } else {
            Object number = this.arithmetic.negate(val);
            if (valNode instanceof ASTNumberLiteral && number instanceof Number) {
               number = this.arithmetic.narrowNumber((Number)number, ((ASTNumberLiteral)valNode).getLiteralClass());
            }

            return number;
         }
      } catch (ArithmeticException xrt) {
         throw new JexlException(valNode, "- error", xrt);
      }
   }

   protected Object visit(ASTBitwiseComplNode node, Object data) {
      Object arg = node.jjtGetChild(0).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.COMPLEMENT, arg);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.complement(arg);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "~ error", xrt);
      }
   }

   protected Object visit(ASTNotNode node, Object data) {
      Object val = node.jjtGetChild(0).jjtAccept(this, data);

      try {
         Object result = this.operators.tryOverload(node, JexlOperator.NOT, val);
         return result != JexlEngine.TRY_FAILED ? result : this.arithmetic.not(val);
      } catch (ArithmeticException xrt) {
         throw new JexlException(node, "! error", xrt);
      }
   }

   protected Object visit(ASTIfStatement node, Object data) {
      int n = 0;

      try {
         Object result = null;
         Object expression = node.jjtGetChild(0).jjtAccept(this, null);
         if (this.arithmetic.toBoolean(expression)) {
            n = 1;
            result = node.jjtGetChild(n).jjtAccept(this, null);
         } else if (node.jjtGetNumChildren() == 3) {
            n = 2;
            result = node.jjtGetChild(n).jjtAccept(this, null);
         }

         return result;
      } catch (ArithmeticException xrt) {
         throw new JexlException(node.jjtGetChild(n), "if error", xrt);
      }
   }

   protected Object visit(ASTBlock node, Object data) {
      int numChildren = node.jjtGetNumChildren();
      Object result = null;

      for(int i = 0; i < numChildren; ++i) {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         result = node.jjtGetChild(i).jjtAccept(this, data);
      }

      return result;
   }

   protected Object visit(ASTReturnStatement node, Object data) {
      Object val = node.jjtGetChild(0).jjtAccept(this, data);
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         throw new JexlException.Return(node, null, val);
      }
   }

   protected Object visit(ASTContinue node, Object data) {
      throw new JexlException.Continue(node);
   }

   protected Object visit(ASTBreak node, Object data) {
      throw new JexlException.Break(node);
   }

   protected Object visit(ASTForeachStatement node, Object data) {
      Object result = null;
      ASTReference loopReference = (ASTReference)node.jjtGetChild(0);
      ASTIdentifier loopVariable = (ASTIdentifier)loopReference.jjtGetChild(0);
      int symbol = loopVariable.getSymbol();
      Object iterableValue = node.jjtGetChild(1).jjtAccept(this, data);
      if (iterableValue != null && node.jjtGetNumChildren() >= 3) {
         JexlNode statement = node.jjtGetChild(2);
         Object forEach = null;

         try {
            forEach = this.operators.tryOverload(node, JexlOperator.FOR_EACH, iterableValue);
            Iterator<?> itemsIterator = forEach instanceof Iterator ? (Iterator)forEach : this.uberspect.getIterator(iterableValue);
            if (itemsIterator != null) {
               while(itemsIterator.hasNext()) {
                  if (this.isCancelled()) {
                     throw new JexlException.Cancel(node);
                  }

                  Object value = itemsIterator.next();
                  if (symbol < 0) {
                     this.context.set(loopVariable.getName(), value);
                  } else {
                     this.frame.set(symbol, value);
                  }

                  try {
                     result = statement.jjtAccept(this, data);
                  } catch (JexlException.Break var17) {
                     break;
                  } catch (JexlException.Continue var18) {
                  }
               }
            }
         } finally {
            this.closeIfSupported(forEach);
         }
      }

      return result;
   }

   protected Object visit(ASTWhileStatement node, Object data) {
      Object result = null;
      Node expressionNode = node.jjtGetChild(0);

      while(this.arithmetic.toBoolean(expressionNode.jjtAccept(this, data))) {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         if (node.jjtGetNumChildren() > 1) {
            try {
               result = node.jjtGetChild(1).jjtAccept(this, data);
            } catch (JexlException.Break var6) {
               break;
            } catch (JexlException.Continue var7) {
            }
         }
      }

      return result;
   }

   protected Object visit(ASTAndNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);

      try {
         boolean leftValue = this.arithmetic.toBoolean(left);
         if (!leftValue) {
            return Boolean.FALSE;
         }
      } catch (ArithmeticException xrt) {
         throw new JexlException(node.jjtGetChild(0), "boolean coercion error", xrt);
      }

      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         boolean rightValue = this.arithmetic.toBoolean(right);
         if (!rightValue) {
            return Boolean.FALSE;
         }
      } catch (ArithmeticException xrt) {
         throw new JexlException(node.jjtGetChild(1), "boolean coercion error", xrt);
      }

      return Boolean.TRUE;
   }

   protected Object visit(ASTOrNode node, Object data) {
      Object left = node.jjtGetChild(0).jjtAccept(this, data);

      try {
         boolean leftValue = this.arithmetic.toBoolean(left);
         if (leftValue) {
            return Boolean.TRUE;
         }
      } catch (ArithmeticException xrt) {
         throw new JexlException(node.jjtGetChild(0), "boolean coercion error", xrt);
      }

      Object right = node.jjtGetChild(1).jjtAccept(this, data);

      try {
         boolean rightValue = this.arithmetic.toBoolean(right);
         if (rightValue) {
            return Boolean.TRUE;
         }
      } catch (ArithmeticException xrt) {
         throw new JexlException(node.jjtGetChild(1), "boolean coercion error", xrt);
      }

      return Boolean.FALSE;
   }

   protected Object visit(ASTNullLiteral node, Object data) {
      return null;
   }

   protected Object visit(ASTTrueNode node, Object data) {
      return Boolean.TRUE;
   }

   protected Object visit(ASTFalseNode node, Object data) {
      return Boolean.FALSE;
   }

   protected Object visit(ASTNumberLiteral node, Object data) {
      return data != null && node.isInteger() ? this.getAttribute(data, node.getLiteral(), node) : node.getLiteral();
   }

   protected Object visit(ASTStringLiteral node, Object data) {
      return data != null ? this.getAttribute(data, node.getLiteral(), node) : node.getLiteral();
   }

   protected Object visit(ASTArrayLiteral node, Object data) {
      int childCount = node.jjtGetNumChildren();
      JexlArithmetic.ArrayBuilder ab = this.arithmetic.arrayBuilder(childCount);
      boolean extended = false;

      for(int i = 0; i < childCount; ++i) {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         JexlNode child = node.jjtGetChild(i);
         if (child instanceof ASTExtendedLiteral) {
            extended = true;
         } else {
            Object entry = node.jjtGetChild(i).jjtAccept(this, data);
            ab.add(entry);
         }
      }

      return ab.create(extended);
   }

   protected Object visit(ASTExtendedLiteral node, Object data) {
      return node;
   }

   protected Object visit(ASTSetLiteral node, Object data) {
      int childCount = node.jjtGetNumChildren();
      JexlArithmetic.SetBuilder mb = this.arithmetic.setBuilder(childCount);

      for(int i = 0; i < childCount; ++i) {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         Object entry = node.jjtGetChild(i).jjtAccept(this, data);
         mb.add(entry);
      }

      return mb.create();
   }

   protected Object visit(ASTMapLiteral node, Object data) {
      int childCount = node.jjtGetNumChildren();
      JexlArithmetic.MapBuilder mb = this.arithmetic.mapBuilder(childCount);

      for(int i = 0; i < childCount; ++i) {
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         Object[] entry = (Object[])node.jjtGetChild(i).jjtAccept(this, data);
         mb.put(entry[0], entry[1]);
      }

      return mb.create();
   }

   protected Object visit(ASTMapEntry node, Object data) {
      Object key = node.jjtGetChild(0).jjtAccept(this, data);
      Object value = node.jjtGetChild(1).jjtAccept(this, data);
      return new Object[]{key, value};
   }

   protected Object visit(ASTTernaryNode node, Object data) {
      Object condition = node.jjtGetChild(0).jjtAccept(this, data);
      if (node.jjtGetNumChildren() == 3) {
         return condition != null && this.arithmetic.toBoolean(condition) ? node.jjtGetChild(1).jjtAccept(this, data) : node.jjtGetChild(2).jjtAccept(this, data);
      } else {
         return condition != null && this.arithmetic.toBoolean(condition) ? condition : node.jjtGetChild(1).jjtAccept(this, data);
      }
   }

   protected Object visit(ASTSizeFunction node, Object data) {
      try {
         Object val = node.jjtGetChild(0).jjtAccept(this, data);
         return this.operators.size(node, val);
      } catch (JexlException var4) {
         return 0;
      }
   }

   protected Object visit(ASTSizeMethod node, Object data) {
      Object val = node.jjtGetChild(0).jjtAccept(this, data);
      return this.operators.size(node, val);
   }

   protected Object visit(ASTEmptyFunction node, Object data) {
      try {
         Object value = node.jjtGetChild(0).jjtAccept(this, data);
         return this.operators.empty(node, value);
      } catch (JexlException var4) {
         return true;
      }
   }

   protected Object visit(ASTEmptyMethod node, Object data) {
      Object val = node.jjtGetChild(0).jjtAccept(this, data);
      return this.operators.empty(node, val);
   }

   protected Object visit(ASTJexlScript node, Object data) {
      if (node instanceof ASTJexlLambda && !((ASTJexlLambda)node).isTopLevel()) {
         return new Closure(this, (ASTJexlLambda)node);
      } else {
         int numChildren = node.jjtGetNumChildren();
         Object result = null;

         for(int i = 0; i < numChildren; ++i) {
            JexlNode child = node.jjtGetChild(i);
            result = child.jjtAccept(this, data);
            if (this.isCancelled()) {
               throw new JexlException.Cancel(child);
            }
         }

         return result;
      }
   }

   protected Object visit(ASTVar node, Object data) {
      return this.visit((ASTIdentifier)node, data);
   }

   protected Object visit(ASTReferenceExpression node, Object data) {
      return node.jjtGetChild(0).jjtAccept(this, data);
   }

   protected Object visit(ASTIdentifier node, Object data) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         String name = node.getName();
         if (data == null) {
            int symbol = node.getSymbol();
            if (symbol >= 0) {
               return this.frame.get(symbol);
            } else {
               Object value = this.context.get(name);
               return value == null && !(node.jjtGetParent() instanceof ASTReference) && !this.context.has(name) && !this.isTernaryProtected(node) ? this.unsolvableVariable(node, name, true) : value;
            }
         } else {
            return this.getAttribute(data, name, node);
         }
      }
   }

   protected Object visit(ASTArrayAccess node, Object data) {
      Object object = data;
      int numChildren = node.jjtGetNumChildren();

      for(int i = 0; i < numChildren; ++i) {
         JexlNode nindex = node.jjtGetChild(i);
         if (object == null) {
            return null;
         }

         Object index = nindex.jjtAccept(this, null);
         if (this.isCancelled()) {
            throw new JexlException.Cancel(node);
         }

         object = this.getAttribute(object, index, nindex);
      }

      return object;
   }

   protected boolean isTernaryProtected(JexlNode node) {
      for(JexlNode walk = node.jjtGetParent(); walk != null; walk = walk.jjtGetParent()) {
         if (walk instanceof ASTTernaryNode) {
            return true;
         }

         if (!(walk instanceof ASTReference) && !(walk instanceof ASTArrayAccess)) {
            break;
         }
      }

      return false;
   }

   protected boolean isLocalVariable(ASTReference node, int which) {
      return node.jjtGetNumChildren() > which && node.jjtGetChild(which) instanceof ASTIdentifier && ((ASTIdentifier)node.jjtGetChild(which)).getSymbol() >= 0;
   }

   protected Object visit(ASTIdentifierAccess node, Object data) {
      return data != null ? this.getAttribute(data, node.getIdentifier(), node) : null;
   }

   protected Object visit(ASTReference node, Object data) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         int numChildren = node.jjtGetNumChildren();
         JexlNode parent = node.jjtGetParent();
         Object object = null;
         StringBuilder ant = null;
         boolean antish = !(parent instanceof ASTReference);
         boolean pty = true;
         int v = 1;

         for(int c = 0; c < numChildren; ++c) {
            JexlNode objectNode = node.jjtGetChild(c);
            if (objectNode instanceof ASTMethodNode) {
               if (object == null) {
                  break;
               }

               antish = false;
            }

            object = objectNode.jjtAccept(this, object);
            if (this.isCancelled()) {
               throw new JexlException.Cancel(node);
            }

            if (object != null) {
               antish = false;
            } else {
               if (!antish) {
                  break;
               }

               if (ant == null) {
                  JexlNode first = node.jjtGetChild(0);
                  if (!(first instanceof ASTIdentifier)) {
                     pty = false;
                     break;
                  }

                  if (((ASTIdentifier)first).getSymbol() >= 0) {
                     break;
                  }

                  ant = new StringBuilder(((ASTIdentifier)first).getName());
               }

               while(v <= c) {
                  JexlNode child = node.jjtGetChild(v);
                  if (!(child instanceof ASTIdentifierAccess)) {
                     break;
                  }

                  ant.append('.');
                  ant.append(((ASTIdentifierAccess)objectNode).getName());
                  ++v;
               }

               object = this.context.get(ant.toString());
            }
         }

         if (object == null && !this.isTernaryProtected(node)) {
            if (antish && ant != null) {
               boolean undefined = !this.context.has(ant.toString()) && !this.isLocalVariable(node, 0);
               return this.unsolvableVariable(node, ant.toString(), undefined);
            }

            if (!pty) {
               return this.unsolvableProperty(node, "<null>.<?>", (Throwable)null);
            }
         }

         return object;
      }
   }

   protected Object visit(ASTAssignment node, Object data) {
      return this.executeAssign(node, (JexlOperator)null, data);
   }

   protected Object visit(ASTSetAddNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_ADD, data);
   }

   protected Object visit(ASTSetSubNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_SUBTRACT, data);
   }

   protected Object visit(ASTSetMultNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_MULTIPLY, data);
   }

   protected Object visit(ASTSetDivNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_DIVIDE, data);
   }

   protected Object visit(ASTSetModNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_MOD, data);
   }

   protected Object visit(ASTSetAndNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_AND, data);
   }

   protected Object visit(ASTSetOrNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_OR, data);
   }

   protected Object visit(ASTSetXorNode node, Object data) {
      return this.executeAssign(node, JexlOperator.SELF_XOR, data);
   }

   protected Object executeAssign(JexlNode node, JexlOperator assignop, Object data) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         JexlNode left = node.jjtGetChild(0);
         Object right = node.jjtGetChild(1).jjtAccept(this, data);
         Object object = null;
         int symbol = -1;
         boolean antish = true;
         int last = left.jjtGetNumChildren() - 1;
         if (left instanceof ASTIdentifier) {
            ASTIdentifier var = (ASTIdentifier)left;
            symbol = var.getSymbol();
            if (symbol >= 0) {
               if (last < 0) {
                  if (assignop != null) {
                     Object self = this.frame.get(symbol);
                     right = this.operators.tryAssignOverload(node, assignop, self, right);
                     if (right == JexlOperator.ASSIGN) {
                        return self;
                     }
                  }

                  this.frame.set(symbol, right);
                  if (right instanceof Closure) {
                     ((Closure)right).setHoisted(symbol, right);
                  }

                  return right;
               }

               object = this.frame.get(symbol);
               antish = false;
            } else {
               if (last < 0) {
                  if (assignop != null) {
                     Object self = this.context.get(var.getName());
                     right = this.operators.tryAssignOverload(node, assignop, self, right);
                     if (right == JexlOperator.ASSIGN) {
                        return self;
                     }
                  }

                  try {
                     this.context.set(var.getName(), right);
                     return right;
                  } catch (UnsupportedOperationException xsupport) {
                     throw new JexlException(node, "context is readonly", xsupport);
                  }
               }

               object = this.context.get(var.getName());
               if (object != null) {
                  antish = false;
               }
            }
         } else if (!(left instanceof ASTReference)) {
            throw new JexlException(left, "illegal assignment form 0");
         }

         JexlNode objectNode = null;
         StringBuilder ant = null;
         int v = 1;

         for(int c = symbol >= 0 ? 1 : 0; c < last; ++c) {
            objectNode = left.jjtGetChild(c);
            object = objectNode.jjtAccept(this, object);
            if (object != null) {
               antish = false;
            } else {
               if (!antish) {
                  throw new JexlException(objectNode, "illegal assignment form");
               }

               if (ant == null) {
                  JexlNode first = left.jjtGetChild(0);
                  if (!(first instanceof ASTIdentifier) || ((ASTIdentifier)first).getSymbol() >= 0) {
                     break;
                  }

                  ant = new StringBuilder(((ASTIdentifier)first).getName());
               }

               while(v <= c) {
                  JexlNode child = left.jjtGetChild(v);
                  if (!(child instanceof ASTIdentifierAccess)) {
                     break;
                  }

                  ant.append('.');
                  ant.append(((ASTIdentifierAccess)objectNode).getName());
                  ++v;
               }

               object = this.context.get(ant.toString());
            }
         }

         Object property = null;
         JexlNode propertyNode = left.jjtGetChild(last);
         if (propertyNode instanceof ASTIdentifierAccess) {
            property = ((ASTIdentifierAccess)propertyNode).getIdentifier();
            if (ant != null && object == null) {
               if (last > 0) {
                  ant.append('.');
               }

               ant.append(String.valueOf(property));
               if (assignop != null) {
                  Object self = this.context.get(ant.toString());
                  right = this.operators.tryAssignOverload(node, assignop, self, right);
                  if (right == JexlOperator.ASSIGN) {
                     return self;
                  }
               }

               try {
                  this.context.set(ant.toString(), right);
                  return right;
               } catch (UnsupportedOperationException xsupport) {
                  throw new JexlException(node, "context is readonly", xsupport);
               }
            }
         } else {
            if (!(propertyNode instanceof ASTArrayAccess)) {
               throw new JexlException(objectNode, "illegal assignment form");
            }

            int numChildren = propertyNode.jjtGetNumChildren() - 1;

            for(int i = 0; i < numChildren; ++i) {
               JexlNode nindex = propertyNode.jjtGetChild(i);
               Object index = nindex.jjtAccept(this, null);
               object = this.getAttribute(object, index, nindex);
            }

            propertyNode = propertyNode.jjtGetChild(numChildren);
            property = propertyNode.jjtAccept(this, null);
         }

         if (property == null) {
            return this.unsolvableProperty(propertyNode, "<?>.<null>", (Throwable)null);
         } else if (object == null) {
            return this.unsolvableProperty(objectNode, "<null>.<?>", (Throwable)null);
         } else {
            if (assignop != null) {
               Object self = this.getAttribute(object, property, propertyNode);
               right = this.operators.tryAssignOverload(node, assignop, self, right);
               if (right == JexlOperator.ASSIGN) {
                  return self;
               }
            }

            this.setAttribute(object, property, right, propertyNode);
            return right;
         }
      }
   }

   protected Object[] visit(ASTArguments node, Object data) {
      int argc = node.jjtGetNumChildren();
      Object[] argv = new Object[argc];

      for(int i = 0; i < argc; ++i) {
         argv[i] = node.jjtGetChild(i).jjtAccept(this, data);
      }

      return argv;
   }

   protected Object visit(ASTMethodNode node, Object data) {
      JexlNode methodNode = node.jjtGetChild(0);
      Object object = null;
      JexlNode objectNode = null;
      Object method;
      if (methodNode instanceof ASTIdentifierAccess) {
         method = methodNode;
         object = data;
         if (data == null) {
            return this.unsolvableMethod(objectNode, "<null>.<?>(...)");
         }
      } else {
         method = methodNode.jjtAccept(this, data);
      }

      Object result = method;

      for(int a = 1; a < node.jjtGetNumChildren(); ++a) {
         if (result == null) {
            return this.unsolvableMethod(methodNode, "<?>.<null>(...)");
         }

         ASTArguments argNode = (ASTArguments)node.jjtGetChild(a);
         result = this.call(node, object, result, argNode);
         object = result;
      }

      return result;
   }

   protected Object visit(ASTFunctionNode node, Object data) {
      int argc = node.jjtGetNumChildren();
      if (argc == 2) {
         ASTIdentifier functionNode = (ASTIdentifier)node.jjtGetChild(0);
         ASTArguments argNode = (ASTArguments)node.jjtGetChild(1);
         return this.call(node, this.context, functionNode, argNode);
      } else {
         String prefix = ((ASTIdentifier)node.jjtGetChild(0)).getName();
         Object namespace = this.resolveNamespace(prefix, node);
         ASTIdentifier functionNode = (ASTIdentifier)node.jjtGetChild(1);
         ASTArguments argNode = (ASTArguments)node.jjtGetChild(2);
         return this.call(node, namespace, functionNode, argNode);
      }
   }

   private Object[] functionArguments(Object target, boolean narrow, Object[] args) {
      if (target != null && target != this.context) {
         Object[] nargv = new Object[args.length + 1];
         if (narrow) {
            nargv[0] = this.functionArgument(true, target);

            for(int a = 1; a <= args.length; ++a) {
               nargv[a] = this.functionArgument(true, args[a - 1]);
            }
         } else {
            nargv[0] = target;
            System.arraycopy(args, 0, nargv, 1, args.length);
         }

         return nargv;
      } else {
         if (narrow) {
            this.arithmetic.narrowArguments(args);
         }

         return args;
      }
   }

   private Object functionArgument(boolean narrow, Object arg) {
      return narrow && arg instanceof Number ? this.arithmetic.narrow((Number)arg) : arg;
   }

   protected Object call(JexlNode node, Object target, Object functor, ASTArguments argNode) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         Object[] argv = this.visit((ASTArguments)argNode, null);
         int symbol;
         String methodName;
         if (functor instanceof ASTIdentifier) {
            ASTIdentifier methodIdentifier = (ASTIdentifier)functor;
            symbol = methodIdentifier.getSymbol();
            methodName = methodIdentifier.getName();
            functor = null;
         } else if (functor instanceof ASTIdentifierAccess) {
            methodName = ((ASTIdentifierAccess)functor).getName();
            symbol = -1;
            functor = null;
         } else {
            if (functor == null) {
               return this.unsolvableMethod(node, "?");
            }

            symbol = -2;
            methodName = null;
         }

         Object caller = target;

         try {
            boolean cacheable = this.cache;
            if (methodName != null) {
               if (target == this.context) {
                  boolean isavar = true;
                  if (symbol >= 0) {
                     functor = this.frame.get(symbol);
                  } else if (this.context.has(methodName)) {
                     functor = this.context.get(methodName);
                  } else {
                     isavar = false;
                  }

                  if (isavar) {
                     if (functor == null) {
                        return this.unsolvableMethod(node, methodName);
                     }

                     cacheable = false;
                  }
               }

               if (cacheable) {
                  Object cached = node.jjtGetValue();
                  if (cached instanceof Funcall) {
                     Object eval = ((Funcall)cached).tryInvoke(this, methodName, target, argv);
                     if (JexlEngine.TRY_FAILED != eval) {
                        return eval;
                     }
                  }
               }
            }

            boolean narrow = false;
            JexlMethod vm = null;
            Funcall funcall = null;

            while(true) {
               if (functor == null) {
                  vm = this.uberspect.getMethod(target, methodName, argv);
                  if (vm != null) {
                     if (cacheable && vm.isCacheable()) {
                        funcall = new Funcall(vm, narrow);
                     }
                     break;
                  }

                  if (target == this.context) {
                     Object namespace = this.resolveNamespace(null, node);
                     if (namespace == this.context) {
                        break;
                     }

                     if (namespace != null) {
                        target = namespace;
                        caller = null;
                        continue;
                     }
                  } else if (!narrow) {
                     JexlPropertyGet get = this.uberspect.getPropertyGet(target, methodName);
                     if (get != null) {
                        functor = get.tryInvoke(target, methodName);
                     }
                  }
               }

               if (functor != null) {
                  if (functor instanceof JexlScript) {
                     return ((JexlScript)functor).execute(this.context, argv);
                  }

                  if (functor instanceof JexlMethod) {
                     return ((JexlMethod)functor).invoke(target, argv);
                  }

                  vm = this.uberspect.getMethod(functor, "call", argv);
                  if (vm != null) {
                     return vm.invoke(functor, argv);
                  }
                  break;
               }

               Object[] nargv = this.functionArguments(caller, narrow, argv);
               vm = this.uberspect.getMethod(this.context, methodName, nargv);
               if (vm != null) {
                  argv = nargv;
                  target = this.context;
                  if (cacheable && vm.isCacheable()) {
                     funcall = new ContextFuncall(vm, narrow);
                  }
                  break;
               }

               vm = this.uberspect.getMethod(this.arithmetic, methodName, nargv);
               if (vm != null) {
                  argv = nargv;
                  target = this.arithmetic;
                  if (cacheable && vm.isCacheable()) {
                     funcall = new ArithmeticFuncall(vm, narrow);
                  }
                  break;
               }

               if (!this.arithmetic.narrowArguments(argv)) {
                  break;
               }

               narrow = true;
            }

            if (vm != null) {
               Object eval = vm.invoke(target, argv);
               if (funcall != null) {
                  node.jjtSetValue(funcall);
               }

               return eval;
            } else {
               return this.unsolvableMethod(node, methodName);
            }
         } catch (JexlException xthru) {
            throw xthru;
         } catch (Exception xany) {
            throw this.invocationException(node, methodName, xany);
         }
      }
   }

   protected Object visit(ASTConstructorNode node, Object data) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         Object cobject = node.jjtGetChild(0).jjtAccept(this, data);
         int argc = node.jjtGetNumChildren() - 1;
         Object[] argv = new Object[argc];

         for(int i = 0; i < argc; ++i) {
            argv[i] = node.jjtGetChild(i + 1).jjtAccept(this, data);
         }

         try {
            if (this.cache) {
               Object cached = node.jjtGetValue();
               if (cached instanceof JexlMethod) {
                  JexlMethod mctor = (JexlMethod)cached;
                  Object eval = mctor.tryInvoke(null, cobject, argv);
                  if (!mctor.tryFailed(eval)) {
                     return eval;
                  }
               }
            }

            JexlMethod ctor = this.uberspect.getConstructor(cobject, argv);
            if (ctor == null) {
               if (this.arithmetic.narrowArguments(argv)) {
                  ctor = this.uberspect.getConstructor(cobject, argv);
               }

               if (ctor == null) {
                  String dbgStr = cobject != null ? cobject.toString() : null;
                  return this.unsolvableMethod(node, dbgStr);
               }
            }

            Object instance = ctor.invoke(cobject, argv);
            if (this.cache && ctor.isCacheable()) {
               node.jjtSetValue(ctor);
            }

            return instance;
         } catch (JexlException xthru) {
            throw xthru;
         } catch (Exception xany) {
            String dbgStr = cobject != null ? cobject.toString() : null;
            throw this.invocationException(node, dbgStr, xany);
         }
      }
   }

   public Object getAttribute(Object object, Object attribute) {
      return this.getAttribute(object, attribute, (JexlNode)null);
   }

   protected Object getAttribute(Object object, Object attribute, JexlNode node) {
      if (object == null) {
         throw new JexlException(node, "object is null");
      } else if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         JexlOperator operator = node != null && node.jjtGetParent() instanceof ASTArrayAccess ? JexlOperator.ARRAY_GET : JexlOperator.PROPERTY_GET;
         Object result = this.operators.tryOverload(node, operator, object, attribute);
         if (result != JexlEngine.TRY_FAILED) {
            return result;
         } else {
            Exception xcause = null;

            try {
               if (node != null && this.cache) {
                  Object cached = node.jjtGetValue();
                  if (cached instanceof JexlPropertyGet) {
                     JexlPropertyGet vg = (JexlPropertyGet)cached;
                     Object value = vg.tryInvoke(object, attribute);
                     if (!vg.tryFailed(value)) {
                        return value;
                     }
                  }
               }

               List<JexlUberspect.PropertyResolver> resolvers = this.uberspect.getResolvers(operator, object);
               JexlPropertyGet vg = this.uberspect.getPropertyGet(resolvers, object, attribute);
               if (vg != null) {
                  Object value = vg.invoke(object);
                  if (node != null && this.cache && vg.isCacheable()) {
                     node.jjtSetValue(vg);
                  }

                  return value;
               }
            } catch (Exception xany) {
               xcause = xany;
            }

            if (node != null) {
               String attrStr = attribute != null ? attribute.toString() : null;
               return this.unsolvableProperty(node, attrStr, xcause);
            } else {
               String error = "unable to get object property, class: " + object.getClass().getName() + ", property: " + attribute;
               throw new UnsupportedOperationException(error, xcause);
            }
         }
      }
   }

   public void setAttribute(Object object, Object attribute, Object value) {
      this.setAttribute(object, attribute, value, (JexlNode)null);
   }

   protected void setAttribute(Object object, Object attribute, Object value, JexlNode node) {
      if (this.isCancelled()) {
         throw new JexlException.Cancel(node);
      } else {
         JexlOperator operator = node != null && node.jjtGetParent() instanceof ASTArrayAccess ? JexlOperator.ARRAY_SET : JexlOperator.PROPERTY_SET;
         Object result = this.operators.tryOverload(node, operator, object, attribute, value);
         if (result == JexlEngine.TRY_FAILED) {
            Exception xcause = null;

            try {
               if (node != null && this.cache) {
                  Object cached = node.jjtGetValue();
                  if (cached instanceof JexlPropertySet) {
                     JexlPropertySet setter = (JexlPropertySet)cached;
                     Object eval = setter.tryInvoke(object, attribute, value);
                     if (!setter.tryFailed(eval)) {
                        return;
                     }
                  }
               }

               List<JexlUberspect.PropertyResolver> resolvers = this.uberspect.getResolvers(operator, object);
               JexlPropertySet vs = this.uberspect.getPropertySet(resolvers, object, attribute, value);
               if (vs == null) {
                  Object[] narrow = new Object[]{value};
                  if (this.arithmetic.narrowArguments(narrow)) {
                     vs = this.uberspect.getPropertySet(resolvers, object, attribute, narrow[0]);
                  }
               }

               if (vs != null) {
                  vs.invoke(object, value);
                  if (node != null && this.cache && vs.isCacheable()) {
                     node.jjtSetValue(vs);
                  }

                  return;
               }
            } catch (Exception xany) {
               xcause = xany;
            }

            if (node != null) {
               String attrStr = attribute != null ? attribute.toString() : null;
               this.unsolvableProperty(node, attrStr, xcause);
            } else {
               String error = "unable to set object property, class: " + object.getClass().getName() + ", property: " + attribute + ", argument: " + value.getClass().getSimpleName();
               throw new UnsupportedOperationException(error, xcause);
            }
         }
      }
   }

   protected Object visit(ASTJxltLiteral node, Object data) {
      TemplateEngine.TemplateExpression tp = (TemplateEngine.TemplateExpression)node.jjtGetValue();
      if (tp == null) {
         TemplateEngine jxlt = this.jexl.jxlt();
         tp = jxlt.parseExpression(node.jexlInfo(), node.getLiteral(), this.frame != null ? this.frame.getScope() : null);
         node.jjtSetValue(tp);
      }

      return tp != null ? tp.evaluate(this.frame, this.context) : null;
   }

   protected Object visit(ASTAnnotation node, Object data) {
      throw new UnsupportedOperationException(ASTAnnotation.class.getName() + ": Not supported.");
   }

   protected Object visit(ASTAnnotatedStatement node, Object data) {
      return this.processAnnotation(node, 0, data);
   }

   protected Object processAnnotation(final ASTAnnotatedStatement stmt, final int index, final Object data) {
      int last = stmt.jjtGetNumChildren() - 1;
      if (index == last) {
         JexlNode block = stmt.jjtGetChild(last);
         JexlArithmetic jexla = this.arithmetic.options(this.context);
         if (jexla != this.arithmetic) {
            if (!this.arithmetic.getClass().equals(jexla.getClass())) {
               this.logger.warn("expected arithmetic to be " + this.arithmetic.getClass().getSimpleName() + ", got " + jexla.getClass().getSimpleName());
            }

            Interpreter ii = new Interpreter(this, jexla);
            Object r = block.jjtAccept(ii, data);
            if (ii.isCancelled()) {
               this.cancel();
            }

            return r;
         } else {
            return block.jjtAccept(this, data);
         }
      } else {
         final boolean[] processed = new boolean[]{false};
         Callable<Object> jstmt = new Callable<Object>() {
            public Object call() throws Exception {
               processed[0] = true;
               return Interpreter.this.processAnnotation(stmt, index + 1, data);
            }
         };
         ASTAnnotation anode = (ASTAnnotation)stmt.jjtGetChild(index);
         String aname = anode.getName();
         Object[] argv = anode.jjtGetNumChildren() > 0 ? this.visit((ASTArguments)((ASTArguments)anode.jjtGetChild(0)), null) : null;

         try {
            Object result = this.processAnnotation(aname, argv, jstmt);
            return !processed[0] ? this.annotationError(anode, aname, (Throwable)null) : result;
         } catch (JexlException xjexl) {
            throw xjexl;
         } catch (Exception xany) {
            return this.annotationError(anode, aname, xany);
         }
      }
   }

   protected Object processAnnotation(String annotation, Object[] args, Callable<Object> stmt) throws Exception {
      return this.context instanceof JexlContext.AnnotationProcessor ? ((JexlContext.AnnotationProcessor)this.context).processAnnotation(annotation, args, stmt) : stmt.call();
   }

   private static class Funcall {
      protected final boolean narrow;
      protected final JexlMethod me;

      protected Funcall(JexlMethod jme, boolean flag) {
         this.me = jme;
         this.narrow = flag;
      }

      protected Object tryInvoke(Interpreter ii, String name, Object target, Object[] args) {
         return this.me.tryInvoke(name, target, ii.functionArguments(null, this.narrow, args));
      }
   }

   private static class ArithmeticFuncall extends Funcall {
      protected ArithmeticFuncall(JexlMethod jme, boolean flag) {
         super(jme, flag);
      }

      protected Object tryInvoke(Interpreter ii, String name, Object target, Object[] args) {
         return this.me.tryInvoke(name, ii.arithmetic, ii.functionArguments(target, this.narrow, args));
      }
   }

   private static class ContextFuncall extends Funcall {
      protected ContextFuncall(JexlMethod jme, boolean flag) {
         super(jme, flag);
      }

      protected Object tryInvoke(Interpreter ii, String name, Object target, Object[] args) {
         return this.me.tryInvoke(name, ii.context, ii.functionArguments(target, this.narrow, args));
      }
   }
}
