package org.apache.commons.jexl3.internal;

import org.apache.commons.jexl3.JexlArithmetic;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlEngine;
import org.apache.commons.jexl3.JexlException;
import org.apache.commons.jexl3.JexlOperator;
import org.apache.commons.jexl3.introspection.JexlMethod;
import org.apache.commons.jexl3.introspection.JexlUberspect;
import org.apache.commons.jexl3.parser.JexlNode;
import org.apache.commons.jexl3.parser.ParserVisitor;
import org.apache.commons.logging.Log;

public abstract class InterpreterBase extends ParserVisitor {
   protected final Engine jexl;
   protected final Log logger;
   protected final JexlUberspect uberspect;
   protected final JexlArithmetic arithmetic;
   protected final JexlContext context;
   protected volatile boolean cancelled = false;
   protected static final Object[] EMPTY_PARAMS = new Object[0];
   protected static final Class<?> AUTOCLOSEABLE;

   protected InterpreterBase(Engine engine, JexlContext aContext) {
      this.jexl = engine;
      this.logger = this.jexl.logger;
      this.uberspect = this.jexl.uberspect;
      this.context = aContext != null ? aContext : Engine.EMPTY_CONTEXT;
      JexlArithmetic jexla = this.jexl.arithmetic;
      this.arithmetic = jexla.options(this.context);
      if (this.arithmetic != jexla && !this.arithmetic.getClass().equals(jexla.getClass())) {
         this.logger.warn("expected arithmetic to be " + jexla.getClass().getSimpleName() + ", got " + this.arithmetic.getClass().getSimpleName());
      }

   }

   protected InterpreterBase(InterpreterBase ii, JexlArithmetic jexla) {
      this.jexl = ii.jexl;
      this.logger = ii.logger;
      this.uberspect = ii.uberspect;
      this.context = ii.context;
      this.arithmetic = ii.arithmetic;
   }

   protected void closeIfSupported(Object closeable) {
      if (closeable != null) {
         JexlMethod mclose = this.uberspect.getMethod(closeable, "close", EMPTY_PARAMS);
         if (mclose != null) {
            try {
               mclose.invoke(closeable, EMPTY_PARAMS);
            } catch (Exception xignore) {
               this.logger.warn(xignore);
            }
         }
      }

   }

   protected boolean isStrictEngine() {
      if (this.context instanceof JexlEngine.Options) {
         JexlEngine.Options opts = (JexlEngine.Options)this.context;
         Boolean strict = opts.isStrict();
         if (strict != null) {
            return strict;
         }
      }

      return this.jexl.isStrict();
   }

   protected boolean isSilent() {
      if (this.context instanceof JexlEngine.Options) {
         JexlEngine.Options opts = (JexlEngine.Options)this.context;
         Boolean silent = opts.isSilent();
         if (silent != null) {
            return silent;
         }
      }

      return this.jexl.isSilent();
   }

   protected boolean isCancellable() {
      if (this.context instanceof JexlEngine.Options) {
         JexlEngine.Options opts = (JexlEngine.Options)this.context;
         Boolean ocancellable = opts.isCancellable();
         if (ocancellable != null) {
            return ocancellable;
         }
      }

      return this.jexl.isCancellable();
   }

   protected JexlNode findNullOperand(RuntimeException xrt, JexlNode node, Object left, Object right) {
      if (xrt instanceof JexlArithmetic.NullOperand) {
         if (left == null) {
            return node.jjtGetChild(0);
         }

         if (right == null) {
            return node.jjtGetChild(1);
         }
      }

      return node;
   }

   protected Object unsolvableVariable(JexlNode node, String var, boolean undef) {
      if (!this.isStrictEngine() || !undef && !this.arithmetic.isStrict()) {
         if (this.logger.isDebugEnabled()) {
            this.logger.debug(JexlException.variableError(node, var, undef));
         }

         return null;
      } else {
         throw new JexlException.Variable(node, var, undef);
      }
   }

   protected Object unsolvableMethod(JexlNode node, String method) {
      if (this.isStrictEngine()) {
         throw new JexlException.Method(node, method);
      } else {
         if (this.logger.isDebugEnabled()) {
            this.logger.debug(JexlException.methodError(node, method));
         }

         return null;
      }
   }

   protected Object unsolvableProperty(JexlNode node, String var, Throwable cause) {
      if (this.isStrictEngine()) {
         throw new JexlException.Property(node, var, cause);
      } else {
         if (this.logger.isDebugEnabled()) {
            this.logger.debug(JexlException.propertyError(node, var), cause);
         }

         return null;
      }
   }

   protected Object operatorError(JexlNode node, JexlOperator operator, Throwable cause) {
      if (this.isStrictEngine()) {
         throw new JexlException.Operator(node, operator.getOperatorSymbol(), cause);
      } else {
         if (this.logger.isDebugEnabled()) {
            this.logger.debug(JexlException.operatorError(node, operator.getOperatorSymbol()), cause);
         }

         return null;
      }
   }

   protected Object annotationError(JexlNode node, String annotation, Throwable cause) {
      if (this.isStrictEngine()) {
         throw new JexlException.Annotation(node, annotation, cause);
      } else {
         if (this.logger.isDebugEnabled()) {
            this.logger.debug(JexlException.annotationError(node, annotation), cause);
         }

         return null;
      }
   }

   protected JexlException invocationException(JexlNode node, String methodName, Exception xany) {
      Throwable cause = xany.getCause();
      if (cause instanceof JexlException) {
         return (JexlException)cause;
      } else if (cause instanceof InterruptedException) {
         this.cancelled = true;
         return new JexlException.Cancel(node);
      } else {
         return new JexlException(node, methodName, xany);
      }
   }

   protected synchronized boolean cancel() {
      if (this.cancelled) {
         return false;
      } else {
         this.cancelled = true;
         return true;
      }
   }

   protected synchronized boolean isCancelled() {
      if (!this.cancelled) {
         this.cancelled = Thread.currentThread().isInterrupted();
      }

      return this.cancelled;
   }

   static {
      Class<?> c;
      try {
         c = Class.forName("java.lang.AutoCloseable");
      } catch (ClassNotFoundException var2) {
         c = null;
      }

      AUTOCLOSEABLE = c;
   }
}
