package org.apache.commons.jexl3.internal;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;

public abstract class LongRange implements Collection<Long> {
   protected final long min;
   protected final long max;

   public static LongRange create(long from, long to) {
      return (LongRange)(from <= to ? new Ascending(from, to) : new Descending(to, from));
   }

   protected LongRange(long from, long to) {
      this.min = from;
      this.max = to;
   }

   public long getMin() {
      return this.min;
   }

   public long getMax() {
      return this.max;
   }

   public int hashCode() {
      int hash = this.getClass().hashCode();
      hash = 13 * hash + (int)(this.min ^ this.min >>> 32);
      hash = 13 * hash + (int)(this.max ^ this.max >>> 32);
      return hash;
   }

   public boolean equals(Object obj) {
      if (obj == null) {
         return false;
      } else if (this.getClass() != obj.getClass()) {
         return false;
      } else {
         LongRange other = (LongRange)obj;
         if (this.min != other.min) {
            return false;
         } else {
            return this.max == other.max;
         }
      }
   }

   public abstract Iterator<Long> iterator();

   public int size() {
      return (int)(this.max - this.min + 1L);
   }

   public boolean isEmpty() {
      return false;
   }

   public boolean contains(Object o) {
      if (!(o instanceof Number)) {
         return false;
      } else {
         long v = ((Number)o).longValue();
         return this.min <= v && v <= this.max;
      }
   }

   public Object[] toArray() {
      int size = this.size();
      Object[] array = new Object[size];

      for(int a = 0; a < size; ++a) {
         array[a] = this.min + (long)a;
      }

      return array;
   }

   public <T> T[] toArray(T[] array) {
      Class<?> ct = array.getClass().getComponentType();
      int length = this.size();
      T[] copy = array;
      if (!ct.isAssignableFrom(Long.class)) {
         throw new UnsupportedOperationException();
      } else {
         if (array.length < length) {
            copy = (T[])(ct == Object.class ? (Object[])(new Object[length]) : (Object[])((Object[])Array.newInstance(ct, length)));
         }

         for(int a = 0; a < length; ++a) {
            Array.set(copy, a, this.min + (long)a);
         }

         if (length < array.length) {
            array[length] = null;
         }

         return copy;
      }
   }

   public boolean containsAll(Collection<?> c) {
      for(Object cc : c) {
         if (!this.contains(cc)) {
            return false;
         }
      }

      return true;
   }

   public boolean add(Long e) {
      throw new UnsupportedOperationException();
   }

   public boolean remove(Object o) {
      throw new UnsupportedOperationException();
   }

   public boolean addAll(Collection<? extends Long> c) {
      throw new UnsupportedOperationException();
   }

   public boolean removeAll(Collection<?> c) {
      throw new UnsupportedOperationException();
   }

   public boolean retainAll(Collection<?> c) {
      throw new UnsupportedOperationException();
   }

   public void clear() {
      throw new UnsupportedOperationException();
   }

   public static class Ascending extends LongRange {
      protected Ascending(long from, long to) {
         super(from, to);
      }

      public Iterator<Long> iterator() {
         return new AscLongIterator(this.min, this.max);
      }
   }

   public static class Descending extends LongRange {
      protected Descending(long from, long to) {
         super(from, to);
      }

      public Iterator<Long> iterator() {
         return new DescLongIterator(this.min, this.max);
      }
   }
}
