package org.apache.commons.jexl3.internal;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;

public final class Scope {
   private Scope parent = null;
   private int parms;
   private int vars;
   private Map<String, Integer> namedVariables = null;
   private Map<Integer, Integer> hoistedVariables = null;

   public Scope(Scope scope, String... parameters) {
      if (parameters != null) {
         this.parms = parameters.length;
         this.namedVariables = new LinkedHashMap();

         for(int p = 0; p < this.parms; ++p) {
            this.namedVariables.put(parameters[p], p);
         }
      } else {
         this.parms = 0;
      }

      this.vars = 0;
      this.parent = scope;
   }

   public int hashCode() {
      return this.namedVariables == null ? 0 : this.parms ^ this.namedVariables.hashCode();
   }

   public boolean equals(Object o) {
      if (this == o) {
         return true;
      } else if (!(o instanceof Scope)) {
         return false;
      } else {
         Scope scope = (Scope)o;
         if (this.parms != scope.parms) {
            return false;
         } else if (this.namedVariables == null) {
            return scope.namedVariables == null;
         } else {
            return this.namedVariables.equals(scope.namedVariables);
         }
      }
   }

   public Integer getSymbol(String name) {
      return this.getSymbol(name, true);
   }

   private Integer getSymbol(String name, boolean hoist) {
      Integer register = this.namedVariables != null ? (Integer)this.namedVariables.get(name) : null;
      if (register == null && hoist && this.parent != null) {
         Integer pr = this.parent.getSymbol(name, false);
         if (pr != null) {
            if (this.hoistedVariables == null) {
               this.hoistedVariables = new LinkedHashMap();
            }

            if (this.namedVariables == null) {
               this.namedVariables = new LinkedHashMap();
            }

            register = this.namedVariables.size();
            this.namedVariables.put(name, register);
            this.hoistedVariables.put(register, pr);
         }
      }

      return register;
   }

   public boolean isHoistedSymbol(int symbol) {
      return this.hoistedVariables != null && this.hoistedVariables.containsKey(symbol);
   }

   public void declareParameter(String name) {
      if (this.namedVariables == null) {
         this.namedVariables = new LinkedHashMap();
      } else if (this.vars > 0) {
         throw new IllegalStateException("cant declare parameters after variables");
      }

      Integer register = (Integer)this.namedVariables.get(name);
      if (register == null) {
         register = this.namedVariables.size();
         this.namedVariables.put(name, register);
         ++this.parms;
      }

   }

   public Integer declareVariable(String name) {
      if (this.namedVariables == null) {
         this.namedVariables = new LinkedHashMap();
      }

      Integer register = (Integer)this.namedVariables.get(name);
      if (register == null) {
         register = this.namedVariables.size();
         this.namedVariables.put(name, register);
         ++this.vars;
      }

      return register;
   }

   public Frame createFrame(Frame frame) {
      if (this.namedVariables == null) {
         return null;
      } else {
         Object[] arguments = new Object[this.namedVariables.size()];
         if (frame != null && this.hoistedVariables != null && this.parent != null) {
            for(Map.Entry<Integer, Integer> hoist : this.hoistedVariables.entrySet()) {
               Integer target = (Integer)hoist.getKey();
               Integer source = (Integer)hoist.getValue();
               Object arg = frame.get(source);
               arguments[target] = arg;
            }
         }

         return new Frame(this, arguments, 0);
      }
   }

   public Integer getHoisted(int symbol) {
      if (this.hoistedVariables != null) {
         for(Map.Entry<Integer, Integer> hoist : this.hoistedVariables.entrySet()) {
            Integer source = (Integer)hoist.getValue();
            if (source == symbol) {
               return (Integer)hoist.getKey();
            }
         }
      }

      return null;
   }

   public int getArgCount() {
      return this.parms;
   }

   public String[] getSymbols() {
      return this.namedVariables != null ? (String[])this.namedVariables.keySet().toArray(new String[0]) : new String[0];
   }

   public String[] getParameters() {
      if (this.namedVariables != null && this.parms > 0) {
         String[] pa = new String[this.parms];
         int p = 0;

         for(Map.Entry<String, Integer> entry : this.namedVariables.entrySet()) {
            if ((Integer)entry.getValue() < this.parms) {
               pa[p++] = (String)entry.getKey();
            }
         }

         return pa;
      } else {
         return null;
      }
   }

   public String[] getLocalVariables() {
      if (this.namedVariables != null && this.vars > 0) {
         String[] pa = new String[this.parms - (this.hoistedVariables == null ? 0 : this.hoistedVariables.size())];
         int p = 0;

         for(Map.Entry<String, Integer> entry : this.namedVariables.entrySet()) {
            int symnum = (Integer)entry.getValue();
            if (symnum >= this.parms && (this.hoistedVariables == null || !this.hoistedVariables.containsKey(symnum))) {
               pa[p++] = (String)entry.getKey();
            }
         }

         return pa;
      } else {
         return null;
      }
   }

   public static final class Frame {
      private final Scope scope;
      private final Object[] stack;
      private int curried = 0;

      public Frame(Scope s, Object[] r, int c) {
         this.scope = s;
         this.stack = r;
         this.curried = c;
      }

      public Scope getScope() {
         return this.scope;
      }

      public int hashCode() {
         return Arrays.deepHashCode(this.stack);
      }

      public boolean equals(Object obj) {
         if (obj == null) {
            return false;
         } else if (this.getClass() != obj.getClass()) {
            return false;
         } else {
            Frame other = (Frame)obj;
            return Arrays.deepEquals(this.stack, other.stack);
         }
      }

      public Object get(int s) {
         return this.stack[s];
      }

      public void set(int r, Object value) {
         this.stack[r] = value;
      }

      public Frame assign(Object... values) {
         if (this.stack != null && values != null && values.length > 0) {
            Object[] copy = this.stack.clone();
            int ncopy = Math.min(copy.length - this.curried, values.length);
            System.arraycopy(values, 0, copy, this.curried, ncopy);
            return new Frame(this.scope, copy, this.curried + ncopy);
         } else {
            return this;
         }
      }
   }
}
