package org.apache.commons.jexl3.internal;

import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlEngine;
import org.apache.commons.jexl3.JexlExpression;
import org.apache.commons.jexl3.JexlScript;
import org.apache.commons.jexl3.parser.ASTJexlScript;
import org.apache.commons.jexl3.parser.JexlNode;

public class Script implements JexlScript, JexlExpression {
   protected final Engine jexl;
   protected final String source;
   protected final ASTJexlScript script;
   protected int version;

   protected Script(Engine engine, String expr, ASTJexlScript ref) {
      this.jexl = engine;
      this.source = expr;
      this.script = ref;
      this.version = this.jexl.getUberspect().getVersion();
   }

   protected void checkCacheVersion() {
      int uberVersion = this.jexl.getUberspect().getVersion();
      if (this.version != uberVersion) {
         if (this.version > 0) {
            this.script.clearCache();
         }

         this.version = uberVersion;
      }

   }

   protected Scope.Frame createFrame(Object[] args) {
      return this.script.createFrame(args);
   }

   protected Interpreter createInterpreter(JexlContext context, Scope.Frame frame) {
      return this.jexl.createInterpreter(context, frame);
   }

   public JexlEngine getEngine() {
      return this.jexl;
   }

   public String getSourceText() {
      return this.source;
   }

   public String getParsedText() {
      return this.getParsedText(2);
   }

   public String getParsedText(int indent) {
      Debugger debug = new Debugger();
      debug.setIndentation(indent);
      debug.debug((JexlNode)this.script);
      return debug.toString();
   }

   public int hashCode() {
      int hash = 17;
      hash = 31 * hash + (this.jexl != null ? this.jexl.hashCode() : 0);
      hash = 31 * hash + (this.source != null ? this.source.hashCode() : 0);
      return hash;
   }

   public boolean equals(Object obj) {
      if (obj == null) {
         return false;
      } else if (this.getClass() != obj.getClass()) {
         return false;
      } else {
         Script other = (Script)obj;
         if (this.jexl != other.jexl) {
            return false;
         } else {
            if (this.source == null) {
               if (other.source != null) {
                  return false;
               }
            } else if (!this.source.equals(other.source)) {
               return false;
            }

            return true;
         }
      }
   }

   public String toString() {
      CharSequence src = this.source;
      if (src == null) {
         Debugger debug = new Debugger();
         debug.debug((JexlNode)this.script);
         src = debug.toString();
      }

      return src.toString();
   }

   public Object evaluate(JexlContext context) {
      return this.execute(context);
   }

   public Object execute(JexlContext context) {
      this.checkCacheVersion();
      Scope.Frame frame = this.createFrame((Object[])null);
      Interpreter interpreter = this.createInterpreter(context, frame);
      return interpreter.interpret(this.script);
   }

   public Object execute(JexlContext context, Object... args) {
      this.checkCacheVersion();
      Scope.Frame frame = this.createFrame(args != null && args.length > 0 ? args : null);
      Interpreter interpreter = this.createInterpreter(context, frame);
      return interpreter.interpret(this.script);
   }

   public JexlScript curry(Object... args) {
      String[] parms = this.script.getParameters();
      return (JexlScript)(parms != null && parms.length != 0 ? new Curried(this, args) : this);
   }

   public String[] getParameters() {
      return this.script.getParameters();
   }

   public String[] getLocalVariables() {
      return this.script.getLocalVariables();
   }

   public Set<List<String>> getVariables() {
      return this.jexl.getVariables(this.script);
   }

   public Map<String, Object> getPragmas() {
      return this.script.getPragmas();
   }

   public Callable callable(JexlContext context) {
      return this.callable(context, (Object[])null);
   }

   public Callable callable(JexlContext context, Object... args) {
      return new Callable(this.jexl.createInterpreter(context, this.script.createFrame(args)));
   }

   public static class Curried extends Script {
      private final Scope.Frame frame;

      protected Curried(Script base, Object[] args) {
         super(base.jexl, base.source, base.script);
         Scope.Frame sf = base instanceof Curried ? ((Curried)base).frame : null;
         if (sf != null) {
            this.frame = sf.assign(args);
         } else {
            this.frame = this.script.createFrame(args);
         }

      }

      protected Scope.Frame createFrame(Object[] args) {
         return this.frame != null ? this.frame.assign(args) : super.createFrame(args);
      }

      public boolean equals(Object obj) {
         return this == obj;
      }

      public int hashCode() {
         return System.identityHashCode(this);
      }

      public Object execute(JexlContext context) {
         return this.execute(context, (Object[])null);
      }

      public Object execute(JexlContext context, Object... args) {
         Scope.Frame callFrame = null;
         if (this.frame != null) {
            callFrame = this.frame.assign(args);
         }

         Interpreter interpreter = this.jexl.createInterpreter(context, callFrame);
         JexlNode block = this.script.jjtGetChild(this.script.jjtGetNumChildren() - 1);
         return interpreter.interpret(block);
      }
   }

   public class Callable implements java.util.concurrent.Callable<Object> {
      protected final Interpreter interpreter;
      protected volatile Object result;

      protected Callable(Interpreter intrprtr) {
         this.interpreter = intrprtr;
         this.result = intrprtr;
      }

      protected Object interpret() {
         return this.interpreter.interpret(Script.this.script);
      }

      public Object call() throws Exception {
         synchronized(this) {
            if (this.result == this.interpreter) {
               Script.this.checkCacheVersion();
               this.result = this.interpret();
            }

            return this.result;
         }
      }

      public boolean cancel() {
         return this.interpreter.cancel();
      }

      public boolean isCancelled() {
         return this.interpreter.isCancelled();
      }

      public boolean isCancellable() {
         return this.interpreter.isCancellable();
      }
   }
}
