package org.apache.commons.jexl3.internal;

import org.apache.commons.jexl3.JxltEngine;
import org.apache.commons.jexl3.parser.ASTBlock;
import org.apache.commons.jexl3.parser.ASTFunctionNode;
import org.apache.commons.jexl3.parser.ASTIdentifier;
import org.apache.commons.jexl3.parser.ASTJexlScript;
import org.apache.commons.jexl3.parser.ASTNumberLiteral;
import org.apache.commons.jexl3.parser.JexlNode;

public class TemplateDebugger extends Debugger {
   private ASTJexlScript script;
   private TemplateEngine.TemplateExpression[] exprs;

   public boolean debug(JxltEngine.Expression je) {
      if (je instanceof TemplateEngine.TemplateExpression) {
         TemplateEngine.TemplateExpression te = (TemplateEngine.TemplateExpression)je;
         return this.visit((TemplateEngine.TemplateExpression)te, this) != null;
      } else {
         return false;
      }
   }

   public boolean debug(JxltEngine.Template jt) {
      if (!(jt instanceof TemplateScript)) {
         return false;
      } else {
         TemplateScript ts = (TemplateScript)jt;
         this.exprs = ts.getExpressions() == null ? new TemplateEngine.TemplateExpression[0] : ts.getExpressions();
         this.script = ts.getScript();
         this.start = 0;
         this.end = 0;
         this.indentLevel = 0;
         this.builder.setLength(0);
         this.cause = this.script;
         int num = this.script.jjtGetNumChildren();

         for(int i = 0; i < num; ++i) {
            JexlNode child = this.script.jjtGetChild(i);
            this.acceptStatement(child, null);
         }

         if (this.builder.length() > 0 && this.builder.charAt(this.builder.length() - 1) != '\n') {
            this.builder.append('\n');
         }

         this.end = this.builder.length();
         return this.end > 0;
      }
   }

   protected Object visit(ASTBlock node, Object data) {
      if (this.exprs == null) {
         return super.visit(node, data);
      } else {
         this.builder.append('{');
         if (this.indent > 0) {
            ++this.indentLevel;
            this.builder.append('\n');
         } else {
            this.builder.append(' ');
         }

         int num = node.jjtGetNumChildren();

         for(int i = 0; i < num; ++i) {
            JexlNode child = node.jjtGetChild(i);
            this.acceptStatement(child, data);
         }

         this.newJexlLine();
         if (this.indent > 0) {
            --this.indentLevel;

            for(int i = 0; i < this.indentLevel; ++i) {
               for(int s = 0; s < this.indent; ++s) {
                  this.builder.append(' ');
               }
            }
         }

         this.builder.append('}');
         return data;
      }
   }

   protected Object acceptStatement(JexlNode child, Object data) {
      if (this.exprs != null) {
         int printe = this.getPrintStatement(child);
         if (printe >= 0) {
            TemplateEngine.TemplateExpression te = this.exprs[printe];
            return this.visit(te, data);
         }

         this.newJexlLine();
      }

      return super.acceptStatement(child, data);
   }

   private int getPrintStatement(JexlNode child) {
      if (child instanceof ASTFunctionNode) {
         ASTFunctionNode node = (ASTFunctionNode)child;
         int num = node.jjtGetNumChildren();
         if (num == 3) {
            ASTIdentifier ns = (ASTIdentifier)node.jjtGetChild(0);
            ASTIdentifier fn = (ASTIdentifier)node.jjtGetChild(1);
            JexlNode args = node.jjtGetChild(2);
            if ("jexl".equals(ns.getName()) && "print".equals(fn.getName()) && args.jjtGetNumChildren() == 1 && args.jjtGetChild(0) instanceof ASTNumberLiteral) {
               ASTNumberLiteral exprn = (ASTNumberLiteral)args.jjtGetChild(0);
               int n = exprn.getLiteral().intValue();
               if (this.exprs != null && n >= 0 && n < this.exprs.length) {
                  return n;
               }
            }
         }
      }

      return -1;
   }

   private void newJexlLine() {
      int length = this.builder.length();
      if (length == 0) {
         this.builder.append("$$ ");
      } else {
         for(int i = length - 1; i >= 0; --i) {
            char c = this.builder.charAt(i);
            if (c == '\n') {
               this.builder.append("$$ ");
               break;
            }

            if (c == '}') {
               this.builder.append("\n$$ ");
               break;
            }

            if (c != ' ') {
               break;
            }
         }
      }

   }

   private Object visit(TemplateEngine.TemplateExpression expr, Object data) {
      Object r;
      switch (expr.getType()) {
         case CONSTANT:
            r = this.visit((TemplateEngine.ConstantExpression)expr, data);
            break;
         case IMMEDIATE:
            r = this.visit((TemplateEngine.ImmediateExpression)expr, data);
            break;
         case DEFERRED:
            r = this.visit((TemplateEngine.DeferredExpression)expr, data);
            break;
         case NESTED:
            r = this.visit((TemplateEngine.NestedExpression)expr, data);
            break;
         case COMPOSITE:
            r = this.visit((TemplateEngine.CompositeExpression)expr, data);
            break;
         default:
            r = null;
      }

      return r;
   }

   private Object visit(TemplateEngine.ConstantExpression expr, Object data) {
      expr.asString(this.builder);
      return data;
   }

   private Object visit(TemplateEngine.ImmediateExpression expr, Object data) {
      this.builder.append((char)(expr.isImmediate() ? '$' : '#'));
      this.builder.append('{');
      super.accept(expr.node, data);
      this.builder.append('}');
      return data;
   }

   private Object visit(TemplateEngine.DeferredExpression expr, Object data) {
      this.builder.append((char)(expr.isImmediate() ? '$' : '#'));
      this.builder.append('{');
      super.accept(expr.node, data);
      this.builder.append('}');
      return data;
   }

   private Object visit(TemplateEngine.NestedExpression expr, Object data) {
      super.accept(expr.node, data);
      return data;
   }

   private Object visit(TemplateEngine.CompositeExpression expr, Object data) {
      for(TemplateEngine.TemplateExpression ce : expr.exprs) {
         this.visit(ce, data);
      }

      return data;
   }
}
