package org.apache.commons.jexl3.internal;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlException;
import org.apache.commons.jexl3.JexlInfo;
import org.apache.commons.jexl3.JxltEngine;
import org.apache.commons.jexl3.parser.ASTJexlScript;
import org.apache.commons.jexl3.parser.JexlNode;
import org.apache.commons.jexl3.parser.StringParser;

public final class TemplateEngine extends JxltEngine {
   private final SoftCache<String, TemplateExpression> cache;
   private final Engine jexl;
   private final char immediateChar;
   private final char deferredChar;
   private boolean noscript = true;

   public TemplateEngine(Engine aJexl, boolean noScript, int cacheSize, char immediate, char deferred) {
      this.jexl = aJexl;
      this.cache = new SoftCache<String, TemplateExpression>(cacheSize);
      this.immediateChar = immediate;
      this.deferredChar = deferred;
      this.noscript = noScript;
   }

   char getImmediateChar() {
      return this.immediateChar;
   }

   char getDeferredChar() {
      return this.deferredChar;
   }

   public Engine getEngine() {
      return this.jexl;
   }

   public void clearCache() {
      synchronized(this.cache) {
         this.cache.clear();
      }
   }

   public Expression createExpression(JexlInfo info, String expression) {
      if (info == null) {
         info = this.jexl.createInfo();
      }

      Exception xuel = null;
      TemplateExpression stmt = null;

      try {
         stmt = this.cache.get(expression);
         if (stmt == null) {
            stmt = this.parseExpression(info, expression, (Scope)null);
            this.cache.put(expression, stmt);
         }
      } catch (JexlException xjexl) {
         xuel = new Exception(xjexl.getInfo(), "failed to parse '" + expression + "'", xjexl);
      }

      if (xuel != null) {
         if (!this.jexl.isSilent()) {
            throw xuel;
         }

         this.jexl.logger.warn(xuel.getMessage(), xuel.getCause());
         stmt = null;
      }

      return stmt;
   }

   static Exception createException(JexlInfo info, String action, TemplateExpression expr, java.lang.Exception xany) {
      StringBuilder strb = new StringBuilder("failed to ");
      strb.append(action);
      if (expr != null) {
         strb.append(" '");
         strb.append(expr.toString());
         strb.append("'");
      }

      Throwable cause = xany.getCause();
      if (cause != null) {
         String causeMsg = cause.getMessage();
         if (causeMsg != null) {
            strb.append(", ");
            strb.append(causeMsg);
         }
      }

      return new Exception(info, strb.toString(), xany);
   }

   TemplateExpression parseExpression(JexlInfo info, String expr, Scope scope) {
      int size = expr.length();
      ExpressionBuilder builder = new ExpressionBuilder(0);
      StringBuilder strb = new StringBuilder(size);
      ParseState state = ParseState.CONST;
      int immediate1 = 0;
      int deferred1 = 0;
      int inner1 = 0;
      boolean nested = false;
      int inested = -1;
      int lineno = info.getLine();

      for(int column = 0; column < size; ++column) {
         char c = expr.charAt(column);
         switch (state) {
            case CONST:
               if (c == this.immediateChar) {
                  state = ParseState.IMMEDIATE0;
               } else if (c == this.deferredChar) {
                  inested = column;
                  state = ParseState.DEFERRED0;
               } else if (c == '\\') {
                  state = ParseState.ESCAPE;
               } else {
                  strb.append(c);
               }
               break;
            case IMMEDIATE0:
               if (c == '{') {
                  state = ParseState.IMMEDIATE1;
                  if (strb.length() > 0) {
                     TemplateExpression cexpr = new ConstantExpression(strb.toString(), (TemplateExpression)null);
                     builder.add(cexpr);
                     strb.delete(0, Integer.MAX_VALUE);
                  }
               } else {
                  strb.append(this.immediateChar);
                  strb.append(c);
                  state = ParseState.CONST;
               }
               break;
            case DEFERRED0:
               if (c == '{') {
                  state = ParseState.DEFERRED1;
                  if (strb.length() > 0) {
                     TemplateExpression cexpr = new ConstantExpression(strb.toString(), (TemplateExpression)null);
                     builder.add(cexpr);
                     strb.delete(0, Integer.MAX_VALUE);
                  }
               } else {
                  strb.append(this.deferredChar);
                  strb.append(c);
                  state = ParseState.CONST;
               }
               break;
            case IMMEDIATE1:
               if (c == '}') {
                  if (immediate1 > 0) {
                     --immediate1;
                     strb.append(c);
                  } else {
                     String src = strb.toString();
                     TemplateExpression iexpr = new ImmediateExpression(src, this.jexl.parse(info.at(lineno, column), src, scope, false, this.noscript), (TemplateExpression)null);
                     builder.add(iexpr);
                     strb.delete(0, Integer.MAX_VALUE);
                     state = ParseState.CONST;
                  }
               } else {
                  if (c == '{') {
                     ++immediate1;
                  }

                  strb.append(c);
               }
               break;
            case DEFERRED1:
               if (c == '"' || c == '\'') {
                  strb.append(c);
                  column = StringParser.readString(strb, expr, column + 1, c);
                  continue;
               }

               if (c == '{') {
                  if (expr.charAt(column - 1) == this.immediateChar) {
                     ++inner1;
                     strb.deleteCharAt(strb.length() - 1);
                     nested = true;
                  } else {
                     ++deferred1;
                     strb.append(c);
                  }
                  continue;
               }

               if (c == '}') {
                  if (deferred1 > 0) {
                     --deferred1;
                     strb.append(c);
                  } else if (inner1 > 0) {
                     --inner1;
                  } else {
                     String src = strb.toString();
                     TemplateExpression dexpr;
                     if (nested) {
                        dexpr = new NestedExpression(expr.substring(inested, column + 1), this.jexl.parse(info.at(lineno, column), src, scope, false, this.noscript), (TemplateExpression)null);
                     } else {
                        dexpr = new DeferredExpression(strb.toString(), this.jexl.parse(info.at(lineno, column), src, scope, false, this.noscript), (TemplateExpression)null);
                     }

                     builder.add(dexpr);
                     strb.delete(0, Integer.MAX_VALUE);
                     nested = false;
                     state = ParseState.CONST;
                  }
               } else {
                  strb.append(c);
               }
               break;
            case ESCAPE:
               if (c == this.deferredChar) {
                  strb.append(this.deferredChar);
               } else if (c == this.immediateChar) {
                  strb.append(this.immediateChar);
               } else {
                  strb.append('\\');
                  strb.append(c);
               }

               state = ParseState.CONST;
               break;
            default:
               throw new UnsupportedOperationException("unexpected unified expression type");
         }

         if (c == '\n') {
            ++lineno;
         }
      }

      if (state != ParseState.CONST) {
         throw new Exception(info.at(lineno, 0), "malformed expression: " + expr, (Throwable)null);
      } else {
         if (strb.length() > 0) {
            TemplateExpression cexpr = new ConstantExpression(strb.toString(), (TemplateExpression)null);
            builder.add(cexpr);
         }

         return builder.build(this, (TemplateExpression)null);
      }
   }

   protected int startsWith(CharSequence sequence, CharSequence pattern) {
      int length = sequence.length();

      int s;
      for(s = 0; s < length && Character.isSpaceChar(sequence.charAt(s)); ++s) {
      }

      if (s < length && pattern.length() <= length - s) {
         sequence = sequence.subSequence(s, length);
         if (sequence.subSequence(0, pattern.length()).equals(pattern)) {
            return s + pattern.length();
         }
      }

      return -1;
   }

   protected static Iterator<CharSequence> readLines(final Reader reader) {
      if (!reader.markSupported()) {
         throw new IllegalArgumentException("mark support in reader required");
      } else {
         return new Iterator<CharSequence>() {
            private CharSequence next = this.doNext();

            private CharSequence doNext() {
               StringBuffer strb = new StringBuffer(64);
               boolean eol = false;

               int c;
               try {
                  while((c = reader.read()) >= 0) {
                     if (eol) {
                        reader.reset();
                        break;
                     }

                     if (c == 10) {
                        eol = true;
                     }

                     strb.append((char)c);
                     reader.mark(1);
                  }
               } catch (IOException var5) {
                  return null;
               }

               return strb.length() > 0 ? strb : null;
            }

            public boolean hasNext() {
               return this.next != null;
            }

            public CharSequence next() {
               CharSequence current = this.next;
               if (current != null) {
                  this.next = this.doNext();
               }

               return current;
            }

            public void remove() {
               throw new UnsupportedOperationException("Not supported.");
            }
         };
      }
   }

   protected List<Block> readTemplate(String prefix, Reader source) {
      ArrayList<Block> blocks = new ArrayList();
      BufferedReader reader;
      if (source instanceof BufferedReader) {
         reader = (BufferedReader)source;
      } else {
         reader = new BufferedReader(source);
      }

      StringBuilder strb = new StringBuilder();
      BlockType type = null;
      Iterator<CharSequence> lines = readLines(reader);
      int lineno = 0;

      int start;
      for(start = 0; lines.hasNext(); ++lineno) {
         CharSequence line = (CharSequence)lines.next();
         if (line == null) {
            break;
         }

         if (type == null) {
            int prefixLen = this.startsWith(line, prefix);
            if (prefixLen >= 0) {
               type = BlockType.DIRECTIVE;
               strb.append(line.subSequence(prefixLen, line.length()));
            } else {
               type = BlockType.VERBATIM;
               strb.append(line.subSequence(0, line.length()));
            }

            start = lineno;
         } else if (type == BlockType.DIRECTIVE) {
            int prefixLen = this.startsWith(line, prefix);
            if (prefixLen < 0) {
               Block directive = new Block(BlockType.DIRECTIVE, start, strb.toString());
               strb.delete(0, Integer.MAX_VALUE);
               blocks.add(directive);
               type = BlockType.VERBATIM;
               strb.append(line.subSequence(0, line.length()));
               start = lineno;
            } else {
               strb.append(line.subSequence(prefixLen, line.length()));
            }
         } else if (type == BlockType.VERBATIM) {
            int prefixLen = this.startsWith(line, prefix);
            if (prefixLen >= 0) {
               Block verbatim = new Block(BlockType.VERBATIM, start, strb.toString());
               strb.delete(0, Integer.MAX_VALUE);
               blocks.add(verbatim);
               type = BlockType.DIRECTIVE;
               strb.append(line.subSequence(prefixLen, line.length()));
               start = lineno;
            } else {
               strb.append(line.subSequence(0, line.length()));
            }
         }
      }

      if (type != null && strb.length() > 0) {
         Block block = new Block(type, start, strb.toString());
         blocks.add(block);
      }

      blocks.trimToSize();
      return blocks;
   }

   public TemplateScript createTemplate(JexlInfo info, String prefix, Reader source, String... parms) {
      return new TemplateScript(this, info, prefix, source, parms);
   }

   static enum ExpressionType {
      CONSTANT(0),
      IMMEDIATE(1),
      DEFERRED(2),
      NESTED(2),
      COMPOSITE(-1);

      private final int index;

      private ExpressionType(int idx) {
         this.index = idx;
      }
   }

   static final class ExpressionBuilder {
      private final int[] counts;
      private final ArrayList<TemplateExpression> expressions;

      private ExpressionBuilder(int size) {
         this.counts = new int[]{0, 0, 0};
         this.expressions = new ArrayList(size <= 0 ? 3 : size);
      }

      private void add(TemplateExpression expr) {
         int var10002 = this.counts[expr.getType().index]++;
         this.expressions.add(expr);
      }

      public String toString() {
         return this.toString(new StringBuilder()).toString();
      }

      private StringBuilder toString(StringBuilder error) {
         error.append("exprs{");
         error.append(this.expressions.size());
         error.append(", constant:");
         error.append(this.counts[ExpressionType.CONSTANT.index]);
         error.append(", immediate:");
         error.append(this.counts[ExpressionType.IMMEDIATE.index]);
         error.append(", deferred:");
         error.append(this.counts[ExpressionType.DEFERRED.index]);
         error.append("}");
         return error;
      }

      private TemplateExpression build(TemplateEngine el, TemplateExpression source) {
         int sum = 0;

         for(int count : this.counts) {
            sum += count;
         }

         if (this.expressions.size() != sum) {
            StringBuilder error = new StringBuilder("parsing algorithm error: ");
            throw new IllegalStateException(this.toString(error).toString());
         } else {
            return (TemplateExpression)(this.expressions.size() == 1 ? (TemplateExpression)this.expressions.get(0) : el.new CompositeExpression(this.counts, this.expressions, source));
         }
      }
   }

   abstract class TemplateExpression implements Expression {
      protected final TemplateExpression source;

      TemplateExpression(TemplateExpression src) {
         this.source = src != null ? src : this;
      }

      public boolean isImmediate() {
         return true;
      }

      public final boolean isDeferred() {
         return !this.isImmediate();
      }

      abstract ExpressionType getType();

      JexlInfo getInfo() {
         return null;
      }

      public final String toString() {
         StringBuilder strb = new StringBuilder();
         this.asString(strb);
         if (this.source != this) {
            strb.append(" /*= ");
            strb.append(this.source.toString());
            strb.append(" */");
         }

         return strb.toString();
      }

      public String asString() {
         StringBuilder strb = new StringBuilder();
         this.asString(strb);
         return strb.toString();
      }

      public Set<List<String>> getVariables() {
         return Collections.emptySet();
      }

      public final TemplateExpression getSource() {
         return this.source;
      }

      protected void getVariables(Engine.VarCollector collector) {
      }

      public final TemplateExpression prepare(JexlContext context) {
         return this.prepare((Scope.Frame)null, context);
      }

      protected final TemplateExpression prepare(Scope.Frame frame, JexlContext context) {
         try {
            Interpreter interpreter = new TemplateInterpreter(TemplateEngine.this.jexl, context, frame, (TemplateExpression[])null, (Writer)null);
            return this.prepare(interpreter);
         } catch (JexlException xjexl) {
            JexlException xuel = TemplateEngine.createException(xjexl.getInfo(), "prepare", this, xjexl);
            if (TemplateEngine.this.jexl.isSilent()) {
               TemplateEngine.this.jexl.logger.warn(xuel.getMessage(), xuel.getCause());
               return null;
            } else {
               throw xuel;
            }
         }
      }

      protected TemplateExpression prepare(Interpreter interpreter) {
         return this;
      }

      public final Object evaluate(JexlContext context) {
         return this.evaluate((Scope.Frame)null, context);
      }

      protected final Object evaluate(Scope.Frame frame, JexlContext context) {
         try {
            Interpreter interpreter = new TemplateInterpreter(TemplateEngine.this.jexl, context, frame, (TemplateExpression[])null, (Writer)null);
            return this.evaluate(interpreter);
         } catch (JexlException xjexl) {
            JexlException xuel = TemplateEngine.createException(xjexl.getInfo(), "evaluate", this, xjexl);
            if (TemplateEngine.this.jexl.isSilent()) {
               TemplateEngine.this.jexl.logger.warn(xuel.getMessage(), xuel.getCause());
               return null;
            } else {
               throw xuel;
            }
         }
      }

      protected abstract Object evaluate(Interpreter var1);
   }

   class ConstantExpression extends TemplateExpression {
      private final Object value;

      ConstantExpression(Object val, TemplateExpression source) {
         super(source);
         if (val == null) {
            throw new NullPointerException("constant can not be null");
         } else {
            if (val instanceof String) {
               val = StringParser.buildString((String)val, false);
            }

            this.value = val;
         }
      }

      ExpressionType getType() {
         return ExpressionType.CONSTANT;
      }

      public StringBuilder asString(StringBuilder strb) {
         if (this.value != null) {
            strb.append(this.value.toString());
         }

         return strb;
      }

      protected Object evaluate(Interpreter interpreter) {
         return this.value;
      }
   }

   abstract class JexlBasedExpression extends TemplateExpression {
      protected final CharSequence expr;
      protected final JexlNode node;

      protected JexlBasedExpression(CharSequence theExpr, JexlNode theNode, TemplateExpression theSource) {
         super(theSource);
         this.expr = theExpr;
         this.node = theNode;
      }

      public StringBuilder asString(StringBuilder strb) {
         strb.append(this.isImmediate() ? TemplateEngine.this.immediateChar : TemplateEngine.this.deferredChar);
         strb.append("{");
         strb.append(this.expr);
         strb.append("}");
         return strb;
      }

      protected Object evaluate(Interpreter interpreter) {
         return interpreter.interpret(this.node);
      }

      public Set<List<String>> getVariables() {
         Engine.VarCollector collector = new Engine.VarCollector();
         this.getVariables(collector);
         return collector.collected();
      }

      protected void getVariables(Engine.VarCollector collector) {
         TemplateEngine.this.jexl.getVariables(this.node instanceof ASTJexlScript ? (ASTJexlScript)this.node : null, this.node, collector);
      }

      JexlInfo getInfo() {
         return this.node.jexlInfo();
      }
   }

   class ImmediateExpression extends JexlBasedExpression {
      ImmediateExpression(CharSequence expr, JexlNode node, TemplateExpression source) {
         super(expr, node, source);
      }

      ExpressionType getType() {
         return ExpressionType.IMMEDIATE;
      }

      protected TemplateExpression prepare(Interpreter interpreter) {
         Object value = this.evaluate(interpreter);
         return value != null ? TemplateEngine.this.new ConstantExpression(value, this.source) : null;
      }
   }

   class DeferredExpression extends JexlBasedExpression {
      DeferredExpression(CharSequence expr, JexlNode node, TemplateExpression source) {
         super(expr, node, source);
      }

      public boolean isImmediate() {
         return false;
      }

      ExpressionType getType() {
         return ExpressionType.DEFERRED;
      }

      protected TemplateExpression prepare(Interpreter interpreter) {
         return TemplateEngine.this.new ImmediateExpression(this.expr, this.node, this.source);
      }

      protected void getVariables(Engine.VarCollector collector) {
      }
   }

   class NestedExpression extends JexlBasedExpression {
      NestedExpression(CharSequence expr, JexlNode node, TemplateExpression source) {
         super(expr, node, source);
         if (this.source != this) {
            throw new IllegalArgumentException("Nested TemplateExpression can not have a source");
         }
      }

      public StringBuilder asString(StringBuilder strb) {
         strb.append(this.expr);
         return strb;
      }

      public boolean isImmediate() {
         return false;
      }

      ExpressionType getType() {
         return ExpressionType.NESTED;
      }

      protected TemplateExpression prepare(Interpreter interpreter) {
         String value = interpreter.interpret(this.node).toString();
         JexlNode dnode = TemplateEngine.this.jexl.parse(this.node.jexlInfo(), value, (Scope)null, false, TemplateEngine.this.noscript);
         return TemplateEngine.this.new ImmediateExpression(value, dnode, this);
      }

      protected Object evaluate(Interpreter interpreter) {
         return this.prepare(interpreter).evaluate(interpreter);
      }
   }

   class CompositeExpression extends TemplateExpression {
      private final int meta;
      protected final TemplateExpression[] exprs;

      CompositeExpression(int[] counters, ArrayList<TemplateExpression> list, TemplateExpression src) {
         super(src);
         this.exprs = (TemplateExpression[])list.toArray(new TemplateExpression[list.size()]);
         this.meta = (counters[ExpressionType.DEFERRED.index] > 0 ? 2 : 0) | (counters[ExpressionType.IMMEDIATE.index] > 0 ? 1 : 0);
      }

      public boolean isImmediate() {
         return (this.meta & 2) == 0;
      }

      ExpressionType getType() {
         return ExpressionType.COMPOSITE;
      }

      public StringBuilder asString(StringBuilder strb) {
         for(TemplateExpression e : this.exprs) {
            e.asString(strb);
         }

         return strb;
      }

      public Set<List<String>> getVariables() {
         Engine.VarCollector collector = new Engine.VarCollector();

         for(TemplateExpression expr : this.exprs) {
            expr.getVariables(collector);
         }

         return collector.collected();
      }

      protected void getVariables(Engine.VarCollector collector) {
         for(TemplateExpression expr : this.exprs) {
            expr.getVariables(collector);
         }

      }

      protected TemplateExpression prepare(Interpreter interpreter) {
         if (this.source != this) {
            return this;
         } else {
            int size = this.exprs.length;
            ExpressionBuilder builder = new ExpressionBuilder(size);
            boolean eq = true;

            for(int e = 0; e < size; ++e) {
               TemplateExpression expr = this.exprs[e];
               TemplateExpression prepared = expr.prepare(interpreter);
               if (prepared != null) {
                  builder.add(prepared);
               }

               eq &= expr == prepared;
            }

            return (TemplateExpression)(eq ? this : builder.build(TemplateEngine.this, this));
         }
      }

      protected Object evaluate(Interpreter interpreter) {
         int size = this.exprs.length;
         StringBuilder strb = new StringBuilder();

         for(int e = 0; e < size; ++e) {
            Object value = this.exprs[e].evaluate(interpreter);
            if (value != null) {
               strb.append(value.toString());
            }
         }

         Object value = strb.toString();
         return value;
      }
   }

   private static enum ParseState {
      CONST,
      IMMEDIATE0,
      DEFERRED0,
      IMMEDIATE1,
      DEFERRED1,
      ESCAPE;
   }

   static enum BlockType {
      VERBATIM,
      DIRECTIVE;
   }

   static final class Block {
      private final BlockType type;
      private final int line;
      private final String body;

      Block(BlockType theType, int theLine, String theBlock) {
         this.type = theType;
         this.line = theLine;
         this.body = theBlock;
      }

      BlockType getType() {
         return this.type;
      }

      int getLine() {
         return this.line;
      }

      String getBody() {
         return this.body;
      }

      public String toString() {
         if (BlockType.VERBATIM.equals(this.type)) {
            return this.body;
         } else {
            StringBuilder strb = new StringBuilder(64);
            Iterator<CharSequence> lines = TemplateEngine.readLines(new StringReader(this.body));

            while(lines.hasNext()) {
               strb.append("$$").append((CharSequence)lines.next());
            }

            return strb.toString();
         }
      }

      protected void toString(StringBuilder strb, String prefix) {
         if (BlockType.VERBATIM.equals(this.type)) {
            strb.append(this.body);
         } else {
            Iterator<CharSequence> lines = TemplateEngine.readLines(new StringReader(this.body));

            while(lines.hasNext()) {
               strb.append(prefix).append((CharSequence)lines.next());
            }
         }

      }
   }
}
