package org.apache.commons.jexl3.internal;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlInfo;
import org.apache.commons.jexl3.JxltEngine;
import org.apache.commons.jexl3.introspection.JexlMethod;
import org.apache.commons.jexl3.introspection.JexlUberspect;
import org.apache.commons.jexl3.parser.ASTArguments;
import org.apache.commons.jexl3.parser.ASTFunctionNode;
import org.apache.commons.jexl3.parser.ASTIdentifier;
import org.apache.commons.jexl3.parser.JexlNode;

public class TemplateInterpreter extends Interpreter {
   private final TemplateEngine.TemplateExpression[] exprs;
   private final Writer writer;

   TemplateInterpreter(Engine jexl, JexlContext jcontext, Scope.Frame jframe, TemplateEngine.TemplateExpression[] expressions, Writer out) {
      super(jexl, jcontext, jframe);
      this.exprs = expressions;
      this.writer = out;
   }

   public void include(TemplateScript script, Object... args) {
      script.evaluate(this.context, this.writer, args);
   }

   public void print(int e) {
      if (e >= 0 && e < this.exprs.length) {
         TemplateEngine.TemplateExpression expr = this.exprs[e];
         if (expr.isDeferred()) {
            expr = expr.prepare(this.frame, this.context);
         }

         if (expr instanceof TemplateEngine.CompositeExpression) {
            this.printComposite((TemplateEngine.CompositeExpression)expr);
         } else {
            this.doPrint(expr.getInfo(), expr.evaluate((Interpreter)this));
         }

      }
   }

   private void printComposite(TemplateEngine.CompositeExpression composite) {
      TemplateEngine.TemplateExpression[] cexprs = composite.exprs;
      int size = cexprs.length;

      for(int e = 0; e < size; ++e) {
         Object value = cexprs[e].evaluate((Interpreter)this);
         this.doPrint(cexprs[e].getInfo(), value);
      }

   }

   private void doPrint(JexlInfo info, Object arg) {
      try {
         if (this.writer != null) {
            if (arg instanceof CharSequence) {
               this.writer.write(arg.toString());
            } else if (arg != null) {
               Object[] value = new Object[]{arg};
               JexlUberspect uber = this.jexl.getUberspect();
               JexlMethod method = uber.getMethod(this.writer, "print", value);
               if (method != null) {
                  method.invoke(this.writer, value);
               } else {
                  this.writer.write(arg.toString());
               }
            }
         }

      } catch (IOException xio) {
         throw TemplateEngine.createException(info, "call print", (TemplateEngine.TemplateExpression)null, xio);
      } catch (Exception xany) {
         throw TemplateEngine.createException(info, "invoke print", (TemplateEngine.TemplateExpression)null, xany);
      }
   }

   protected Object resolveNamespace(String prefix, JexlNode node) {
      return "jexl".equals(prefix) ? this : super.resolveNamespace(prefix, node);
   }

   protected Object visit(ASTFunctionNode node, Object data) {
      int argc = node.jjtGetNumChildren();
      if (argc > 2) {
         String prefix = ((ASTIdentifier)node.jjtGetChild(0)).getName();
         if ("jexl".equals(prefix)) {
            ASTIdentifier functionNode = (ASTIdentifier)node.jjtGetChild(1);
            ASTArguments argNode = (ASTArguments)node.jjtGetChild(2);
            String fname = functionNode.getName();
            if ("print".equals(fname)) {
               Object[] argv = this.visit((ASTArguments)argNode, null);
               this.print((Integer)argv[0]);
               return null;
            }

            if ("include".equals(fname)) {
               Object[] argv = this.visit((ASTArguments)argNode, null);
               if (argv != null && argv.length > 0 && argv[0] instanceof TemplateScript) {
                  TemplateScript script = (TemplateScript)argv[0];
                  if (argv.length > 1) {
                     argv = Arrays.copyOfRange(argv, 1, argv.length);
                  } else {
                     argv = null;
                  }

                  this.include(script, argv);
                  return null;
               }
            }

            throw new JxltEngine.Exception(node.jexlInfo(), "no callable template function " + fname, (Throwable)null);
         }
      }

      return super.visit(node, data);
   }

   protected Object visit(ASTIdentifier node, Object data) {
      String name = node.getName();
      return "$jexl".equals(name) ? this.writer : super.visit(node, data);
   }
}
