package org.apache.commons.jexl3.internal;

import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.jexl3.JexlContext;
import org.apache.commons.jexl3.JexlInfo;
import org.apache.commons.jexl3.JxltEngine;
import org.apache.commons.jexl3.parser.ASTJexlScript;

public final class TemplateScript implements JxltEngine.Template {
   private final String prefix;
   private final TemplateEngine.Block[] source;
   private final ASTJexlScript script;
   private final TemplateEngine.TemplateExpression[] exprs;
   private final TemplateEngine jxlt;

   public TemplateScript(TemplateEngine engine, JexlInfo info, String directive, Reader reader, String... parms) {
      if (directive == null) {
         throw new NullPointerException("null prefix");
      } else if (!Character.toString(engine.getImmediateChar()).equals(directive) && !(Character.toString(engine.getImmediateChar()) + "{").equals(directive) && !Character.toString(engine.getDeferredChar()).equals(directive) && !(Character.toString(engine.getDeferredChar()) + "{").equals(directive)) {
         if (reader == null) {
            throw new NullPointerException("null input");
         } else {
            this.jxlt = engine;
            Scope scope = parms == null ? null : new Scope((Scope)null, parms);
            this.prefix = directive;
            List<TemplateEngine.Block> blocks = this.jxlt.readTemplate(this.prefix, reader);
            List<TemplateEngine.TemplateExpression> uexprs = new ArrayList();
            StringBuilder strb = new StringBuilder();
            int nuexpr = 0;
            int codeStart = -1;

            for(int b = 0; b < blocks.size(); ++b) {
               TemplateEngine.Block block = (TemplateEngine.Block)blocks.get(b);
               if (block.getType() == TemplateEngine.BlockType.VERBATIM) {
                  strb.append("jexl:print(");
                  strb.append(nuexpr++);
                  strb.append(");\n");
               } else {
                  if (codeStart < 0) {
                     codeStart = b;
                  }

                  strb.append(block.getBody());
               }
            }

            if (info == null) {
               info = this.jxlt.getEngine().createInfo();
            }

            this.script = this.jxlt.getEngine().parse(info.at(0, 0), strb.toString(), scope, false, false).script();
            scope = this.script.getScope();

            for(int b = 0; b < blocks.size(); ++b) {
               TemplateEngine.Block block = (TemplateEngine.Block)blocks.get(b);
               if (block.getType() == TemplateEngine.BlockType.VERBATIM) {
                  uexprs.add(this.jxlt.parseExpression(info.at(block.getLine(), 0), block.getBody(), b > codeStart ? scope : null));
               }
            }

            this.source = (TemplateEngine.Block[])blocks.toArray(new TemplateEngine.Block[blocks.size()]);
            this.exprs = (TemplateEngine.TemplateExpression[])uexprs.toArray(new TemplateEngine.TemplateExpression[uexprs.size()]);
         }
      } else {
         throw new IllegalArgumentException(directive + ": is not a valid directive pattern");
      }
   }

   TemplateScript(TemplateEngine engine, String thePrefix, TemplateEngine.Block[] theSource, ASTJexlScript theScript, TemplateEngine.TemplateExpression[] theExprs) {
      this.jxlt = engine;
      this.prefix = thePrefix;
      this.source = theSource;
      this.script = theScript;
      this.exprs = theExprs;
   }

   ASTJexlScript getScript() {
      return this.script;
   }

   TemplateEngine.TemplateExpression[] getExpressions() {
      return this.exprs;
   }

   public String toString() {
      StringBuilder strb = new StringBuilder();

      for(TemplateEngine.Block block : this.source) {
         block.toString(strb, this.prefix);
      }

      return strb.toString();
   }

   public String asString() {
      StringBuilder strb = new StringBuilder();
      int e = 0;

      for(TemplateEngine.Block block : this.source) {
         if (block.getType() == TemplateEngine.BlockType.DIRECTIVE) {
            strb.append(this.prefix);
         } else {
            this.exprs[e++].asString(strb);
         }
      }

      return strb.toString();
   }

   public TemplateScript prepare(JexlContext context) {
      Scope.Frame frame = this.script.createFrame((Object[])null);
      TemplateEngine.TemplateExpression[] immediates = new TemplateEngine.TemplateExpression[this.exprs.length];

      for(int e = 0; e < this.exprs.length; ++e) {
         immediates[e] = this.exprs[e].prepare(frame, context);
      }

      return new TemplateScript(this.jxlt, this.prefix, this.source, this.script, immediates);
   }

   public void evaluate(JexlContext context, Writer writer) {
      this.evaluate(context, writer, (Object[])null);
   }

   public void evaluate(JexlContext context, Writer writer, Object... args) {
      Scope.Frame frame = this.script.createFrame(args);
      Interpreter interpreter = new TemplateInterpreter(this.jxlt.getEngine(), context, frame, this.exprs, writer);
      interpreter.interpret(this.script);
   }

   public Set<List<String>> getVariables() {
      Engine.VarCollector collector = new Engine.VarCollector();

      for(TemplateEngine.TemplateExpression expr : this.exprs) {
         expr.getVariables(collector);
      }

      return collector.collected();
   }

   public String[] getParameters() {
      return this.script.getParameters();
   }

   public Map<String, Object> getPragmas() {
      return this.script.getPragmas();
   }
}
