package org.apache.commons.jexl3.internal.introspection;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.commons.logging.Log;

final class ClassMap {
   private static final Method CACHE_MISS = cacheMiss();
   private final ConcurrentMap<MethodKey, Method> byKey = new ConcurrentHashMap();
   private final Map<String, Method[]> byName = new HashMap();
   private final Map<String, Field> fieldCache;

   public static Method cacheMiss() {
      try {
         return ClassMap.class.getMethod("cacheMiss");
      } catch (Exception var1) {
         return null;
      }
   }

   ClassMap(Class<?> aClass, Log log) {
      create(this, aClass, log);
      Field[] fields = aClass.getFields();
      if (fields.length > 0) {
         Map<String, Field> cache = new HashMap();

         for(Field field : fields) {
            if (Modifier.isPublic(field.getModifiers()) && Permissions.allow(field)) {
               cache.put(field.getName(), field);
            }
         }

         this.fieldCache = cache;
      } else {
         this.fieldCache = Collections.emptyMap();
      }

   }

   Field getField(String fname) {
      return (Field)this.fieldCache.get(fname);
   }

   String[] getFieldNames() {
      return (String[])this.fieldCache.keySet().toArray(new String[this.fieldCache.size()]);
   }

   String[] getMethodNames() {
      return (String[])this.byName.keySet().toArray(new String[this.byName.size()]);
   }

   Method[] getMethods(String methodName) {
      Method[] lm = (Method[])this.byName.get(methodName);
      return lm != null && lm.length > 0 ? (Method[])((Method[])lm).clone() : null;
   }

   Method getMethod(MethodKey methodKey) throws MethodKey.AmbiguousException {
      Method cacheEntry = (Method)this.byKey.get(methodKey);
      if (cacheEntry == CACHE_MISS) {
         return null;
      } else {
         if (cacheEntry == null) {
            try {
               Method[] methodList = (Method[])this.byName.get(methodKey.getMethod());
               if (methodList != null) {
                  cacheEntry = methodKey.getMostSpecificMethod(methodList);
               }

               if (cacheEntry == null) {
                  this.byKey.put(methodKey, CACHE_MISS);
               } else {
                  this.byKey.put(methodKey, cacheEntry);
               }
            } catch (MethodKey.AmbiguousException ae) {
               this.byKey.put(methodKey, CACHE_MISS);
               throw ae;
            }
         }

         return cacheEntry;
      }
   }

   private static void create(ClassMap cache, Class<?> classToReflect, Log log) {
      while(classToReflect != null) {
         if (Modifier.isPublic(classToReflect.getModifiers())) {
            populateWithClass(cache, classToReflect, log);
         }

         Class<?>[] interfaces = classToReflect.getInterfaces();

         for(int i = 0; i < interfaces.length; ++i) {
            populateWithInterface(cache, interfaces[i], log);
         }

         classToReflect = classToReflect.getSuperclass();
      }

      if (!cache.byKey.isEmpty()) {
         List<Method> lm = new ArrayList(cache.byKey.size());

         for(Method method : cache.byKey.values()) {
            lm.add(method);
         }

         Collections.sort(lm, new Comparator<Method>() {
            public int compare(Method o1, Method o2) {
               return o1.getName().compareTo(o2.getName());
            }
         });

         int end;
         for(int start = 0; start < lm.size(); start = end) {
            String name = ((Method)lm.get(start)).getName();

            for(end = start + 1; end < lm.size(); ++end) {
               String walk = ((Method)lm.get(end)).getName();
               if (!walk.equals(name)) {
                  break;
               }
            }

            Method[] lmn = (Method[])lm.subList(start, end).toArray(new Method[end - start]);
            cache.byName.put(name, lmn);
         }
      }

   }

   private static void populateWithInterface(ClassMap cache, Class<?> iface, Log log) {
      if (Modifier.isPublic(iface.getModifiers())) {
         populateWithClass(cache, iface, log);
      }

      Class<?>[] supers = iface.getInterfaces();

      for(int i = 0; i < supers.length; ++i) {
         populateWithInterface(cache, supers[i], log);
      }

   }

   private static void populateWithClass(ClassMap cache, Class<?> clazz, Log log) {
      try {
         Method[] methods = clazz.getDeclaredMethods();

         for(int i = 0; i < methods.length; ++i) {
            Method mi = methods[i];
            if (Modifier.isPublic(mi.getModifiers()) && Permissions.allow(mi)) {
               cache.byKey.putIfAbsent(new MethodKey(mi), mi);
            }
         }
      } catch (SecurityException se) {
         if (log.isDebugEnabled()) {
            log.debug("While accessing methods of " + clazz + ": ", se);
         }
      }

   }
}
