package org.apache.commons.jexl3.internal.introspection;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.logging.Log;

public final class Introspector {
   private static final Constructor<?> CTOR_MISS = CacheMiss.class.getConstructors()[0];
   protected final Log rlog;
   private ClassLoader loader;
   private final ReadWriteLock lock = new ReentrantReadWriteLock();
   private final Map<Class<?>, ClassMap> classMethodMaps = new HashMap();
   private final Map<MethodKey, Constructor<?>> constructorsMap = new HashMap();
   private final Map<String, Class<?>> constructibleClasses = new HashMap();

   public Introspector(Log log, ClassLoader cloader) {
      this.rlog = log;
      this.loader = cloader;
   }

   public Class<?> getClassByName(String className) {
      try {
         return Class.forName(className, false, this.loader);
      } catch (ClassNotFoundException var3) {
         return null;
      }
   }

   public Method getMethod(Class<?> c, String name, Object[] params) {
      return this.getMethod(c, new MethodKey(name, params));
   }

   public Method getMethod(Class<?> c, MethodKey key) {
      try {
         return this.getMap(c).getMethod(key);
      } catch (MethodKey.AmbiguousException xambiguous) {
         if (this.rlog != null && this.rlog.isInfoEnabled()) {
            this.rlog.info("ambiguous method invocation: " + c.getName() + "." + key.debugString(), xambiguous);
         }

         return null;
      }
   }

   public Field getField(Class<?> c, String key) {
      return this.getMap(c).getField(key);
   }

   public String[] getFieldNames(Class<?> c) {
      if (c == null) {
         return new String[0];
      } else {
         ClassMap classMap = this.getMap(c);
         return classMap.getFieldNames();
      }
   }

   public String[] getMethodNames(Class<?> c) {
      if (c == null) {
         return new String[0];
      } else {
         ClassMap classMap = this.getMap(c);
         return classMap.getMethodNames();
      }
   }

   public Method[] getMethods(Class<?> c, String methodName) {
      if (c == null) {
         return null;
      } else {
         ClassMap classMap = this.getMap(c);
         return classMap.getMethods(methodName);
      }
   }

   public Constructor<?> getConstructor(MethodKey key) {
      return this.getConstructor((Class)null, key);
   }

   public Constructor<?> getConstructor(Class<?> c, MethodKey key) {
      try {
         this.lock.readLock().lock();
         Constructor<?> ctor = (Constructor)this.constructorsMap.get(key);
         if (ctor != null) {
            Constructor var4 = CTOR_MISS.equals(ctor) ? null : ctor;
            return var4;
         }
      } finally {
         this.lock.readLock().unlock();
      }

      Constructor var26;
      try {
         this.lock.writeLock().lock();
         Constructor<?> var24 = (Constructor)this.constructorsMap.get(key);
         if (var24 == null) {
            String cname = key.getMethod();
            Class<?> clazz = (Class)this.constructibleClasses.get(cname);

            try {
               if (clazz == null) {
                  if (c != null && c.getName().equals(key.getMethod())) {
                     clazz = c;
                  } else {
                     clazz = this.loader.loadClass(cname);
                  }

                  this.constructibleClasses.put(cname, clazz);
               }

               List<Constructor<?>> l = new ArrayList();

               for(Constructor<?> ictor : clazz.getConstructors()) {
                  if (Modifier.isPublic(ictor.getModifiers()) && Permissions.allow(ictor)) {
                     l.add(ictor);
                  }
               }

               var24 = key.getMostSpecificConstructor((Constructor[])l.toArray(new Constructor[l.size()]));
               if (var24 != null) {
                  this.constructorsMap.put(key, var24);
               } else {
                  this.constructorsMap.put(key, CTOR_MISS);
               }
            } catch (ClassNotFoundException xnotfound) {
               if (this.rlog != null && this.rlog.isInfoEnabled()) {
                  this.rlog.info("unable to find class: " + cname + "." + key.debugString(), xnotfound);
               }

               var24 = null;
            } catch (MethodKey.AmbiguousException xambiguous) {
               if (this.rlog != null && this.rlog.isInfoEnabled()) {
                  this.rlog.info("ambiguous constructor invocation: " + cname + "." + key.debugString(), xambiguous);
               }

               var24 = null;
            }

            Constructor xnotfound = var24;
            return xnotfound;
         }

         var26 = CTOR_MISS.equals(var24) ? null : var24;
      } finally {
         this.lock.writeLock().unlock();
      }

      return var26;
   }

   private ClassMap getMap(Class<?> c) {
      ClassMap classMap;
      try {
         this.lock.readLock().lock();
         classMap = (ClassMap)this.classMethodMaps.get(c);
      } finally {
         this.lock.readLock().unlock();
      }

      if (classMap == null) {
         try {
            this.lock.writeLock().lock();
            classMap = (ClassMap)this.classMethodMaps.get(c);
            if (classMap == null) {
               classMap = new ClassMap(c, this.rlog);
               this.classMethodMaps.put(c, classMap);
            }
         } finally {
            this.lock.writeLock().unlock();
         }
      }

      return classMap;
   }

   public void setLoader(ClassLoader cloader) {
      ClassLoader previous = this.loader;
      if (cloader == null) {
         cloader = this.getClass().getClassLoader();
      }

      if (!cloader.equals(this.loader)) {
         try {
            this.lock.writeLock().lock();
            Iterator<Map.Entry<MethodKey, Constructor<?>>> mentries = this.constructorsMap.entrySet().iterator();

            while(mentries.hasNext()) {
               Map.Entry<MethodKey, Constructor<?>> entry = (Map.Entry)mentries.next();
               Class<?> clazz = ((Constructor)entry.getValue()).getDeclaringClass();
               if (isLoadedBy(previous, clazz)) {
                  mentries.remove();
                  this.constructibleClasses.remove(((MethodKey)entry.getKey()).getMethod());
               }
            }

            Iterator<Map.Entry<Class<?>, ClassMap>> centries = this.classMethodMaps.entrySet().iterator();

            while(centries.hasNext()) {
               Map.Entry<Class<?>, ClassMap> entry = (Map.Entry)centries.next();
               Class<?> clazz = (Class)entry.getKey();
               if (isLoadedBy(previous, clazz)) {
                  centries.remove();
               }
            }

            this.loader = cloader;
         } finally {
            this.lock.writeLock().unlock();
         }
      }

   }

   public ClassLoader getLoader() {
      return this.loader;
   }

   private static boolean isLoadedBy(ClassLoader loader, Class<?> clazz) {
      if (loader != null) {
         for(ClassLoader cloader = clazz.getClassLoader(); cloader != null; cloader = cloader.getParent()) {
            if (cloader.equals(loader)) {
               return true;
            }
         }
      }

      return false;
   }

   private static class CacheMiss {
      public CacheMiss() {
      }
   }
}
