package org.apache.commons.jexl3.internal.introspection;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;

public final class MethodKey {
   private static final int PRIMITIVE_SIZE = 13;
   private static final Map<Class<?>, Class<?>> PRIMITIVE_TYPES = new HashMap(13);
   private final int hashCode;
   private final String method;
   private final Class<?>[] params;
   private static final Class<?>[] NOARGS;
   private static final int HASH = 37;
   private static final int MORE_SPECIFIC = 0;
   private static final int LESS_SPECIFIC = 1;
   private static final int INCOMPARABLE = 2;
   private static final Parameters<Method> METHODS;
   private static final Parameters<Constructor<?>> CONSTRUCTORS;

   static Class<?> primitiveClass(Class<?> parm) {
      Class<?> prim = (Class)PRIMITIVE_TYPES.get(parm);
      return prim == null ? parm : prim;
   }

   public MethodKey(String aMethod, Object[] args) {
      this.method = aMethod;
      int hash = this.method.hashCode();
      int size;
      if (args != null && (size = args.length) > 0) {
         this.params = new Class[size];

         for(int p = 0; p < size; ++p) {
            Object arg = args[p];
            Class<?> parm = arg == null ? Void.class : arg.getClass();
            hash = 37 * hash + parm.hashCode();
            this.params[p] = parm;
         }
      } else {
         this.params = NOARGS;
      }

      this.hashCode = hash;
   }

   MethodKey(Method aMethod) {
      this(aMethod.getName(), aMethod.getParameterTypes());
   }

   MethodKey(Constructor<?> aCtor) {
      this(aCtor.getDeclaringClass().getName(), aCtor.getParameterTypes());
   }

   MethodKey(String aMethod, Class<?>[] args) {
      this.method = aMethod.intern();
      int hash = this.method.hashCode();
      int size;
      if (args != null && (size = args.length) > 0) {
         this.params = new Class[size];

         for(int p = 0; p < size; ++p) {
            Class<?> parm = primitiveClass(args[p]);
            hash = 37 * hash + parm.hashCode();
            this.params[p] = parm;
         }
      } else {
         this.params = NOARGS;
      }

      this.hashCode = hash;
   }

   String getMethod() {
      return this.method;
   }

   Class<?>[] getParameters() {
      return this.params;
   }

   public int hashCode() {
      return this.hashCode;
   }

   public boolean equals(Object obj) {
      if (!(obj instanceof MethodKey)) {
         return false;
      } else {
         MethodKey key = (MethodKey)obj;
         return this.method.equals(key.method) && Arrays.equals(this.params, key.params);
      }
   }

   public String toString() {
      StringBuilder builder = new StringBuilder(this.method);

      for(Class<?> c : this.params) {
         builder.append(c == Void.class ? "null" : c.getName());
      }

      return builder.toString();
   }

   public String debugString() {
      StringBuilder builder = new StringBuilder(this.method);
      builder.append('(');

      for(int i = 0; i < this.params.length; ++i) {
         if (i > 0) {
            builder.append(", ");
         }

         builder.append(Void.class == this.params[i] ? "null" : this.params[i].getName());
      }

      builder.append(')');
      return builder.toString();
   }

   public static boolean isVarArgs(Method method) {
      if (method == null) {
         return false;
      } else if (method.isVarArgs()) {
         return true;
      } else {
         Class<?>[] ptypes = method.getParameterTypes();
         if (ptypes.length > 0 && ptypes[ptypes.length - 1].getComponentType() == null) {
            return false;
         } else {
            String mname = method.getName();
            Class<?> clazz = method.getDeclaringClass();

            do {
               try {
                  Method m = clazz.getMethod(mname, ptypes);
                  if (m.isVarArgs()) {
                     return true;
                  }
               } catch (NoSuchMethodException var5) {
               }

               clazz = clazz.getSuperclass();
            } while(clazz != null);

            return false;
         }
      }
   }

   public Method getMostSpecificMethod(Method[] methods) {
      return METHODS.getMostSpecific(methods, this.params);
   }

   public Constructor<?> getMostSpecificConstructor(Constructor<?>[] methods) {
      return CONSTRUCTORS.getMostSpecific(methods, this.params);
   }

   public static boolean isInvocationConvertible(Class<?> formal, Class<?> actual, boolean possibleVarArg) {
      if (actual == null && !formal.isPrimitive()) {
         return true;
      } else if (actual != null && formal.isAssignableFrom(actual)) {
         return true;
      } else if (formal == Object.class) {
         return true;
      } else {
         if (formal.isPrimitive()) {
            if (formal == Boolean.TYPE && actual == Boolean.class) {
               return true;
            }

            if (formal == Character.TYPE && actual == Character.class) {
               return true;
            }

            if (formal == Byte.TYPE && actual == Byte.class) {
               return true;
            }

            if (formal == Short.TYPE && (actual == Short.class || actual == Byte.class)) {
               return true;
            }

            if (formal == Integer.TYPE && (actual == Integer.class || actual == Short.class || actual == Byte.class)) {
               return true;
            }

            if (formal == Long.TYPE && (actual == Long.class || actual == Integer.class || actual == Short.class || actual == Byte.class)) {
               return true;
            }

            if (formal == Float.TYPE && (actual == Float.class || actual == Long.class || actual == Integer.class || actual == Short.class || actual == Byte.class)) {
               return true;
            }

            if (formal == Double.TYPE && (actual == Double.class || actual == Float.class || actual == Long.class || actual == Integer.class || actual == Short.class || actual == Byte.class)) {
               return true;
            }
         }

         if (possibleVarArg && formal.isArray()) {
            if (actual != null && actual.isArray()) {
               actual = actual.getComponentType();
            }

            return isInvocationConvertible(formal.getComponentType(), actual, false);
         } else {
            return false;
         }
      }
   }

   public static boolean isStrictInvocationConvertible(Class<?> formal, Class<?> actual, boolean possibleVarArg) {
      if (actual == null && !formal.isPrimitive()) {
         return true;
      } else if (formal.isAssignableFrom(actual) && actual != null && formal.isArray() == actual.isArray()) {
         return true;
      } else {
         if (formal.isPrimitive()) {
            if (formal == Short.TYPE && actual == Byte.TYPE) {
               return true;
            }

            if (formal == Integer.TYPE && (actual == Short.TYPE || actual == Byte.TYPE)) {
               return true;
            }

            if (formal == Long.TYPE && (actual == Integer.TYPE || actual == Short.TYPE || actual == Byte.TYPE)) {
               return true;
            }

            if (formal == Float.TYPE && (actual == Long.TYPE || actual == Integer.TYPE || actual == Short.TYPE || actual == Byte.TYPE)) {
               return true;
            }

            if (formal == Double.TYPE && (actual == Float.TYPE || actual == Long.TYPE || actual == Integer.TYPE || actual == Short.TYPE || actual == Byte.TYPE)) {
               return true;
            }
         }

         if (possibleVarArg && formal.isArray()) {
            if (actual != null && actual.isArray()) {
               actual = actual.getComponentType();
            }

            return isStrictInvocationConvertible(formal.getComponentType(), actual, false);
         } else {
            return false;
         }
      }
   }

   static {
      PRIMITIVE_TYPES.put(Boolean.TYPE, Boolean.class);
      PRIMITIVE_TYPES.put(Byte.TYPE, Byte.class);
      PRIMITIVE_TYPES.put(Character.TYPE, Character.class);
      PRIMITIVE_TYPES.put(Double.TYPE, Double.class);
      PRIMITIVE_TYPES.put(Float.TYPE, Float.class);
      PRIMITIVE_TYPES.put(Integer.TYPE, Integer.class);
      PRIMITIVE_TYPES.put(Long.TYPE, Long.class);
      PRIMITIVE_TYPES.put(Short.TYPE, Short.class);
      NOARGS = new Class[0];
      METHODS = new Parameters<Method>() {
         protected Class<?>[] getParameterTypes(Method app) {
            return app.getParameterTypes();
         }

         public boolean isVarArgs(Method app) {
            return MethodKey.isVarArgs(app);
         }
      };
      CONSTRUCTORS = new Parameters<Constructor<?>>() {
         protected Class<?>[] getParameterTypes(Constructor<?> app) {
            return app.getParameterTypes();
         }

         public boolean isVarArgs(Constructor<?> app) {
            return app.isVarArgs();
         }
      };
   }

   public static class AmbiguousException extends RuntimeException {
      private static final long serialVersionUID = -2314636505414551664L;
   }

   private abstract static class Parameters<T> {
      private Parameters() {
      }

      protected abstract Class<?>[] getParameterTypes(T var1);

      protected abstract boolean isVarArgs(T var1);

      private T getMostSpecific(T[] methods, Class<?>[] classes) {
         LinkedList<T> applicables = this.getApplicables(methods, classes);
         if (applicables.isEmpty()) {
            return null;
         } else if (applicables.size() == 1) {
            return (T)applicables.getFirst();
         } else {
            LinkedList<T> maximals = new LinkedList();

            for(T app : applicables) {
               Class<?>[] appArgs = this.getParameterTypes(app);
               boolean lessSpecific = false;
               Iterator<T> maximal = maximals.iterator();

               while(!lessSpecific && maximal.hasNext()) {
                  T max = (T)maximal.next();
                  switch (this.moreSpecific(appArgs, this.getParameterTypes(max))) {
                     case 0:
                        maximal.remove();
                        break;
                     case 1:
                        lessSpecific = true;
                  }
               }

               if (!lessSpecific) {
                  maximals.addLast(app);
               }
            }

            if (maximals.size() > 1) {
               throw new AmbiguousException();
            } else {
               return (T)maximals.getFirst();
            }
         }
      }

      private int moreSpecific(Class<?>[] c1, Class<?>[] c2) {
         boolean c1MoreSpecific = false;
         boolean c2MoreSpecific = false;
         if (c1.length > c2.length) {
            return 0;
         } else if (c2.length > c1.length) {
            return 1;
         } else {
            int length = c1.length;
            int ultimate = c1.length - 1;

            for(int i = 0; i < length; ++i) {
               if (c1[i] != c2[i]) {
                  boolean last = i == ultimate;
                  c1MoreSpecific = c1MoreSpecific || this.isStrictConvertible(c2[i], c1[i], last);
                  c2MoreSpecific = c2MoreSpecific || this.isStrictConvertible(c1[i], c2[i], last);
               }
            }

            if (c1MoreSpecific) {
               if (c2MoreSpecific) {
                  return 2;
               } else {
                  return 0;
               }
            } else if (c2MoreSpecific) {
               return 1;
            } else {
               int primDiff = 0;

               for(int c = 0; c < length; ++c) {
                  boolean last = c == ultimate;
                  if (this.isPrimitive(c1[c], last)) {
                     primDiff += 1 << c;
                  }

                  if (this.isPrimitive(c2[c], last)) {
                     primDiff -= 1 << c;
                  }
               }

               if (primDiff > 0) {
                  return 0;
               } else if (primDiff < 0) {
                  return 1;
               } else {
                  return 2;
               }
            }
         }
      }

      private boolean isPrimitive(Class<?> c, boolean possibleVarArg) {
         if (c != null) {
            if (c.isPrimitive()) {
               return true;
            }

            if (possibleVarArg) {
               Class<?> t = c.getComponentType();
               return t != null && t.isPrimitive();
            }
         }

         return false;
      }

      private LinkedList<T> getApplicables(T[] methods, Class<?>[] classes) {
         LinkedList<T> list = new LinkedList();

         for(T method : methods) {
            if (this.isApplicable(method, classes)) {
               list.add(method);
            }
         }

         return list;
      }

      private boolean isApplicable(T method, Class<?>[] actuals) {
         Class<?>[] formals = this.getParameterTypes(method);
         if (formals.length == actuals.length) {
            for(int i = 0; i < actuals.length; ++i) {
               if (!this.isConvertible(formals[i], actuals[i], false)) {
                  if (i == actuals.length - 1 && formals[i].isArray()) {
                     return this.isConvertible(formals[i], actuals[i], true);
                  }

                  return false;
               }
            }

            return true;
         } else if (!this.isVarArgs(method)) {
            return false;
         } else if (formals.length > actuals.length) {
            if (formals.length - actuals.length > 1) {
               return false;
            } else {
               for(int i = 0; i < actuals.length; ++i) {
                  if (!this.isConvertible(formals[i], actuals[i], false)) {
                     return false;
                  }
               }

               return true;
            }
         } else if (formals.length > 0 && actuals.length > 0) {
            for(int i = 0; i < formals.length - 1; ++i) {
               if (!this.isConvertible(formals[i], actuals[i], false)) {
                  return false;
               }
            }

            Class<?> vararg = formals[formals.length - 1].getComponentType();

            for(int i = formals.length - 1; i < actuals.length; ++i) {
               if (!this.isConvertible(vararg, actuals[i], false)) {
                  return false;
               }
            }

            return true;
         } else {
            return false;
         }
      }

      private boolean isConvertible(Class<?> formal, Class<?> actual, boolean possibleVarArg) {
         return MethodKey.isInvocationConvertible(formal, actual.equals(Void.class) ? null : actual, possibleVarArg);
      }

      private boolean isStrictConvertible(Class<?> formal, Class<?> actual, boolean possibleVarArg) {
         return MethodKey.isStrictInvocationConvertible(formal, actual.equals(Void.class) ? null : actual, possibleVarArg);
      }
   }
}
