package org.apache.commons.jexl3.internal.introspection;

import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.jexl3.JexlArithmetic;
import org.apache.commons.jexl3.JexlEngine;
import org.apache.commons.jexl3.JexlOperator;
import org.apache.commons.jexl3.introspection.JexlMethod;
import org.apache.commons.jexl3.introspection.JexlPropertyGet;
import org.apache.commons.jexl3.introspection.JexlPropertySet;
import org.apache.commons.jexl3.introspection.JexlUberspect;
import org.apache.commons.logging.Log;

public class Uberspect implements JexlUberspect {
   public static final Object TRY_FAILED;
   protected final Log rlog;
   private final ResolverStrategy strategy;
   private final AtomicInteger version;
   private volatile Reference<Introspector> ref;
   private volatile Reference<ClassLoader> loader;
   private final Map<Class<? extends JexlArithmetic>, Set<JexlOperator>> operatorMap;

   public Uberspect(Log runtimeLogger, ResolverStrategy sty) {
      this.rlog = runtimeLogger;
      this.strategy = sty == null ? JexlUberspect.JEXL_STRATEGY : sty;
      this.ref = new SoftReference(null);
      this.loader = new SoftReference(this.getClass().getClassLoader());
      this.operatorMap = new ConcurrentHashMap();
      this.version = new AtomicInteger(0);
   }

   protected final Introspector base() {
      Introspector intro = (Introspector)this.ref.get();
      if (intro == null) {
         synchronized(this) {
            intro = (Introspector)this.ref.get();
            if (intro == null) {
               intro = new Introspector(this.rlog, (ClassLoader)this.loader.get());
               this.ref = new SoftReference(intro);
               this.loader = new SoftReference(intro.getLoader());
               this.version.incrementAndGet();
            }
         }
      }

      return intro;
   }

   public void setClassLoader(ClassLoader nloader) {
      synchronized(this) {
         Introspector intro = (Introspector)this.ref.get();
         if (intro != null) {
            intro.setLoader(nloader);
         } else {
            intro = new Introspector(this.rlog, nloader);
            this.ref = new SoftReference(intro);
         }

         this.loader = new SoftReference(intro.getLoader());
         this.operatorMap.clear();
         this.version.incrementAndGet();
      }
   }

   public int getVersion() {
      return this.version.intValue();
   }

   public final Class<?> getClassByName(String className) {
      return this.base().getClassByName(className);
   }

   public final Field getField(Class<?> c, String key) {
      return this.base().getField(c, key);
   }

   public final String[] getFieldNames(Class<?> c) {
      return this.base().getFieldNames(c);
   }

   public final Method getMethod(Class<?> c, String name, Object[] params) {
      return this.base().getMethod(c, new MethodKey(name, params));
   }

   public final Method getMethod(Class<?> c, MethodKey key) {
      return this.base().getMethod(c, key);
   }

   public final String[] getMethodNames(Class<?> c) {
      return this.base().getMethodNames(c);
   }

   public final Method[] getMethods(Class<?> c, String methodName) {
      return this.base().getMethods(c, methodName);
   }

   public JexlMethod getMethod(Object obj, String method, Object... args) {
      return MethodExecutor.discover(this.base(), obj, method, args);
   }

   public List<PropertyResolver> getResolvers(JexlOperator op, Object obj) {
      return this.strategy.apply(op, obj);
   }

   public JexlPropertyGet getPropertyGet(Object obj, Object identifier) {
      return this.getPropertyGet((List)null, obj, identifier);
   }

   public JexlPropertyGet getPropertyGet(List<PropertyResolver> resolvers, Object obj, Object identifier) {
      Class<?> claz = obj.getClass();
      String property = AbstractExecutor.castString(identifier);
      Introspector is = this.base();
      List<PropertyResolver> r = resolvers == null ? this.strategy.apply((JexlOperator)null, obj) : resolvers;
      JexlPropertyGet executor = null;

      for(PropertyResolver resolver : r) {
         if (resolver instanceof JexlResolver) {
            switch ((JexlResolver)resolver) {
               case PROPERTY:
                  executor = PropertyGetExecutor.discover(is, claz, property);
                  if (executor == null) {
                     executor = BooleanGetExecutor.discover(is, claz, property);
                  }
                  break;
               case MAP:
                  executor = MapGetExecutor.discover(is, claz, identifier);
                  break;
               case LIST:
                  Integer index = AbstractExecutor.castInteger(identifier);
                  if (index != null) {
                     executor = ListGetExecutor.discover(is, claz, index);
                  }
                  break;
               case DUCK:
                  executor = DuckGetExecutor.discover(is, claz, identifier);
                  if (executor == null && property != null && property != identifier) {
                     executor = DuckGetExecutor.discover(is, claz, property);
                  }
                  break;
               case FIELD:
                  executor = FieldGetExecutor.discover(is, claz, property);
                  if (obj instanceof Class) {
                     executor = FieldGetExecutor.discover(is, (Class)obj, property);
                  }
                  break;
               case CONTAINER:
                  executor = IndexedType.discover(is, obj, property);
                  break;
               default:
                  continue;
            }
         } else {
            executor = resolver.getPropertyGet(this, obj, identifier);
         }

         if (executor != null) {
            return executor;
         }
      }

      return null;
   }

   public JexlPropertySet getPropertySet(Object obj, Object identifier, Object arg) {
      return this.getPropertySet((List)null, obj, identifier, arg);
   }

   public JexlPropertySet getPropertySet(List<PropertyResolver> resolvers, Object obj, Object identifier, Object arg) {
      Class<?> claz = obj.getClass();
      String property = AbstractExecutor.castString(identifier);
      Introspector is = this.base();
      List<PropertyResolver> actual = resolvers == null ? this.strategy.apply((JexlOperator)null, obj) : resolvers;
      JexlPropertySet executor = null;

      for(PropertyResolver resolver : actual) {
         if (resolver instanceof JexlResolver) {
            switch ((JexlResolver)resolver) {
               case PROPERTY:
                  executor = PropertySetExecutor.discover(is, claz, property, arg);
                  break;
               case MAP:
                  executor = MapSetExecutor.discover(is, claz, identifier, arg);
                  break;
               case LIST:
                  Integer index = AbstractExecutor.castInteger(identifier);
                  if (index != null) {
                     executor = ListSetExecutor.discover(is, claz, identifier, arg);
                  }
                  break;
               case DUCK:
                  executor = DuckSetExecutor.discover(is, claz, identifier, arg);
                  if (executor == null && property != null && property != identifier) {
                     executor = DuckSetExecutor.discover(is, claz, property, arg);
                  }
                  break;
               case FIELD:
                  executor = FieldSetExecutor.discover(is, claz, property, arg);
                  break;
               case CONTAINER:
               default:
                  continue;
            }
         } else {
            executor = resolver.getPropertySet(this, obj, identifier, arg);
         }

         if (executor != null) {
            return executor;
         }
      }

      return null;
   }

   public Iterator<?> getIterator(Object obj) {
      if (obj instanceof Iterator) {
         return (Iterator)obj;
      } else if (obj.getClass().isArray()) {
         return new ArrayIterator(obj);
      } else if (obj instanceof Map) {
         return ((Map)obj).values().iterator();
      } else if (obj instanceof Enumeration) {
         return new EnumerationIterator((Enumeration)obj);
      } else if (obj instanceof Iterable) {
         return ((Iterable)obj).iterator();
      } else {
         try {
            JexlMethod it = this.getMethod(obj, "iterator", (Object[])null);
            if (it != null && Iterator.class.isAssignableFrom(it.getReturnType())) {
               return (Iterator)it.invoke(obj, (Object[])null);
            }
         } catch (Exception xany) {
            if (this.rlog != null && this.rlog.isDebugEnabled()) {
               this.rlog.info("unable to solve iterator()", xany);
            }
         }

         return null;
      }
   }

   public JexlMethod getConstructor(Object ctorHandle, Object... args) {
      return ConstructorMethod.discover(this.base(), ctorHandle, args);
   }

   public JexlArithmetic.Uberspect getArithmetic(JexlArithmetic arithmetic) {
      JexlArithmetic.Uberspect jau = null;
      if (arithmetic != null) {
         Class<? extends JexlArithmetic> aclass = arithmetic.getClass();
         Set<JexlOperator> ops = (Set)this.operatorMap.get(aclass);
         if (ops == null) {
            ops = EnumSet.noneOf(JexlOperator.class);

            for(JexlOperator op : JexlOperator.values()) {
               Method[] methods = this.getMethods(arithmetic.getClass(), op.getMethodName());
               if (methods != null) {
                  for(Method method : methods) {
                     Class<?>[] parms = method.getParameterTypes();
                     if (parms.length == op.getArity()) {
                        boolean root = true;

                        for(int p = 0; root && p < parms.length; ++p) {
                           if (!Object.class.equals(parms[p])) {
                              root = false;
                           }
                        }

                        if (!root) {
                           ops.add(op);
                        }
                     }
                  }
               }
            }

            this.operatorMap.put(aclass, ops);
         }

         jau = new ArithmeticUberspect(arithmetic, ops);
      }

      return jau;
   }

   static {
      TRY_FAILED = JexlEngine.TRY_FAILED;
   }

   protected class ArithmeticUberspect implements JexlArithmetic.Uberspect {
      private final JexlArithmetic arithmetic;
      private final Set<JexlOperator> overloads;

      private ArithmeticUberspect(JexlArithmetic theArithmetic, Set<JexlOperator> theOverloads) {
         this.arithmetic = theArithmetic;
         this.overloads = theOverloads;
      }

      public JexlMethod getOperator(JexlOperator operator, Object... args) {
         return this.overloads.contains(operator) && args != null ? Uberspect.this.getMethod(this.arithmetic, operator.getMethodName(), args) : null;
      }

      public boolean overloads(JexlOperator operator) {
         return this.overloads.contains(operator);
      }
   }
}
