package org.apache.commons.jexl3.introspection;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public final class JexlSandbox {
   private final Map<String, Permissions> sandbox;
   private final boolean white;
   private static final Names WHITE_NAMES = new Names() {
      public boolean add(String name) {
         return false;
      }

      protected Names copy() {
         return this;
      }
   };
   private static final Permissions ALL_WHITE;

   public JexlSandbox() {
      this(true, new HashMap());
   }

   public JexlSandbox(boolean wb) {
      this(wb, new HashMap());
   }

   protected JexlSandbox(Map<String, Permissions> map) {
      this(true, map);
   }

   protected JexlSandbox(boolean wb, Map<String, Permissions> map) {
      this.white = wb;
      this.sandbox = map;
   }

   public JexlSandbox copy() {
      Map<String, Permissions> map = new HashMap();

      for(Map.Entry<String, Permissions> entry : this.sandbox.entrySet()) {
         map.put(entry.getKey(), ((Permissions)entry.getValue()).copy());
      }

      return new JexlSandbox(this.white, map);
   }

   public String read(Class<?> clazz, String name) {
      return this.read(clazz.getName(), name);
   }

   public String read(String clazz, String name) {
      Permissions permissions = (Permissions)this.sandbox.get(clazz);
      if (permissions == null) {
         return this.white ? name : null;
      } else {
         return permissions.read().get(name);
      }
   }

   public String write(Class<?> clazz, String name) {
      return this.write(clazz.getName(), name);
   }

   public String write(String clazz, String name) {
      Permissions permissions = (Permissions)this.sandbox.get(clazz);
      if (permissions == null) {
         return this.white ? name : null;
      } else {
         return permissions.write().get(name);
      }
   }

   public String execute(Class<?> clazz, String name) {
      return this.execute(clazz.getName(), name);
   }

   public String execute(String clazz, String name) {
      Permissions permissions = (Permissions)this.sandbox.get(clazz);
      if (permissions == null) {
         return this.white ? name : null;
      } else {
         return permissions.execute().get(name);
      }
   }

   public Permissions permissions(String clazz, boolean readFlag, boolean writeFlag, boolean executeFlag) {
      Permissions box = new Permissions(readFlag, writeFlag, executeFlag);
      this.sandbox.put(clazz, box);
      return box;
   }

   public Permissions white(String clazz) {
      return this.permissions(clazz, true, true, true);
   }

   public Permissions black(String clazz) {
      return this.permissions(clazz, false, false, false);
   }

   public Permissions get(String clazz) {
      Permissions permissions = (Permissions)this.sandbox.get(clazz);
      return permissions == null ? ALL_WHITE : permissions;
   }

   static {
      ALL_WHITE = new Permissions(WHITE_NAMES, WHITE_NAMES, WHITE_NAMES);
   }

   public abstract static class Names {
      public abstract boolean add(String var1);

      public boolean alias(String name, String alias) {
         return false;
      }

      public String get(String name) {
         return name;
      }

      protected Names copy() {
         return this;
      }
   }

   public static final class WhiteSet extends Names {
      private Map<String, String> names = null;

      protected Names copy() {
         WhiteSet copy = new WhiteSet();
         copy.names = this.names == null ? null : new HashMap(this.names);
         return copy;
      }

      public boolean add(String name) {
         if (this.names == null) {
            this.names = new HashMap();
         }

         return this.names.put(name, name) == null;
      }

      public boolean alias(String name, String alias) {
         if (this.names == null) {
            this.names = new HashMap();
         }

         return this.names.put(alias, name) == null;
      }

      public String get(String name) {
         return this.names == null ? name : (String)this.names.get(name);
      }
   }

   public static final class BlackSet extends Names {
      private Set<String> names = null;

      protected Names copy() {
         BlackSet copy = new BlackSet();
         copy.names = this.names == null ? null : new HashSet(this.names);
         return copy;
      }

      public boolean add(String name) {
         if (this.names == null) {
            this.names = new HashSet();
         }

         return this.names.add(name);
      }

      public String get(String name) {
         return this.names != null && !this.names.contains(name) ? name : null;
      }
   }

   public static final class Permissions {
      private final Names read;
      private final Names write;
      private final Names execute;

      Permissions(boolean readFlag, boolean writeFlag, boolean executeFlag) {
         this((Names)(readFlag ? new WhiteSet() : new BlackSet()), (Names)(writeFlag ? new WhiteSet() : new BlackSet()), (Names)(executeFlag ? new WhiteSet() : new BlackSet()));
      }

      Permissions(Names nread, Names nwrite, Names nexecute) {
         this.read = nread != null ? nread : JexlSandbox.WHITE_NAMES;
         this.write = nwrite != null ? nwrite : JexlSandbox.WHITE_NAMES;
         this.execute = nexecute != null ? nexecute : JexlSandbox.WHITE_NAMES;
      }

      Permissions copy() {
         return new Permissions(this.read.copy(), this.write.copy(), this.execute.copy());
      }

      public Permissions read(String... pnames) {
         for(String pname : pnames) {
            this.read.add(pname);
         }

         return this;
      }

      public Permissions write(String... pnames) {
         for(String pname : pnames) {
            this.write.add(pname);
         }

         return this;
      }

      public Permissions execute(String... mnames) {
         for(String mname : mnames) {
            this.execute.add(mname);
         }

         return this;
      }

      public Names read() {
         return this.read;
      }

      public Names write() {
         return this.write;
      }

      public Names execute() {
         return this.execute;
      }
   }
}
