package org.apache.commons.jexl3.parser;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;
import org.apache.commons.jexl3.JexlException;
import org.apache.commons.jexl3.JexlInfo;
import org.apache.commons.jexl3.internal.Scope;

public abstract class JexlParser extends StringParser {
   boolean ALLOW_REGISTERS = false;
   String source = null;
   Scope frame = null;
   Stack<Scope> frames = new Stack();
   Map<String, Object> pragmas = null;
   private static final Set<Class<? extends JexlNode>> ASSIGN_NODES = new HashSet(Arrays.asList(ASTAssignment.class, ASTSetAddNode.class, ASTSetMultNode.class, ASTSetDivNode.class, ASTSetAndNode.class, ASTSetOrNode.class, ASTSetXorNode.class, ASTSetSubNode.class));

   public void allowRegisters(boolean registers) {
      this.ALLOW_REGISTERS = registers;
   }

   public void setFrame(Scope theFrame) {
      this.frame = theFrame;
   }

   public Scope getFrame() {
      return this.frame;
   }

   public void pushFrame() {
      if (this.frame != null) {
         this.frames.push(this.frame);
      }

      this.frame = new Scope(this.frame, (String[])null);
   }

   public void popFrame() {
      if (!this.frames.isEmpty()) {
         this.frame = (Scope)this.frames.pop();
      } else {
         this.frame = null;
      }

   }

   public String checkVariable(ASTIdentifier identifier, String image) {
      if (this.frame != null) {
         Integer register = this.frame.getSymbol(image);
         if (register != null) {
            identifier.setSymbol(register, image);
         }
      }

      return image;
   }

   public void declareVariable(ASTVar identifier, String image) {
      if (this.frame == null) {
         this.frame = new Scope((Scope)null, (String[])null);
      }

      Integer register = this.frame.declareVariable(image);
      identifier.setSymbol(register, image);
   }

   public void declarePragma(String key, Object value) {
      if (this.pragmas == null) {
         this.pragmas = new TreeMap();
      }

      this.pragmas.put(key, value);
   }

   public void declareParameter(String identifier) {
      if (this.frame == null) {
         this.frame = new Scope((Scope)null, (String[])null);
      }

      this.frame.declareParameter(identifier);
   }

   public void Identifier(boolean top) throws ParseException {
   }

   public final void Identifier() throws ParseException {
      this.Identifier(false);
   }

   public Token getToken(int index) {
      return null;
   }

   void jjtreeOpenNodeScope(JexlNode node) {
   }

   void jjtreeCloseNodeScope(JexlNode node) throws ParseException {
      if (node instanceof ASTJexlScript) {
         ASTJexlScript script = (ASTJexlScript)node;
         if (script.getScope() != this.frame) {
            script.setScope(this.frame);
         }

         this.popFrame();
      } else if (node instanceof ASTAmbiguous) {
         this.throwParsingException(JexlException.Ambiguous.class, node);
      } else if (ASSIGN_NODES.contains(node.getClass())) {
         JexlNode lv = node.jjtGetChild(0);
         if (!lv.isLeftValue()) {
            this.throwParsingException(JexlException.Assignment.class, lv);
         }
      }

   }

   String stringify(List<String> lstr) {
      StringBuilder strb = new StringBuilder();
      boolean dot = false;

      for(String str : lstr) {
         if (!dot) {
            dot = true;
         } else {
            strb.append('.');
         }

         strb.append(str);
      }

      return strb.toString();
   }

   protected void throwParsingException(JexlNode node) {
      this.throwParsingException((Class)null, node);
   }

   private void throwParsingException(Class<? extends JexlException> xclazz, JexlNode node) {
      Token tok = this.getToken(0);
      JexlInfo dbgInfo;
      if (tok != null) {
         dbgInfo = new JexlInfo(tok.image, tok.beginLine, tok.beginColumn);
      } else {
         dbgInfo = node.jexlInfo();
      }

      String msg = null;

      try {
         if (this.source != null) {
            BufferedReader reader = new BufferedReader(new StringReader(this.source));

            for(int l = 0; l < dbgInfo.getLine(); ++l) {
               msg = reader.readLine();
            }
         } else {
            msg = "";
         }
      } catch (IOException var8) {
      }

      if (JexlException.Ambiguous.class.equals(xclazz)) {
         throw new JexlException.Ambiguous(dbgInfo, msg);
      } else if (JexlException.Assignment.class.equals(xclazz)) {
         throw new JexlException.Assignment(dbgInfo, msg);
      } else {
         throw new JexlException.Parsing(dbgInfo, msg);
      }
   }
}
