package org.apache.commons.jexl3.parser;

public class StringParser {
   private static final int UCHAR_LEN = 4;
   private static final int SHIFT = 12;
   private static final int BASE10 = 10;
   private static final char LAST_ASCII = '\u007f';
   private static final char FIRST_ASCII = ' ';

   public static String buildString(CharSequence str, boolean eatsep) {
      StringBuilder strb = new StringBuilder(str.length());
      char sep = eatsep ? str.charAt(0) : 0;
      int end = str.length() - (eatsep ? 1 : 0);
      int begin = eatsep ? 1 : 0;
      read(strb, str, begin, end, sep);
      return strb.toString();
   }

   public static int readString(StringBuilder strb, CharSequence str, int index, char sep) {
      return read(strb, str, index, str.length(), sep);
   }

   private static int read(StringBuilder strb, CharSequence str, int begin, int end, char sep) {
      boolean escape = false;

      int index;
      for(index = begin; index < end; ++index) {
         char c = str.charAt(index);
         if (escape) {
            if (c == 'u' && index + 4 < end && readUnicodeChar(strb, str, index + 1) > 0) {
               index += 4;
            } else {
               boolean notSeparator = sep == 0 ? c != '\'' && c != '"' : c != sep;
               if (notSeparator && c != '\\') {
                  strb.append('\\');
               }

               strb.append(c);
            }

            escape = false;
         } else if (c == '\\') {
            escape = true;
         } else {
            strb.append(c);
            if (c == sep) {
               break;
            }
         }
      }

      return index;
   }

   private static int readUnicodeChar(StringBuilder strb, CharSequence str, int begin) {
      char xc = 0;
      int bits = 12;
      int value = 0;

      for(int offset = 0; offset < 4; ++offset) {
         char c = str.charAt(begin + offset);
         if (c >= '0' && c <= '9') {
            value = c - 48;
         } else if (c >= 'a' && c <= 'h') {
            value = c - 97 + 10;
         } else {
            if (c < 'A' || c > 'H') {
               return 0;
            }

            value = c - 65 + 10;
         }

         xc = (char)(xc | value << bits);
         bits -= 4;
      }

      strb.append(xc);
      return 4;
   }

   public static String escapeString(String str, char delim) {
      if (str == null) {
         return null;
      } else {
         int length = str.length();
         StringBuilder strb = new StringBuilder(length + 2);
         strb.append(delim);

         for(int i = 0; i < length; ++i) {
            char c = str.charAt(i);
            String hex;
            int h;
            switch (c) {
               case '\u0000':
                  continue;
               case '\b':
                  strb.append("\\b");
                  continue;
               case '\t':
                  strb.append("\\t");
                  continue;
               case '\n':
                  strb.append("\\n");
                  continue;
               case '\f':
                  strb.append("\\f");
                  continue;
               case '\r':
                  strb.append("\\r");
                  continue;
               case '"':
                  strb.append("\\\"");
                  continue;
               case '\'':
                  strb.append("\\'");
                  continue;
               case '\\':
                  strb.append("\\\\");
                  continue;
               default:
                  if (c >= ' ' && c <= 127) {
                     strb.append(c);
                     continue;
                  }

                  strb.append('\\');
                  strb.append('u');
                  hex = Integer.toHexString(c);
                  h = hex.length();
            }

            while(h < 4) {
               strb.append('0');
               ++h;
            }

            strb.append(hex);
         }

         strb.append(delim);
         return strb.toString();
      }
   }
}
