package org.apache.http.client.protocol;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.annotation.Contract;
import org.apache.http.annotation.ThreadingBehavior;
import org.apache.http.client.CookieStore;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.config.Lookup;
import org.apache.http.conn.routing.RouteInfo;
import org.apache.http.cookie.Cookie;
import org.apache.http.cookie.CookieOrigin;
import org.apache.http.cookie.CookieSpec;
import org.apache.http.cookie.CookieSpecProvider;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.Args;
import org.apache.http.util.TextUtils;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Contract(
   threading = ThreadingBehavior.IMMUTABLE
)
public class RequestAddCookies implements HttpRequestInterceptor {
   public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
      Args.notNull(request, "HTTP request");
      Args.notNull(context, "HTTP context");
      String method = request.getRequestLine().getMethod();
      if (!method.equalsIgnoreCase("CONNECT")) {
         HttpClientContext clientContext = HttpClientContext.adapt(context);
         CookieStore cookieStore = clientContext.getCookieStore();
         if (cookieStore != null) {
            Lookup<CookieSpecProvider> registry = clientContext.getCookieSpecRegistry();
            if (registry != null) {
               HttpHost targetHost = clientContext.getTargetHost();
               if (targetHost != null) {
                  RouteInfo route = clientContext.getHttpRoute();
                  if (route != null) {
                     RequestConfig config = clientContext.getRequestConfig();
                     String policy = config.getCookieSpec();
                     if (policy == null) {
                        policy = "default";
                     }

                     URI requestURI = null;
                     if (request instanceof HttpUriRequest) {
                        requestURI = ((HttpUriRequest)request).getURI();
                     } else {
                        try {
                           requestURI = new URI(request.getRequestLine().getUri());
                        } catch (URISyntaxException var25) {
                        }
                     }

                     String path = requestURI != null ? requestURI.getPath() : null;
                     String hostName = targetHost.getHostName();
                     int port = targetHost.getPort();
                     if (port < 0) {
                        port = route.getTargetHost().getPort();
                     }

                     CookieOrigin cookieOrigin = new CookieOrigin(hostName, port >= 0 ? port : 0, !TextUtils.isEmpty(path) ? path : "/", route.isSecure());
                     CookieSpecProvider provider = registry.lookup(policy);
                     if (provider != null) {
                        CookieSpec cookieSpec = provider.create(clientContext);
                        List<Cookie> cookies = cookieStore.getCookies();
                        List<Cookie> matchedCookies = new ArrayList();
                        Date now = new Date();
                        boolean expired = false;

                        for(Cookie cookie : cookies) {
                           if (!cookie.isExpired(now)) {
                              if (cookieSpec.match(cookie, cookieOrigin)) {
                                 matchedCookies.add(cookie);
                              }
                           } else {
                              expired = true;
                           }
                        }

                        if (expired) {
                           cookieStore.clearExpired(now);
                        }

                        if (!matchedCookies.isEmpty()) {
                           for(Header header : cookieSpec.formatCookies(matchedCookies)) {
                              request.addHeader(header);
                           }
                        }

                        int ver = cookieSpec.getVersion();
                        if (ver > 0) {
                           Header header = cookieSpec.getVersionHeader();
                           if (header != null) {
                              request.addHeader(header);
                           }
                        }

                        context.setAttribute("http.cookie-spec", cookieSpec);
                        context.setAttribute("http.cookie-origin", cookieOrigin);
                     }
                  }
               }
            }
         }
      }
   }
}
