package com.chenyang.druid.sql.ast.statement;

import com.chenyang.druid.DbType;
import com.chenyang.druid.FastsqlException;
import com.chenyang.druid.sql.SQLUtils;
import com.chenyang.druid.sql.ast.SQLDataType;
import com.chenyang.druid.sql.ast.SQLExpr;
import com.chenyang.druid.sql.ast.SQLObjectImpl;
import com.chenyang.druid.sql.ast.SQLReplaceable;
import com.chenyang.druid.sql.ast.expr.SQLAllColumnExpr;
import com.chenyang.druid.sql.ast.expr.SQLIdentifierExpr;
import com.chenyang.druid.sql.ast.expr.SQLIntegerExpr;
import com.chenyang.druid.sql.ast.expr.SQLPropertyExpr;
import com.chenyang.druid.sql.dialect.dm.ast.DMSQLObject;
import com.chenyang.druid.sql.dialect.kingbase.KingbaseSQLObject;
import com.chenyang.druid.sql.dialect.oracle.ast.OracleSQLObject;
import com.chenyang.druid.sql.visitor.SQLASTVisitor;
import com.chenyang.druid.util.FnvHash;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class SQLSelectItem extends SQLObjectImpl implements SQLReplaceable {
   protected SQLExpr expr;
   protected String alias;
   protected boolean connectByRoot;
   protected transient long aliasHashCode64;
   protected List<String> aliasList;

   public SQLSelectItem() {
      this.connectByRoot = false;
   }

   public SQLSelectItem(SQLExpr expr) {
      this(expr, null);
   }

   public SQLSelectItem(int value) {
      this(new SQLIntegerExpr(value), null);
   }

   public SQLSelectItem(SQLExpr expr, String alias) {
      this.connectByRoot = false;
      this.expr = expr;
      this.alias = alias;
      if (expr != null) {
         expr.setParent(this);
      }

   }

   public SQLSelectItem(SQLExpr expr, String alias, boolean connectByRoot) {
      this.connectByRoot = false;
      this.connectByRoot = connectByRoot;
      this.expr = expr;
      this.alias = alias;
      if (expr != null) {
         expr.setParent(this);
      }

   }

   public SQLSelectItem(SQLExpr expr, List<String> aliasList, boolean connectByRoot) {
      this.connectByRoot = false;
      this.connectByRoot = connectByRoot;
      this.expr = expr;
      this.aliasList = aliasList;
      if (expr != null) {
         expr.setParent(this);
      }

   }

   public SQLExpr getExpr() {
      return this.expr;
   }

   public void setExpr(SQLExpr expr) {
      if (expr != null) {
         expr.setParent(this);
      }

      this.expr = expr;
   }

   public String computeAlias() {
      String alias = this.getAlias();
      if (alias == null) {
         if (this.expr instanceof SQLIdentifierExpr) {
            alias = ((SQLIdentifierExpr)this.expr).getName();
         } else if (this.expr instanceof SQLPropertyExpr) {
            alias = ((SQLPropertyExpr)this.expr).getName();
         }
      }

      return SQLUtils.normalize(alias);
   }

   public SQLDataType computeDataType() {
      return this.expr == null ? null : this.expr.computeDataType();
   }

   public String getAlias() {
      return this.alias;
   }

   public String getAlias2() {
      if (this.alias != null && this.alias.length() != 0) {
         char first = this.alias.charAt(0);
         if (first != '"' && first != '\'') {
            return this.alias;
         } else {
            char[] chars = new char[this.alias.length() - 2];
            int len = 0;

            for(int i = 1; i < this.alias.length() - 1; ++i) {
               char ch = this.alias.charAt(i);
               if (ch == '\\') {
                  ++i;
                  ch = this.alias.charAt(i);
               }

               chars[len++] = ch;
            }

            return new String(chars, 0, len);
         }
      } else {
         return this.alias;
      }
   }

   public void setAlias(String alias) {
      this.alias = alias;
   }

   public void output(Appendable buf) {
      try {
         if (this.connectByRoot) {
            buf.append(" CONNECT_BY_ROOT ");
         }

         this.expr.output(buf);
         if (this.alias != null && this.alias.length() != 0) {
            buf.append(" AS ");
            buf.append(this.alias);
         }

      } catch (IOException ex) {
         throw new FastsqlException("output error", ex);
      }
   }

   protected void accept0(SQLASTVisitor v) {
      if (v.visit(this) && this.expr != null) {
         this.expr.accept(v);
      }

      v.endVisit(this);
   }

   public boolean equals(Object o) {
      if (this == o) {
         return true;
      } else if (o != null && this.getClass() == o.getClass()) {
         SQLSelectItem that = (SQLSelectItem)o;
         if (this.connectByRoot != that.connectByRoot) {
            return false;
         } else if (this.alias_hash() != that.alias_hash()) {
            return false;
         } else {
            if (this.expr != null) {
               if (!this.expr.equals(that.expr)) {
                  return false;
               }
            } else if (that.expr != null) {
               return false;
            }

            return this.aliasList != null ? this.aliasList.equals(that.aliasList) : that.aliasList == null;
         }
      } else {
         return false;
      }
   }

   public int hashCode() {
      int result = this.expr != null ? this.expr.hashCode() : 0;
      result = 31 * result + (this.alias != null ? this.alias.hashCode() : 0);
      result = 31 * result + (this.connectByRoot ? 1 : 0);
      result = 31 * result + (int)(this.alias_hash() ^ this.alias_hash() >>> 32);
      result = 31 * result + (this.aliasList != null ? this.aliasList.hashCode() : 0);
      return result;
   }

   public boolean isConnectByRoot() {
      return this.connectByRoot;
   }

   public void setConnectByRoot(boolean connectByRoot) {
      this.connectByRoot = connectByRoot;
   }

   public SQLSelectItem clone() {
      SQLSelectItem x = new SQLSelectItem();
      x.alias = this.alias;
      if (this.expr != null) {
         x.setExpr(this.expr.clone());
      }

      x.connectByRoot = this.connectByRoot;
      if (this.aliasList != null) {
         x.aliasList = new ArrayList(this.aliasList);
      }

      return x;
   }

   public boolean replace(SQLExpr expr, SQLExpr target) {
      if (this.expr == expr) {
         this.setExpr(target);
         return true;
      } else {
         return false;
      }
   }

   public boolean match(String alias) {
      if (alias == null) {
         return false;
      } else {
         long hash = FnvHash.hashCode64(alias);
         return this.match(hash);
      }
   }

   public long alias_hash() {
      if (this.aliasHashCode64 == 0L) {
         this.aliasHashCode64 = FnvHash.hashCode64(this.alias);
      }

      return this.aliasHashCode64;
   }

   public boolean match(long alias_hash) {
      long hash = this.alias_hash();
      if (hash == alias_hash) {
         return true;
      } else if (this.expr instanceof SQLAllColumnExpr) {
         SQLTableSource resolvedTableSource = ((SQLAllColumnExpr)this.expr).getResolvedTableSource();
         return resolvedTableSource != null && resolvedTableSource.findColumn(alias_hash) != null;
      } else if (this.expr instanceof SQLIdentifierExpr) {
         return ((SQLIdentifierExpr)this.expr).nameHashCode64() == alias_hash;
      } else if (this.expr instanceof SQLPropertyExpr) {
         String ident = ((SQLPropertyExpr)this.expr).getName();
         if ("*".equals(ident)) {
            SQLTableSource resolvedTableSource = ((SQLPropertyExpr)this.expr).getResolvedTableSource();
            return resolvedTableSource != null && resolvedTableSource.findColumn(alias_hash) != null;
         } else {
            return this.alias == null && ((SQLPropertyExpr)this.expr).nameHashCode64() == alias_hash;
         }
      } else {
         return false;
      }
   }

   public List<String> getAliasList() {
      return this.aliasList;
   }

   public String toString() {
      DbType dbType = null;
      if (this.parent instanceof OracleSQLObject) {
         dbType = DbType.oracle;
      } else if (!(this.getExpr() instanceof DMSQLObject) && !(this.parent instanceof DMSQLObject)) {
         if (this.getExpr() instanceof KingbaseSQLObject || this.parent instanceof KingbaseSQLObject) {
            dbType = DbType.kingbase;
         }
      } else {
         dbType = DbType.dm;
      }

      if (dbType == null) {
         dbType = SQLUtils.getDbTypteByParent(this);
      }

      return SQLUtils.toSQLString(this, (DbType)dbType);
   }

   public boolean isUDTFSelectItem() {
      return this.aliasList != null && this.aliasList.size() > 0;
   }
}
