package org.chenyang.http.client.protocol;

import org.chenyang.http.Header;
import org.chenyang.http.HeaderElement;
import org.chenyang.http.HttpEntity;
import org.chenyang.http.HttpException;
import org.chenyang.http.HttpResponse;
import org.chenyang.http.HttpResponseInterceptor;
import org.chenyang.http.annotation.Contract;
import org.chenyang.http.annotation.ThreadingBehavior;
import org.chenyang.http.client.config.RequestConfig;
import org.chenyang.http.client.entity.DecompressingEntity;
import org.chenyang.http.client.entity.DeflateInputStreamFactory;
import org.chenyang.http.client.entity.GZIPInputStreamFactory;
import org.chenyang.http.client.entity.InputStreamFactory;
import org.chenyang.http.config.Lookup;
import org.chenyang.http.config.RegistryBuilder;
import org.chenyang.http.protocol.HttpContext;
import java.io.IOException;
import java.util.Locale;

@Contract(
   threading = ThreadingBehavior.IMMUTABLE_CONDITIONAL
)
public class ResponseContentEncoding implements HttpResponseInterceptor {
   public static final String UNCOMPRESSED = "http.client.response.uncompressed";
   private final Lookup<InputStreamFactory> decoderRegistry;
   private final boolean ignoreUnknown;

   public ResponseContentEncoding(Lookup<InputStreamFactory> decoderRegistry, boolean ignoreUnknown) {
      this.decoderRegistry = (Lookup<InputStreamFactory>)(decoderRegistry != null ? decoderRegistry : RegistryBuilder.create().register("gzip", GZIPInputStreamFactory.getInstance()).register("x-gzip", GZIPInputStreamFactory.getInstance()).register("deflate", DeflateInputStreamFactory.getInstance()).build());
      this.ignoreUnknown = ignoreUnknown;
   }

   public ResponseContentEncoding(boolean ignoreUnknown) {
      this((Lookup)null, ignoreUnknown);
   }

   public ResponseContentEncoding(Lookup<InputStreamFactory> decoderRegistry) {
      this(decoderRegistry, true);
   }

   public ResponseContentEncoding() {
      this((Lookup)null);
   }

   public void process(HttpResponse response, HttpContext context) throws HttpException, IOException {
      HttpEntity entity = response.getEntity();
      HttpClientContext clientContext = HttpClientContext.adapt(context);
      RequestConfig requestConfig = clientContext.getRequestConfig();
      if (requestConfig.isContentCompressionEnabled() && entity != null && entity.getContentLength() != 0L) {
         Header ceheader = entity.getContentEncoding();
         if (ceheader != null) {
            HeaderElement[] codecs = ceheader.getElements();

            for(HeaderElement codec : codecs) {
               String codecname = codec.getName().toLowerCase(Locale.ROOT);
               InputStreamFactory decoderFactory = this.decoderRegistry.lookup(codecname);
               if (decoderFactory != null) {
                  response.setEntity(new DecompressingEntity(response.getEntity(), decoderFactory));
                  response.removeHeaders("Content-Length");
                  response.removeHeaders("Content-Encoding");
                  response.removeHeaders("Content-MD5");
               } else if (!"identity".equals(codecname) && !this.ignoreUnknown) {
                  throw new HttpException("Unsupported Content-Encoding: " + codec.getName());
               }
            }
         }
      }

   }
}
