package com.metro.auth.platform.controller;

import com.metro.auth.platform.domain.ResultCode;
import com.metro.auth.platform.domain.ResultJson;
import com.metro.auth.platform.domain.auth.PlatformPersonnel;
import com.metro.auth.platform.domain.auth.User;
import com.metro.auth.platform.domain.auth.ResponseUserToken;
import com.metro.auth.platform.domain.auth.UserDetail;
import com.metro.auth.platform.generallog.LogAnnotation;
import com.metro.auth.platform.outlineapi.PlatformUrlManager;
import com.metro.auth.platform.service.AuthService;
import com.metro.auth.platform.utils.DateUtil;
import com.metro.auth.platform.utils.MailUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.mail.MessagingException;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

/**
 * @author zhouy
 * createAt: 2019/8/1
 */
@Slf4j
@RestController
@Api(tags = "统一登录相关接口")
@RequestMapping("/auth/v1")
public class AuthController {
    @Value("${jwt.header}")
    private String tokenHeader;

    private final AuthService authService;

    @Autowired
    public AuthController(AuthService authService) {

        this.authService = authService;
    }
    @LogAnnotation(operateContent="用户登录",operateType="登录")
    @PostMapping(value = "/login")
    @ApiOperation(value = "登录", notes = "验证成功返回token,测试管理员账号:admin123,12345678")
    public ResultJson<ResponseUserToken> login(
            @Valid @RequestBody User user){
        final ResponseUserToken response = authService.login(user.getAccount(), user.getPassword());
        return ResultJson.ok(response);
    }

    @PostMapping(value = "/logout")
    @ApiOperation(value = "登出", notes = "退出登录")
    @ApiImplicitParams({@ApiImplicitParam(name = "Authorization", value = "Authorization token", required = true, dataType = "string", paramType = "header")})
    public ResultJson logout(HttpServletRequest request){
        String token = request.getHeader(tokenHeader);
        if (token == null) {
            return ResultJson.failure(ResultCode.UNAUTHORIZED);
        }
        authService.logout(token);
        return ResultJson.ok();
    }

    @GetMapping(value = "/user")
    @ApiOperation(value = "根据token获取用户信息", notes = "根据token获取用户信息")
    @ApiImplicitParams({@ApiImplicitParam(name = "Authorization", value = "参数格式 token", required = true, dataType = "string", paramType = "header")})
    public ResultJson getUser(HttpServletRequest request){
        String token = request.getHeader(tokenHeader);
        if (token == null) {
            log.info(DateUtil.getCurTime()+ "token 为null 导致退出登录=============================================" );
            return ResultJson.failure(ResultCode.UNAUTHORIZED);
        }else{
            log.info("当前===============================================================token"+token);
        }
        UserDetail userDetail = authService.getUserByToken(token);
        if (userDetail==null){
            log.info("查询用户信息失败");
        }
        return ResultJson.ok(userDetail);
    }


    @GetMapping(value = "/refresh")
    @ApiImplicitParams({@ApiImplicitParam(name = "Authorization", value = "参数格式 token", required = true, dataType = "string", paramType = "header")})
    public ResultJson refreshAndGetAuthenticationToken(
            HttpServletRequest request){
        String token = request.getHeader(tokenHeader);
        ResponseUserToken response = authService.refresh(token);
        if(response == null) {
            return ResultJson.failure(ResultCode.BAD_REQUEST, "token无效");
        } else {
            return ResultJson.ok(response);
        }
    }

    /**
     * 功能描述: <br>扫码登录
     * 〈〉
     * @Param: [userId]
     * @Return: [userId]
     * @Author: zhouy
     * @Date: 2019-10-23 13:06
     */

    @GetMapping(value = "/qrcodelogin/{userId}")
    public ResultJson<ResponseUserToken> qrcodelogin(@PathVariable String userId){
        final PlatformPersonnel platformPersonnel = authService.findUserInfoByaccount(userId);
        if(platformPersonnel==null){
            return ResultJson.failure(ResultCode.FORBIDDEN);
        }
        final ResponseUserToken response = authService.login(platformPersonnel.getAccount(), platformPersonnel.getPassword());
        return ResultJson.ok(response);
    }

    /**
     *
     * @param user
     * @return
     */
    @LogAnnotation(operateContent="用户内网扫码登录",operateType="登录")
    @PostMapping(value = "/logincode")
//    @ApiOperation(value = "登录", notes = "根据随机码登录")
    public ResultJson<ResponseUserToken> logincode(@Valid @RequestBody User user){
        //根据code查用户账户和密码
        PlatformPersonnel platformPersonnel = authService.findUserInfoByCode(user.getCode());
        if (platformPersonnel!=null){
            final ResponseUserToken response= authService.login(platformPersonnel.getAccount(), platformPersonnel.getPassword());
            return ResultJson.ok(response);
         }else{
            return ResultJson.ok(false);
        }
    }
    @PostMapping(value = "/savecode")
    public ResultJson savecode(@Valid @RequestBody User user){
       int flag= authService.updateCodeByUserid(user.getAccount() ,user.getCode());
        if (flag>0){
            return ResultJson.ok(ResultCode.SUCCESS);
        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR);
        }
    }

    /**
     * 根据邮箱保存code
     * @param user
     * @return
     */
    @PostMapping(value = "/savecodeByemail")
    public ResultJson savecodeByemail(@Valid @RequestBody User user){
        int flag= authService.updateCodeByUserid(user.getAccount() ,user.getCode());
        if (flag>0){
            return ResultJson.ok(ResultCode.SUCCESS);
        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR);
        }
    }

    /**
     * 检验邮箱，如果存在发送邮件验证码
     * @param user
     * @return
     */
    @PostMapping(value = "/sendcodeToEmail")
    public ResultJson sendcodeToEmail(@Valid @RequestBody User user){
        //查询email是否存在，存在则把验证码发送到邮箱
        String email=user.getAccount();
        boolean sendstatus=false;
        PlatformPersonnel platformPersonnel= authService.findUserInfoByemail(email);
        if (platformPersonnel !=null){
            //发送邮件
            String url = PlatformUrlManager.EMAIL_URL +"?account="+platformPersonnel.getAccount()+"&code=" + user.getCode();
            String emailMsg = "<h3>授权请点击下面的超链接</h3><a href='" + url + "'> 点击授权 </a>";
            MailUtil.sendMail(email,"内网平台登录授权",emailMsg,new String[]{""});
            sendstatus=true;
             if (sendstatus){
                 return ResultJson.ok(ResultCode.SEND_SUCCESS);
             }else{
                 return ResultJson.ok(ResultCode.SEND_ERROR);
             }

        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR);
        }
    }


}
