package com.metro.auth.platform.service;

import com.metro.auth.platform.config.BCryptMD5;
import com.metro.auth.platform.domain.ResultCode;
import com.metro.auth.platform.domain.ResultJson;
import com.metro.auth.platform.domain.auth.*;
import com.metro.auth.platform.exception.CustomException;
import com.metro.auth.platform.mapper.AuthMapper;
import com.metro.auth.platform.mapper.PlatformDepartmentMapper;
import com.metro.auth.platform.mapper.PlatformPersonnelMapper;
import com.metro.auth.platform.mapper.PlatformRoleMenuMapper;
import com.metro.auth.platform.utils.JwtUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @author: zhouy
 * createAt: 2019/8/1
 */
@Service
public class AuthServiceImpl implements AuthService {
    private final AuthenticationManager authenticationManager;
    private final UserDetailsService userDetailsService;
    private final JwtUtils jwtTokenUtil;
    private final AuthMapper authMapper;
    private final PlatformPersonnelMapper platformPersonnelMapper;
    private final PlatformRoleMenuMapper platformRoleMenuMapper;
    private final PlatformDepartmentMapper platformDepartmentMapper;

    @Value("${jwt.tokenHead}")
    private String tokenHead;

    @Autowired
    public AuthServiceImpl(AuthenticationManager authenticationManager,
                           @Qualifier("CustomUserDetailsService") UserDetailsService userDetailsService,
                           JwtUtils jwtTokenUtil, AuthMapper authMapper,
                           PlatformPersonnelMapper platformPersonnelMapper,
                           PlatformRoleMenuMapper platformRoleMenuMapper,
                           PlatformDepartmentMapper platformDepartmentMapper
    ) {
        this.authenticationManager = authenticationManager;
        this.userDetailsService = userDetailsService;
        this.jwtTokenUtil = jwtTokenUtil;
        this.authMapper = authMapper;
        this.platformPersonnelMapper = platformPersonnelMapper;
        this.platformRoleMenuMapper = platformRoleMenuMapper;
        this.platformDepartmentMapper = platformDepartmentMapper;
    }

    @Override
    public UserDetail register(UserDetail userDetail) {
        final String username = userDetail.getUsername();
        if(authMapper.findByUsername(username)!=null) {
            throw new CustomException(ResultJson.failure(ResultCode.BAD_REQUEST, "用户已存在"));
        }
        //BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
        final String rawPassword = userDetail.getPassword();
        userDetail.setPassword(BCryptMD5.encode(rawPassword));
        userDetail.setLastPasswordResetDate(new Date());
        authMapper.insert(userDetail);
        String roleId = userDetail.getRole().getId();
        PlatformSysRole role = authMapper.findRoleById(roleId);
        userDetail.setRole(role);
        authMapper.insertRole(userDetail.getId(), roleId);
        return userDetail;
    }

    @Override
    public ResponseUserToken login(String account, String password) {
        //用户验证
        final Authentication authentication = authenticate(account, password);
        //存储认证信息
        SecurityContextHolder.getContext().setAuthentication(authentication);
        //生成token
        final UserDetail userDetail = (UserDetail) authentication.getPrincipal();

        final String token = jwtTokenUtil.generateAccessToken(userDetail);
        //存储token
        jwtTokenUtil.putToken(account, token);
        return new ResponseUserToken(token, userDetail);

    }

    @Override
    public void logout(String token) {
        token = token.substring(tokenHead.length());
        String account = jwtTokenUtil.getUsernameFromToken(token);
        jwtTokenUtil.deleteToken(account);
    }

    @Override
    public ResponseUserToken refresh(String oldToken) {
        String token = oldToken.substring(tokenHead.length());
        String account = jwtTokenUtil.getUsernameFromToken(token);
        UserDetail userDetail = (UserDetail) userDetailsService.loadUserByUsername(account);
        if (jwtTokenUtil.canTokenBeRefreshed(token, userDetail.getLastPasswordResetDate())){
            token =  jwtTokenUtil.refreshToken(token);
            return new ResponseUserToken(token, userDetail);
        }
        return null;
    }

    @Override
    public UserDetail getUserByToken(String token) {
        token = token.substring(tokenHead.length());
        UserDetail userDetail = jwtTokenUtil.getUserFromToken(token);
        PlatformPersonnel platformPersonnel = platformPersonnelMapper.findByUsername(userDetail.getUsername());
        //获取人员所属中心
        String dept = platformPersonnel.getDepartment();
        if(!"1".equals(dept)){
            String[] id = dept.split(",");
            List<String> result = Arrays.asList(id);
            List<String> centeridlist = new ArrayList<>();
            result.forEach(t->{
               String centerId = getCenterId(t);
               if(StringUtils.isNotEmpty(centerId) && !centeridlist.contains(centerId)){//中心不空
                   centeridlist.add(centerId);
               }
            });
            String centerid = centeridlist.stream().collect(Collectors.joining(","));
//            platformPersonnel.setCenter(centerid);
            platformPersonnel.setCenterid(centerid);
        }

        userDetail.setPlatformPersonnel(platformPersonnel);
        //根据userid查询按钮
        List<PlatformSysRole> rolelist = authMapper.findRolelistByUserId(userDetail.getId());
        List<String> names= new ArrayList<>();
        names.add("ROLE_USER");
        String rolenamelist="";
               for (int i=0;i<rolelist.size();i++){
                   names.add(rolelist.get(i).getName());
            }
        List<PlatformSysButton> platformRoleButtonlist = platformRoleMenuMapper.selectPlatfromRoleListButton(names);
        userDetail.setPermission(platformRoleButtonlist);
        return userDetail;
    }

    /**
     * 根据userid查询用户所有上级部门
     * @param userid
     * @return
     */
    @Override
    public PlatformPersonnel getUserByUserid(int userid) {
        PlatformPersonnel platformPersonnel = platformPersonnelMapper.selectByPrimaryKey(userid);
        //获取人员所属中心
        String dept = platformPersonnel.getDepartment();
        if(!"1".equals(dept)){
            String[] id = dept.split(",");
            List<String> result = Arrays.asList(id);
            List<String> centeridlist = new ArrayList<>();
            result.forEach(t->{
                String centerId = getCenterId(t);
                if(StringUtils.isNotEmpty(centerId) && !centeridlist.contains(centerId)){//中心不空
                    centeridlist.add(centerId);
                }
            });
            String centerid = centeridlist.stream().collect(Collectors.joining(","));
            platformPersonnel.setCenterid(centerid);
        }
        return platformPersonnel;
    }
    /**
     * 根据用户所属机构ID返回中心ID
     */
    public String getCenterId(String t){
        String centerId = "";
        PlatformDepartment platformDepartment =platformDepartmentMapper.selectByPrimaryKey(Integer.parseInt(t));
        String pid = platformDepartment.getPid();
        if (!"0".equals(pid) && !"1".equals(pid) && !"1".equals(t)){//父级ID不为1，证明其不是中心。迭代向上获取
            centerId = getCenterId(pid);
        }else{
            centerId = t;
        }

        return centerId;
    }




    private Authentication authenticate(String account, String password) {
        try {
           Authentication authentication = new UsernamePasswordAuthenticationToken(account, password);
            //该方法会去调用userDetailsService.loadUserByUsername()去验证用户名和密码，如果正确，则存储该用户名密码到“security 的 context中”
            return authenticationManager.authenticate(authentication);
        } catch (DisabledException | BadCredentialsException e) {
            e.printStackTrace();
            throw new CustomException(ResultJson.failure(ResultCode.LOGIN_ERROR, e.getMessage()));
        }
    }

    /**
     * 功能描述: <br>根据用户account查询用户信息
     * 〈〉
     * @Param: []
     * @Return: []
     * @Author: zhouy
     * @Date: 2019-10-23 12:02
     */
    @Override
    public PlatformPersonnel findUserInfoByaccount(String account){
        PlatformPersonnel platformPersonnel = platformPersonnelMapper.findByUsername(account);
        return platformPersonnel;
    }
    @Override
    public PlatformPersonnel findUserInfoByemail(String email){
        PlatformPersonnel platformPersonnel = platformPersonnelMapper.findUserInfoByemail(email);
        return platformPersonnel;
    }
    @Override
    public PlatformPersonnel findUserInfoByCode(String code){
        PlatformPersonnel platformPersonnel = platformPersonnelMapper.findUserInfoByCode(code);
        return platformPersonnel;
    }

    @Override
    public int updateCodeByUserid(String account, String code){
      int flag=  platformPersonnelMapper.updateCodeByUserid(account,code);
      return  flag;
    }
    @Override
    public  int updateCodeByEmail(String email,String code){
        int flag=platformPersonnelMapper.updateCodeByEmail(email,code);
        return  flag;
    }
}
