package com.metro.auth.platform.controller;

import com.github.pagehelper.PageInfo;
import com.metro.auth.platform.domain.ResultCode;
import com.metro.auth.platform.domain.ResultJson;
import com.metro.auth.platform.domain.auth.PlatformMetroworkMenu;
import com.metro.auth.platform.service.PlatformMetroworkMenuService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 地铁工作菜单管理控制器
 */
@RestController
@RequestMapping("/metrowork")
@Api(tags = "地铁工作菜单管理相关接口")
public class PlatformMetroworkMenuController {

    @Autowired
    private PlatformMetroworkMenuService platformMetroworkMenuService;

    /**
     * 获取所有启用的菜单（前端调用）
     */
    @GetMapping("/menus")
    @ApiOperation(value = "获取启用的菜单列表", notes = "获取所有状态为启用的菜单，按排序号排序")
    public ResultJson<List<PlatformMetroworkMenu>> getEnabledMenus() {
        List<PlatformMetroworkMenu> menus = platformMetroworkMenuService.getEnabledMenus();
        return ResultJson.ok(menus);
    }

    /**
     * 分页获取所有菜单（管理端调用）
     */
    @GetMapping("/admin/list")
    @ApiOperation(value = "分页获取菜单列表", notes = "分页获取所有菜单，支持条件查询")
    public ResultJson<PageInfo<PlatformMetroworkMenu>> getMenusPaged(
            @RequestParam(required = false, defaultValue = "1") int page,
            @RequestParam(required = false, defaultValue = "10") int size,
            @RequestParam(required = false) String name,
            @RequestParam(required = false) String code) {

        PlatformMetroworkMenu menu = new PlatformMetroworkMenu();
        if (name != null && !name.trim().isEmpty()) {
            menu.setName(name.trim());
        }
        if (code != null && !code.trim().isEmpty()) {
            menu.setCode(code.trim());
        }

        PageInfo<PlatformMetroworkMenu> pageInfo = platformMetroworkMenuService.getMenusPaged(menu, page, size);
        return ResultJson.ok(pageInfo);
    }

    /**
     * 根据ID获取菜单详情
     */
    @GetMapping("/admin/{id}")
    @ApiOperation(value = "获取菜单详情", notes = "根据菜单ID获取详细信息")
    public ResultJson<PlatformMetroworkMenu> getMenuById(@PathVariable String id) {
        PlatformMetroworkMenu menu = platformMetroworkMenuService.getMenuById(id);
        if (menu != null) {
            return ResultJson.ok(menu);
        } else {
            return ResultJson.failure(ResultCode.NOT_FOUND);
        }
    }

    /**
     * 新增菜单
     */
    @PostMapping("/admin/save")
    @ApiOperation(value = "新增菜单", notes = "创建新的菜单项")
    public ResultJson saveMenu(@RequestBody PlatformMetroworkMenu menu) {
        platformMetroworkMenuService.saveMenu(menu);
        return ResultJson.ok();
    }

    /**
     * 更新菜单
     */
    @PutMapping("/admin/update")
    @ApiOperation(value = "更新菜单", notes = "更新菜单信息")
    public ResultJson updateMenu(@RequestBody PlatformMetroworkMenu menu) {
        platformMetroworkMenuService.updateMenu(menu);
        return ResultJson.ok();
    }

    /**
     * 删除菜单
     */
    @DeleteMapping("/admin/{id}")
    @ApiOperation(value = "删除菜单", notes = "根据ID删除菜单")
    public ResultJson deleteMenu(@PathVariable String id) {
        platformMetroworkMenuService.deleteMenu(id);
        return ResultJson.ok();

    }

    /**
     * 启用菜单
     */
    @PutMapping("/admin/{id}/enable")
    @ApiOperation(value = "启用菜单", notes = "启用指定的菜单")
    public ResultJson enableMenu(@PathVariable String id) {
        platformMetroworkMenuService.enableMenu(id);
        return ResultJson.ok();

    }

    /**
     * 停用菜单
     */
    @PutMapping("/admin/{id}/disable")
    @ApiOperation(value = "停用菜单", notes = "停用指定的菜单")
    public ResultJson disableMenu(@PathVariable String id) {
        platformMetroworkMenuService.disableMenu(id);
        return ResultJson.ok();
    }
}
