package com.metro.auth.platform.controller;

import com.metro.auth.platform.domain.ResultCode;
import com.metro.auth.platform.domain.ResultJson;
import com.metro.auth.platform.domain.auth.PlatformUserFavoriteApps;
import com.metro.auth.platform.service.PlatformUserFavoriteAppsService;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 用户常用应用Controller
 */
@RestController
@RequestMapping("/PlatformUserFavoriteAppsController")
public class PlatformUserFavoriteAppsController {

    @Resource
    private PlatformUserFavoriteAppsService platformUserFavoriteAppsService;

    /**
     * 获取我的常用应用列表
     * @param userId 用户ID
     * @return 常用应用列表
     */
    @GetMapping("/myFavorites/{userId}")
    public ResultJson getMyFavoriteApps(@PathVariable String userId){
        List<PlatformUserFavoriteApps> list = platformUserFavoriteAppsService.getFavoriteAppsByUserId(userId);
        return ResultJson.ok(list);
    }

    /**
     * 获取未设置为常用的应用列表
     * @param userId 用户ID
     * @param keyword 搜索关键词（可选）
     * @return 未设置的应用列表
     */
    @GetMapping("/unfavorites/{userId}")
    public ResultJson getUnfavoriteApps(@PathVariable String userId, 
                                       @RequestParam(required = false) String keyword){
        List<PlatformUserFavoriteApps> list = platformUserFavoriteAppsService.getUnfavoriteApps(userId, keyword);
        return ResultJson.ok(list);
    }

    /**
     * 添加常用应用
     * @param request 请求参数
     * @return 操作结果
     */
    @PostMapping("/addFavorite")
    public ResultJson addFavoriteApp(@RequestBody FavoriteAppRequest request){
        boolean success = platformUserFavoriteAppsService.addFavoriteApp(
            request.getUserId(), 
            request.getAppId(), 
            request.getSortOrder()
        );
        if(success){
            return ResultJson.ok();
        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR, "应用已存在或添加失败");
        }
    }

    /**
     * 删除常用应用
     * @param request 请求参数
     * @return 操作结果
     */
    @PostMapping("/removeFavorite")
    public ResultJson removeFavoriteApp(@RequestBody FavoriteAppRequest request){
        boolean success = platformUserFavoriteAppsService.removeFavoriteApp(
            request.getUserId(), 
            request.getAppId()
        );
        if(success){
            return ResultJson.ok();
        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR, "删除失败");
        }
    }

    /**
     * 更新常用应用排序
     * @param request 请求参数
     * @return 操作结果
     */
    @PutMapping("/updateFavoriteSort")
    public ResultJson updateFavoriteSort(@RequestBody FavoriteAppRequest request){
        boolean success = platformUserFavoriteAppsService.updateFavoriteSort(
            request.getUserId(), 
            request.getAppId(), 
            request.getSortOrder()
        );
        if(success){
            return ResultJson.ok();
        }else{
            return ResultJson.failure(ResultCode.RESPONSE_ERROR, "更新失败");
        }
    }

    /**
     * 批量更新常用应用排序
     * @param request 请求参数
     * @return 操作结果
     */
    @PutMapping("/updateFavoriteSortBatch")
    public ResultJson updateFavoriteSortBatch(@RequestBody BatchSortRequest request){
        try {
            for (FavoriteAppRequest item : request.getFavoriteApps()) {
                platformUserFavoriteAppsService.updateFavoriteSort(
                    request.getUserId(), 
                    item.getAppId(), 
                    item.getSortOrder()
                );
            }
            return ResultJson.ok();
        } catch (Exception e) {
            return ResultJson.failure(ResultCode.RESPONSE_ERROR, "批量更新失败");
        }
    }

    /**
     * 内部类：常用应用请求参数
     */
    public static class FavoriteAppRequest {
        private String userId;
        private Integer appId;
        private Integer sortOrder;

        public String getUserId() {
            return userId;
        }

        public void setUserId(String userId) {
            this.userId = userId;
        }

        public Integer getAppId() {
            return appId;
        }

        public void setAppId(Integer appId) {
            this.appId = appId;
        }

        public Integer getSortOrder() {
            return sortOrder;
        }

        public void setSortOrder(Integer sortOrder) {
            this.sortOrder = sortOrder;
        }
    }

    /**
     * 内部类：批量排序请求参数
     */
    public static class BatchSortRequest {
        private String userId;
        private List<FavoriteAppRequest> favoriteApps;

        public String getUserId() {
            return userId;
        }

        public void setUserId(String userId) {
            this.userId = userId;
        }

        public List<FavoriteAppRequest> getFavoriteApps() {
            return favoriteApps;
        }

        public void setFavoriteApps(List<FavoriteAppRequest> favoriteApps) {
            this.favoriteApps = favoriteApps;
        }
    }
}
