package com.scpyun.platform.jilinsscgsdp.service.impl;

import com.scpyun.base.core.annotation.Api;
import com.scpyun.base.core.annotation.ApiOperation;
import com.scpyun.base.db.service.CommonService;
import com.scpyun.platform.jilinsscgsdp.utils.DataScopeUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

@Api("重点人员统计")
@Service("keyPersonStatistics")
public class KeyPersonStatisticsImpl {

    @Autowired
    CommonService commonService;

    private final String SUCCESS = "success";

    private String namespace = "com.scpyun.platform.standard.jilinsscgsdp.keyPersonStatistics.";

    /**
     * 评分统计数据
     *
     * @return
     */
    @ApiOperation(value = "评分统计数据", desc = "评分统计数据")
    public Map<String, Object> getScoreStatistics(Map<String, Object> map) {
        // 获取重点人员总数
        Map<String, Object> peopleCountMap = commonService.getObject(namespace + "selectPeopleCount", new HashMap<>());

        Map<String, Object> mentalPatient = commonService.getObject(namespace + "selectMentalPatientRatingCount", new HashMap<>());
        Map<String, Object> minors = commonService.getObject(namespace + "selectMinorsRatingCount", new HashMap<>());
        Map<String, Object> released = commonService.getObject(namespace + "selectReleasedRatingCount", new HashMap<>());

        // 构建返回结果
        Map<String, Object> result = new HashMap<>();

        // 精神病患者数据
        Map<String, Object> mentalPatientData = new HashMap<>();
        int mentalTotal = getIntValue(peopleCountMap, "mental_patient_count");
        mentalPatientData.put("total", mentalTotal);
        mentalPatientData.put("categories", buildCategories(mentalPatient));
        result.put("mentalPatient", mentalPatientData);

        // 未成年人数据
        Map<String, Object> minorsData = new HashMap<>();
        int minorsTotal = getIntValue(peopleCountMap, "minors_count");
        minorsData.put("total", minorsTotal);
        minorsData.put("categories", buildCategories(minors));
        result.put("minors", minorsData);

        // 刑满释放数据
        Map<String, Object> releasedData = new HashMap<>();
        int releasedTotal = getIntValue(peopleCountMap, "released_count");
        releasedData.put("total", releasedTotal);
        releasedData.put("categories", buildCategories(released));
        result.put("released", releasedData);

        return result;
    }

    /**
     * 构建分类数据
     */
    private Object buildCategories(Map<String, Object> data) {
        Map<String, Object>[] categories = new Map[3];

        Map<String, Object> categoryA = new HashMap<>();
        categoryA.put("name", "A类");
        categoryA.put("value", getIntValue(data, "count_a"));
        categories[0] = categoryA;

        Map<String, Object> categoryB = new HashMap<>();
        categoryB.put("name", "B类");
        categoryB.put("value", getIntValue(data, "count_b"));
        categories[1] = categoryB;

        Map<String, Object> categoryC = new HashMap<>();
        categoryC.put("name", "C类");
        categoryC.put("value", getIntValue(data, "count_c"));
        categories[2] = categoryC;

        return categories;
    }

    /**
     * 获取整数值
     */
    private int getIntValue(Map<String, Object> data, String key) {
        if (data == null || data.get(key) == null) {
            return 0;
        }
        Object value = data.get(key);
        if (value instanceof Number) {
            return ((Number) value).intValue();
        }
        try {
            return Integer.parseInt(value.toString());
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    /**
     * 重点人员总数 和 待办已办数
     * @return
     */
    @ApiOperation(value = "回访统计数据", desc = "回访统计数据")
    public Map<String, Object> getOverdueWarningStatistics(Map<String, Object> map) {
        Map<String, Object> result = new HashMap<>();

        // 获取重点人员总数
        Map<String, Object> peopleCountMap = commonService.getObject(namespace + "selectPeopleCount", new HashMap<>());

        // 获取精神病、未成年、刑满释放待办数据
        //total_pending: 待办数量 - 办理状态为待办(do_status='0')且未临期(is_overdue='0')的回访记录数
        //total_done: 已办数量 - 办理状态为已办(do_status='1')的回访记录数
        //total_overdue: 超期未办数量 - 办理状态为超期未办(do_status='2')的回访记录数
        //total_overdue_pending: 临期未办数量 - 办理状态为待办(do_status='0')且已临期(is_overdue='1')的回访记录数
        Map<String, Object> mentalPatientMap = commonService.getObject(namespace + "selectMentalPatientOverdueWarningCount", new HashMap<>());
        Map<String, Object> minorsMap = commonService.getObject(namespace + "selectMinorsOverdueWarningCount", new HashMap<>());
        Map<String, Object> releasedMap = commonService.getObject(namespace + "selectReleasedOverdueWarningCount", new HashMap<>());

        // 提取各类人员数据
        int mentalPatientCount = getIntValue(peopleCountMap, "mental_patient_count");
        int minorsCount = getIntValue(peopleCountMap, "minors_count");
        int releasedCount = getIntValue(peopleCountMap, "released_count");

        // 精神病患者统计
        int mentalCompleted = getIntValue(mentalPatientMap, "total_done");
        int mentalUncompleted = getIntValue(mentalPatientMap, "total_pending");
        int mentalNearOverdue = getIntValue(mentalPatientMap, "total_overdue_pending");
        int mentalOverdue = getIntValue(mentalPatientMap, "total_overdue");

        // 未成年人统计
        int minorsCompleted = getIntValue(minorsMap, "total_done");
        int minorsUncompleted = getIntValue(minorsMap, "total_pending");
        int minorsNearOverdue = getIntValue(minorsMap, "total_overdue_pending");
        int minorsOverdue = getIntValue(minorsMap, "total_overdue");

        // 刑满释放人员统计
        int releasedCompleted = getIntValue(releasedMap, "total_done");
        int releasedUncompleted = getIntValue(releasedMap, "total_pending");
        int releasedNearOverdue = getIntValue(releasedMap, "total_overdue_pending");
        int releasedOverdue = getIntValue(releasedMap, "total_overdue");

        // 计算汇总数据
        int summaryCompleted = mentalCompleted + minorsCompleted + releasedCompleted;
        int summaryUncompleted = mentalUncompleted + minorsUncompleted + releasedUncompleted;
        int summaryNearOverdue = mentalNearOverdue + minorsNearOverdue + releasedNearOverdue;
        int summaryOverdue = mentalOverdue + minorsOverdue + releasedOverdue;
        int summaryTotal = summaryCompleted + summaryUncompleted + summaryNearOverdue + summaryOverdue;

        // 构建汇总数据
        Map<String, Object> summary = new HashMap<>();
        summary.put("total", summaryTotal);
        summary.put("completed", summaryCompleted);
        summary.put("uncompleted", summaryUncompleted);
        summary.put("nearOverdue", summaryNearOverdue);
        summary.put("overdue", summaryOverdue);
        result.put("summary", summary);

        // 构建精神病患者数据
        Map<String, Object> mentalPatient = new HashMap<>();
        mentalPatient.put("total", mentalPatientCount);
        mentalPatient.put("completed", mentalCompleted);
        mentalPatient.put("uncompleted", mentalUncompleted);
        mentalPatient.put("nearOverdue", mentalNearOverdue);
        mentalPatient.put("overdue", mentalOverdue);
        result.put("mentalPatient", mentalPatient);

        // 构建刑满释放人员数据
        Map<String, Object> released = new HashMap<>();
        released.put("total", releasedCount);
        released.put("completed", releasedCompleted);
        released.put("uncompleted", releasedUncompleted);
        released.put("nearOverdue", releasedNearOverdue);
        released.put("overdue", releasedOverdue);
        result.put("released", released);

        // 构建未成年人数据
        Map<String, Object> minors = new HashMap<>();
        minors.put("total", minorsCount);
        minors.put("completed", minorsCompleted);
        minors.put("uncompleted", minorsUncompleted);
        minors.put("nearOverdue", minorsNearOverdue);
        minors.put("overdue", minorsOverdue);
        result.put("minors", minors);

        return result;
    }


    @ApiOperation(value = "评分统计数据", desc = "评分统计数据")
    public Map<String, Object> getRatingStatistics(Map<String, Object> params) {
        Object _user = params.get("_user");
        if (_user != null) {
            Map<String, String> pos = DataScopeUtil.getPosition(_user);
            params.put("scope_grade", pos.get("grade"));
            params.put("scope_area_id", pos.get("area_id"));
        }

        Map<String, Object> result = new HashMap<>();

        // 获取重点人员总数
        Map<String, Object> peopleCountMap = commonService.getObject(namespace + "selectPeopleCount", params);

        // 获取精神病、未成年、刑满释放评分统计数据
        // 注意：XML中最后三个重复的查询ID是针对评分历史表(jl_key_person_rating_history)的查询
        // selectReleasedOverdueWarningCount (第二个) -> 精神病患者评分
        // selectMinorsRatingCount (第二个) -> 未成年人评分
        // selectReleasedRatingCount (第二个) -> 刑满释放人员评分
        Map<String, Object> mentalPatientMap = commonService.getObject(namespace + "selectMentalPatientRatingCount1", params);
        Map<String, Object> minorsMap = commonService.getObject(namespace + "selectMinorsRatingCount1", params);
        Map<String, Object> releasedMap = commonService.getObject(namespace + "selectReleasedRatingCount1", params);

        // 提取各类人员数据
        int mentalPatientCount = getIntValue(peopleCountMap, "mental_patient_count");
        int minorsCount = getIntValue(peopleCountMap, "minors_count");
        int releasedCount = getIntValue(peopleCountMap, "released_count");

        // 精神病患者统计
        int mentalCompleted = getIntValue(mentalPatientMap, "total_done");
        int mentalUncompleted = getIntValue(mentalPatientMap, "total_pending");
        int mentalNearOverdue = getIntValue(mentalPatientMap, "total_overdue_pending");
        int mentalOverdue = getIntValue(mentalPatientMap, "total_overdue");

        // 未成年人统计
        int minorsCompleted = getIntValue(minorsMap, "total_done");
        int minorsUncompleted = getIntValue(minorsMap, "total_pending");
        int minorsNearOverdue = getIntValue(minorsMap, "total_overdue_pending");
        int minorsOverdue = getIntValue(minorsMap, "total_overdue");

        // 刑满释放人员统计
        int releasedCompleted = getIntValue(releasedMap, "total_done");
        int releasedUncompleted = getIntValue(releasedMap, "total_pending");
        int releasedNearOverdue = getIntValue(releasedMap, "total_overdue_pending");
        int releasedOverdue = getIntValue(releasedMap, "total_overdue");

        // 计算汇总数据
        int summaryCompleted = mentalCompleted + minorsCompleted + releasedCompleted;
        int summaryUncompleted = mentalUncompleted + minorsUncompleted + releasedUncompleted;
        int summaryNearOverdue = mentalNearOverdue + minorsNearOverdue + releasedNearOverdue;
        int summaryOverdue = mentalOverdue + minorsOverdue + releasedOverdue;
        int summaryTotal = summaryCompleted + summaryUncompleted + summaryNearOverdue + summaryOverdue;

        // 构建汇总数据
        Map<String, Object> summary = new HashMap<>();
        summary.put("total", summaryTotal);
        summary.put("completed", summaryCompleted);
        summary.put("uncompleted", summaryUncompleted);
        summary.put("nearOverdue", summaryNearOverdue);
        summary.put("overdue", summaryOverdue);
        result.put("summary", summary);

        // 构建精神病患者数据
        Map<String, Object> mentalPatient = new HashMap<>();
        mentalPatient.put("total", mentalPatientCount);
        mentalPatient.put("completed", mentalCompleted);
        mentalPatient.put("uncompleted", mentalUncompleted);
        mentalPatient.put("nearOverdue", mentalNearOverdue);
        mentalPatient.put("overdue", mentalOverdue);
        result.put("mentalPatient", mentalPatient);

        // 构建刑满释放人员数据
        Map<String, Object> released = new HashMap<>();
        released.put("total", releasedCount);
        released.put("completed", releasedCompleted);
        released.put("uncompleted", releasedUncompleted);
        released.put("nearOverdue", releasedNearOverdue);
        released.put("overdue", releasedOverdue);
        result.put("released", released);

        // 构建未成年人数据
        Map<String, Object> minors = new HashMap<>();
        minors.put("total", minorsCount);
        minors.put("completed", minorsCompleted);
        minors.put("uncompleted", minorsUncompleted);
        minors.put("nearOverdue", minorsNearOverdue);
        minors.put("overdue", minorsOverdue);
        result.put("minors", minors);

        return result;
    }
}
