package com.scpyun.platform.jilinsscgsdp.service.impl;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.scpyun.base.bean.Page;
import com.scpyun.base.core.annotation.Api;
import com.scpyun.base.core.annotation.ApiOperation;
import com.scpyun.base.core.exception.CustomException;
import com.scpyun.base.core.utils.DataSourceSwitcher;
import com.scpyun.base.db.service.CommonService;
import com.scpyun.platform.jilinsscgsdp.bean.dto.*;
import com.scpyun.platform.jilinsscgsdp.bean.vo.ReqWarningStatisticsVo;
import com.scpyun.platform.jilinsscgsdp.enums.EventStatusEnum;
import com.scpyun.platform.jilinsscgsdp.enums.WarningLevelEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.validation.annotation.Validated;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author sl
 * @date 2025/5/7 13:31
 * @description 预警研判
 */
@Api("预警研判")
@Slf4j
@Service("waringService")
@Validated
public class WaringServiceImpl {

    @Autowired
    CommonService commonService;

    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    private final String nameSpace = "com.scpyun.platform.standard.jilinsscgsdp.event.";
    private final String nameSpaceUser = "com.scpyun.platform.standard.jilinsscgsdp.eventuser.";
    private final String nameSpaceRiskFactor = "com.scpyun.platform.standard.jilinsscgsdp.eventriskfactor.";
    private final String nameSpaceWaiting = "com.scpyun.platform.standard.jilinsscgsdp.eventwaiting.";
    private final String nameSpaceFocusAreas = "com.scpyun.platform.standard.jilinsscgsdp.focusareas.";
    private final String areasNameSpaceHistory = "com.scpyun.platform.standard.jilinsscgsdp.focusareasHisTory.";

    @ApiOperation(value = "重点人数量", desc = "重点人数量")
    public Integer getTriggerEventCount(ImportantUserStatisticsDTO importantUserStatisticsDTO) {
        return commonService.getObject(nameSpaceUser + "getTriggerEventCount", importantUserStatisticsDTO);
    }

    @ApiOperation(value = "矛盾超期预警数量", desc = "矛盾超期预警数量")
    public Integer getOverdueEventCount(OverdueEventStatisticsDTO overdueEventStatisticsDTO) {
        return commonService.getObject(nameSpace + "getOverdueEventCount", overdueEventStatisticsDTO);
    }

    @ApiOperation(value = "涉众预警数量", desc = "涉众预警数量")
    public Integer getInvolveEveryoneCount(InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO) {
        involveEveryoneStatisticsDTO.setTriggerType("3");
        return commonService.getObject(nameSpace + "getInvolveEveryoneCount", involveEveryoneStatisticsDTO);
    }

    @ApiOperation(value = "风险因子预警数量", desc = "风险因子预警数量")
    public Integer getEventRiskFactorCount(EventRiskFactorDTO eventRiskFactorDTO) {
        return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCount", eventRiskFactorDTO);
    }

    @ApiOperation(value = "群体性事件预警数量", desc = "群体性事件预警数量")
    public Integer getGroupEventWarningCount(GroupEventWarningDTO groupEventWarningDTO) {
        groupEventWarningDTO.setTriggerType("2");
        return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCount", groupEventWarningDTO);
    }

    @ApiOperation(value = "重点领域预警数量", desc = "重点领域预警数量")
    public Integer getFocusAreasCount(FocusAreasStatisticsDTO focusAreasStatisticsDTO) {
        return commonService.getObject(nameSpaceFocusAreas + "getFocusAreasCount", focusAreasStatisticsDTO);
    }

    @ApiOperation(value = "综合预警研判数量统计", desc = "综合预警研判数量统计")
    public Map<String, Object> getWarningStatisticsCount(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        if (WarningLevelEnum.THREE_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用三级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 三级预警
            return this.threeLevelWarningCount(reqWarningStatisticsVo);
        } else if (WarningLevelEnum.TWO_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用二级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 二级预警
            return this.oneOrTwoLevelWarningCount(reqWarningStatisticsVo);
        } else if (WarningLevelEnum.FIRST_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用一级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 一级预警
            return this.oneOrTwoLevelWarningCount(reqWarningStatisticsVo);
        } else if (!Optional.ofNullable(reqWarningStatisticsVo.getAlarmLevel()).isPresent()) {
            // 重点事统计
            if (Optional.ofNullable(reqWarningStatisticsVo.getImportantEventStatus()).isPresent()) {
                return this.importantEventCount(reqWarningStatisticsVo);
            }
        }

        throw new CustomException("综合研判预警参数异常！");
    }

    @ApiOperation(value = "综合预警研判历史数量统计", desc = "综合预警研判历史数量统计")
    public Map<String, Object> getWarningStatisticsCountHistory(ReqWarningStatisticsVo reqWarningStatisticsVo) {
        // log.info("综合预警研判历史数量统计> 请求参数：{}", JSONUtil.toJsonStr(reqWarningStatisticsVo));
        if (WarningLevelEnum.THREE_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用三级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 三级预警
            return this.threeLevelWarningCountHistory(reqWarningStatisticsVo);
        } else if (WarningLevelEnum.TWO_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用二级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 二级预警
            return this.oneOrTwoLevelWarningCountHistory(reqWarningStatisticsVo);
        } else if (WarningLevelEnum.FIRST_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel())) {
            // log.info("开始调用一级预警：{}", JSON.toJSONString(reqWarningStatisticsVo));
            // 一级预警
            return this.oneOrTwoLevelWarningCountHistory(reqWarningStatisticsVo);
        }
        throw new CustomException("综合研判预警参数异常！");
    }

    @ApiOperation(value = "首页预警统计数量统计", desc = "首页预警统计数量统计")
    public Map<String, Object> getHomeWarningStatisticsCount(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        ReqWarningStatisticsVo threeWarningStatistics = new ReqWarningStatisticsVo();
        BeanUtils.copyProperties(reqWarningStatisticsVo, threeWarningStatistics);
        threeWarningStatistics.setAlarmLevel(WarningLevelEnum.THREE_LEVEL.getStatus());
        Map<String, Object> threeLevel = this.threeLevelWarningCount(threeWarningStatistics);

        ReqWarningStatisticsVo twoWarningStatistics = new ReqWarningStatisticsVo();
        BeanUtils.copyProperties(reqWarningStatisticsVo, twoWarningStatistics);
        twoWarningStatistics.setAlarmLevel(WarningLevelEnum.TWO_LEVEL.getStatus());
        Map<String, Object> twoLevel = this.oneOrTwoLevelWarningCount(twoWarningStatistics);

        ReqWarningStatisticsVo oneWarningStatistics = new ReqWarningStatisticsVo();
        BeanUtils.copyProperties(reqWarningStatisticsVo, oneWarningStatistics);
        oneWarningStatistics.setAlarmLevel(WarningLevelEnum.FIRST_LEVEL.getStatus());
        Map<String, Object> oneLevel = this.oneOrTwoLevelWarningCount(oneWarningStatistics);

        Integer threeLevelResult = threeLevel.get("total") == null ? 0 : (Integer) threeLevel.get("total");
        Integer twoLevelResult = twoLevel.get("total") == null ? 0 : (Integer) twoLevel.get("total");
        Integer oneLevelResult = oneLevel.get("total") == null ? 0 : (Integer) oneLevel.get("total");

        Map<String, Object> mapResult = new HashMap<>(8);
        mapResult.put("oneLevelCount", oneLevelResult);
        mapResult.put("twoLevelCount", twoLevelResult);
        mapResult.put("threeLevelCount", threeLevelResult);
        // log.info("[首页预警]统计成功，用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }

    /**
     * 三级预警统计
     *
     * @param reqWarningStatisticsVo 入参
     * @return 结果
     */
    private Map<String, Object> threeLevelWarningCount(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        // 并行执行结果
        Map<String, Object> mapResult = new HashMap<>(16);

        // 重点人
        ImportantUserStatisticsDTO importantUserStatisticsDTO = new ImportantUserStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, importantUserStatisticsDTO);
        CompletableFuture<Integer> importantUserFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceUser + "getTriggerEventCount", importantUserStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> importantUserCallbackFuture = importantUserFuture.thenAccept((result) -> {
            // log.info("[三级预警]重点人预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("importantUser", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]重点人预警查询异常，请求参数：{}，异常：", JSON.toJSONString(importantUserStatisticsDTO), e);
            return null;
        });

        // 矛盾超期预警
        OverdueEventStatisticsDTO overdueEventStatisticsDTO = new OverdueEventStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, overdueEventStatisticsDTO);
        overdueEventStatisticsDTO.setStatus(EventStatusEnum.PROCESSING.getStatus());
        CompletableFuture<Integer> overdueEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpace + "getOverdueEventCount", overdueEventStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> overdueEventCallbackFuture = overdueEventFuture.thenAccept((result) -> {
            // log.info("[三级预警]矛盾超期预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("overdueEvent", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]矛盾超期预警查询异常，请求参数：{}，异常：", JSON.toJSONString(overdueEventStatisticsDTO), e);
            return null;
        });

        // 风险因子预警
        EventRiskFactorDTO eventRiskFactor = new EventRiskFactorDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, eventRiskFactor);
        eventRiskFactor.setStatus(EventStatusEnum.PROCESSING.getStatus());
        CompletableFuture<Integer> eventRiskFactorFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCount", eventRiskFactor);
        }, threadPoolExecutor);
        CompletableFuture<Void> eventRiskFactorCallbackFuture = eventRiskFactorFuture.thenAccept((result) -> {
            // log.info("[三级预警]风险因子预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("eventRiskFactor", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]风险因子预警查询异常，请求参数：{}，异常：", JSON.toJSONString(eventRiskFactor), e);
            return null;
        });

        // 群体性事件预警
        GroupEventWarningDTO groupEventTrigger = new GroupEventWarningDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, groupEventTrigger);
        groupEventTrigger.setStatus(1);
        CompletableFuture<Integer> groupEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            groupEventTrigger.setTriggerType("2");
            return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCount", groupEventTrigger);
        }, threadPoolExecutor);
        CompletableFuture<Void> groupEventCallbackFuture = groupEventFuture.thenAccept((result) -> {
            // log.info("[三级预警]群体性事件预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("groupEvent", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]群体性事件预警查询异常，请求参数：{}，异常：", JSON.toJSONString(groupEventTrigger), e);
            return null;
        });

        // 涉众预警
        InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO = new InvolveEveryoneStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, involveEveryoneStatisticsDTO);
        //involveEveryoneStatisticsDTO.setIsInvolveEveryoneCount("0");
        involveEveryoneStatisticsDTO.setStatus("1");
        CompletableFuture<Integer> involveEveryoneFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            involveEveryoneStatisticsDTO.setTriggerType("3");
            return commonService.getObject(nameSpace + "getInvolveEveryoneCount", involveEveryoneStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> involveEveryoneCallbackFuture = involveEveryoneFuture.thenAccept((result) -> {
            // log.info("[三级预警]涉众预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("involveEveryone", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]涉众预警查询异常，请求参数：{}，异常：", JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        CompletableFuture.allOf(importantUserCallbackFuture, overdueEventCallbackFuture, eventRiskFactorCallbackFuture, groupEventCallbackFuture, involveEveryoneCallbackFuture).join();
        // log.info("[三级预警]同步后结果：{}", JSON.toJSONString(mapResult));

        Object importantUserResult = mapResult.get("importantUser");
        if (!Optional.ofNullable(importantUserResult).isPresent()) {
            mapResult.put("importantUser", 0);
            importantUserResult = 0;
        }
        Object overdueEventResult = mapResult.get("overdueEvent");
        if (!Optional.ofNullable(overdueEventResult).isPresent()) {
            mapResult.put("overdueEvent", 0);
            overdueEventResult = 0;
        }
        Object eventRiskFactorResult = mapResult.get("eventRiskFactor");
        if (!Optional.ofNullable(eventRiskFactorResult).isPresent()) {
            mapResult.put("eventRiskFactor", 0);
            eventRiskFactorResult = 0;
        }
        Object groupEventResult = mapResult.get("groupEvent");
        if (!Optional.ofNullable(groupEventResult).isPresent()) {
            mapResult.put("groupEvent", 0);
            groupEventResult = 0;
        }
        Object involveEveryoneResult = mapResult.get("involveEveryone");
        if (!Optional.ofNullable(involveEveryoneResult).isPresent()) {
            mapResult.put("involveEveryone", 0);
            involveEveryoneResult = 0;
        }
        mapResult.put("total", ((Integer) importantUserResult + (Integer) overdueEventResult + (Integer) eventRiskFactorResult + (Integer) groupEventResult + (Integer) involveEveryoneResult));
        // log.info("[三级预警]查询成功，用户：{}，统计结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }


    /**
     * 三级预警统计-历史
     *
     * @param reqWarningStatisticsVo 入参
     * @return 结果
     */
    private Map<String, Object> threeLevelWarningCountHistory(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        // 并行执行结果
        Map<String, Object> mapResult = new HashMap<>(16);

        // 重点人
        ImportantUserStatisticsDTO importantUserStatisticsDTO = new ImportantUserStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, importantUserStatisticsDTO);
        CompletableFuture<Integer> importantUserFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceUser + "getTriggerEventCountHistory", importantUserStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> importantUserCallbackFuture = importantUserFuture.thenAccept((result) -> {
            // log.info("[三级预警历史]重点人预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("importantUser", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]重点人预警查询异常，请求参数：{}，异常：", JSON.toJSONString(importantUserStatisticsDTO), e);
            return null;
        });

        // 矛盾超期预警
        OverdueEventStatisticsDTO overdueEventStatisticsDTO = new OverdueEventStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, overdueEventStatisticsDTO);
        overdueEventStatisticsDTO.setStatus(EventStatusEnum.COMPLETED.getStatus());
        CompletableFuture<Integer> overdueEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpace + "getOverdueEventCountHistory", overdueEventStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> overdueEventCallbackFuture = overdueEventFuture.thenAccept((result) -> {
            // log.info("[三级预警历史]矛盾超期预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("overdueEvent", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]矛盾超期预警查询异常，请求参数：{}，异常：", JSON.toJSONString(overdueEventStatisticsDTO), e);
            return null;
        });

        // 风险因子预警
        EventRiskFactorDTO eventRiskFactor = new EventRiskFactorDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, eventRiskFactor);
        CompletableFuture<Integer> eventRiskFactorFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCountHistory", eventRiskFactor);
        }, threadPoolExecutor);
        CompletableFuture<Void> eventRiskFactorCallbackFuture = eventRiskFactorFuture.thenAccept((result) -> {
            // log.info("[三级预警历史]风险因子预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("eventRiskFactor", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]风险因子预警查询异常，请求参数：{}，异常：", JSON.toJSONString(eventRiskFactor), e);
            return null;
        });

        // 群体性事件预警
        GroupEventWarningDTO groupEventTrigger = new GroupEventWarningDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, groupEventTrigger);
        CompletableFuture<Integer> groupEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            groupEventTrigger.setTriggerType("2");
            return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCountHistory", groupEventTrigger);
        }, threadPoolExecutor);
        CompletableFuture<Void> groupEventCallbackFuture = groupEventFuture.thenAccept((result) -> {
            // log.info("[三级预警历史]群体性事件预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("groupEvent", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]群体性事件预警查询异常，请求参数：{}，异常：", JSON.toJSONString(groupEventTrigger), e);
            return null;
        });

        // 涉众预警
        InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO = new InvolveEveryoneStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, involveEveryoneStatisticsDTO);
        CompletableFuture<Integer> involveEveryoneFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            involveEveryoneStatisticsDTO.setTriggerType("3");
            return commonService.getObject(nameSpace + "getInvolveEveryoneCountHistory", involveEveryoneStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> involveEveryoneCallbackFuture = involveEveryoneFuture.thenAccept((result) -> {
            // log.info("[三级预警历史]涉众预警查询成功，查询用户：{}，结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), result);
            mapResult.put("involveEveryone", result);
        }).exceptionally((e) -> {
            log.error("[三级预警]涉众预警查询异常，请求参数：{}，异常：", JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        CompletableFuture.allOf(importantUserCallbackFuture, overdueEventCallbackFuture, eventRiskFactorCallbackFuture, groupEventCallbackFuture, involveEveryoneCallbackFuture).join();
        // log.info("[三级预警历史]同步后结果：{}", JSON.toJSONString(mapResult));

        Object importantUserResult = mapResult.get("importantUser");
        if (!Optional.ofNullable(importantUserResult).isPresent()) {
            mapResult.put("importantUser", 0);
            importantUserResult = 0;
        }
        Object overdueEventResult = mapResult.get("overdueEvent");
        if (!Optional.ofNullable(overdueEventResult).isPresent()) {
            mapResult.put("overdueEvent", 0);
            overdueEventResult = 0;
        }
        Object eventRiskFactorResult = mapResult.get("eventRiskFactor");
        if (!Optional.ofNullable(eventRiskFactorResult).isPresent()) {
            mapResult.put("eventRiskFactor", 0);
            eventRiskFactorResult = 0;
        }
        Object groupEventResult = mapResult.get("groupEvent");
        if (!Optional.ofNullable(groupEventResult).isPresent()) {
            mapResult.put("groupEvent", 0);
            groupEventResult = 0;
        }
        Object involveEveryoneResult = mapResult.get("involveEveryone");
        if (!Optional.ofNullable(involveEveryoneResult).isPresent()) {
            mapResult.put("involveEveryone", 0);
            involveEveryoneResult = 0;
        }
        mapResult.put("total", ((Integer) importantUserResult + (Integer) overdueEventResult + (Integer) eventRiskFactorResult + (Integer) groupEventResult + (Integer) involveEveryoneResult));
        // log.info("[三级预警历史]查询成功，用户：{}，统计结果：{}", reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }

    @ApiOperation(value = "综合研判预警-重点人预警历史", desc = "综合研判预警-重点人预警历史")
    public Page<Map<String, String>> findImportantUserHistoryPageList(Page<Map<String, String>> page) {
        return commonService.findPage(nameSpaceUser + "findImportantUserHistoryPageList", page);
    }


    /**
     * 一二级预警统计
     *
     * @param reqWarningStatisticsVo 入参
     * @return 结果
     */
    private Map<String, Object> oneOrTwoLevelWarningCount(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        // 并行执行结果
        Map<String, Object> mapResult = new HashMap<>(8);
        String title = WarningLevelEnum.TWO_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel()) ? WarningLevelEnum.TWO_LEVEL.getInfo() : WarningLevelEnum.FIRST_LEVEL.getInfo();

        // 风险因子预警
        EventRiskFactorDTO eventRiskFactor = new EventRiskFactorDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, eventRiskFactor);
        eventRiskFactor.setStatus(EventStatusEnum.PROCESSING.getStatus());
        CompletableFuture<Integer> eventRiskFactorFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCount", eventRiskFactor);
        }, threadPoolExecutor);
        CompletableFuture<Void> eventRiskFactorCallbackFuture = eventRiskFactorFuture.thenAccept((result) -> {
            // log.info("[{}]风险因子预警查询成功：{}", title, result);
            mapResult.put("eventRiskFactor", result);
        }).exceptionally((e) -> {
            log.error("[{}]风险因子预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(eventRiskFactor), e);
            return null;
        });

        // 群体性事件预警
        GroupEventWarningDTO groupEventWarning = new GroupEventWarningDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, groupEventWarning);
        groupEventWarning.setStatus(1);
        CompletableFuture<Integer> groupEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            groupEventWarning.setTriggerType("2");
            return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCount", groupEventWarning);
        }, threadPoolExecutor);
        CompletableFuture<Void> groupEventCallbackFuture = groupEventFuture.thenAccept((result) -> {
            // log.info("[{}]群体性事件预警查询成功：{}", title, result);
            mapResult.put("groupEvent", result);
        }).exceptionally((e) -> {
            log.error("[{}]群体性事件预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(groupEventWarning), e);
            return null;
        });

        // 重点领域预警
        FocusAreasStatisticsDTO focusAreas = new FocusAreasStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, focusAreas);
        CompletableFuture<Integer> focusAreasFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceFocusAreas + "getFocusAreasCount", focusAreas);
        }, threadPoolExecutor);
        CompletableFuture<Void> focusAreasCallbackFuture = focusAreasFuture.thenAccept((result) -> {
            // log.info("[{}]重点领域预警查询成功：{}", title, result);
            mapResult.put("focusAreas", result);
        }).exceptionally((e) -> {
            log.error("[{}]重点领域预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(focusAreas), e);
            return null;
        });

        // 涉众预警
        String involveEveryoneCount = WarningLevelEnum.TWO_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel()) ? "1" : "2";
        InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO = new InvolveEveryoneStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, involveEveryoneStatisticsDTO);
        //involveEveryoneStatisticsDTO.setIsInvolveEveryoneCount(involveEveryoneCount);
        involveEveryoneStatisticsDTO.setStatus("1");
        CompletableFuture<Integer> involveEveryoneFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            involveEveryoneStatisticsDTO.setTriggerType("3");
            return commonService.getObject(nameSpace + "getInvolveEveryoneCount", involveEveryoneStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> involveEveryoneCallbackFuture = involveEveryoneFuture.thenAccept((result) -> {
            // log.info("[{}]涉众预警查询成功：{}", title, result);
            mapResult.put("involveEveryone", result);
        }).exceptionally((e) -> {
            log.error("[{}]涉众预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });


        CompletableFuture<Void> allQuery = CompletableFuture.allOf(eventRiskFactorCallbackFuture, groupEventCallbackFuture, focusAreasCallbackFuture, involveEveryoneCallbackFuture);
        CompletableFuture<Map<String, Object>> future = allQuery.thenApply((result) -> mapResult).exceptionally((e) -> {
            log.error("[{}]获取预警数量异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        future.join();
        // log.info("[{}]同步后结果：{}", title, JSON.toJSONString(mapResult));

        Object eventRiskFactorResult = mapResult.get("eventRiskFactor");
        if (!Optional.ofNullable(eventRiskFactorResult).isPresent()) {
            mapResult.put("eventRiskFactor", 0);
            eventRiskFactorResult = 0;
        }
        Object groupEventResult = mapResult.get("groupEvent");
        if (!Optional.ofNullable(groupEventResult).isPresent()) {
            mapResult.put("groupEvent", 0);
            groupEventResult = 0;
        }
        Object involveEveryoneResult = mapResult.get("involveEveryone");
        if (!Optional.ofNullable(involveEveryoneResult).isPresent()) {
            mapResult.put("involveEveryone", 0);
            involveEveryoneResult = 0;
        }
        Object focusAreasResult = mapResult.get("focusAreas");
        if (!Optional.ofNullable(focusAreasResult).isPresent()) {
            mapResult.put("focusAreas", 0);
            focusAreasResult = 0;
        }
        mapResult.put("total", ((Integer) eventRiskFactorResult + (Integer) groupEventResult + (Integer) involveEveryoneResult + (Integer) focusAreasResult));
        // log.info("[{}]查询成功，用户：{}，统计结果：{}", title, reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }

    /**
     * 一二级预警统计-历史
     *
     * @param reqWarningStatisticsVo 入参
     * @return 结果
     */
    private Map<String, Object> oneOrTwoLevelWarningCountHistory(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        // 并行执行结果
        Map<String, Object> mapResult = new HashMap<>(8);
        String title = WarningLevelEnum.TWO_LEVEL.getStatus().equals(reqWarningStatisticsVo.getAlarmLevel()) ? WarningLevelEnum.TWO_LEVEL.getInfo() : WarningLevelEnum.FIRST_LEVEL.getInfo();

        // 风险因子预警
        EventRiskFactorDTO eventRiskFactor = new EventRiskFactorDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, eventRiskFactor);
        CompletableFuture<Integer> eventRiskFactorFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCountHistory", eventRiskFactor);
        }, threadPoolExecutor);
        CompletableFuture<Void> eventRiskFactorCallbackFuture = eventRiskFactorFuture.thenAccept((result) -> {
            // log.info("[{}]风险因子预警历史查询成功：{}", title, result);
            mapResult.put("eventRiskFactor", result);
        }).exceptionally((e) -> {
            log.error("[{}]风险因子预警历史查询异常，请求参数：{}，异常：", title, JSON.toJSONString(eventRiskFactor), e);
            return null;
        });

        // 群体性事件预警
        GroupEventWarningDTO groupEventWarning = new GroupEventWarningDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, groupEventWarning);
        CompletableFuture<Integer> groupEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            groupEventWarning.setTriggerType("2");
            return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCountHistory", groupEventWarning);
        }, threadPoolExecutor);
        CompletableFuture<Void> groupEventCallbackFuture = groupEventFuture.thenAccept((result) -> {
            // log.info("[{}]群体性事件预警历史查询成功：{}", title, result);
            mapResult.put("groupEvent", result);
        }).exceptionally((e) -> {
            log.error("[{}]群体性事件预警历史查询异常，请求参数：{}，异常：", title, JSON.toJSONString(groupEventWarning), e);
            return null;
        });

        // 重点领域预警
        // TODO 丁畅增加逻辑处理
        FocusAreasStatisticsDTO focusAreas = new FocusAreasStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, focusAreas);
        CompletableFuture<Integer> focusAreasFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(areasNameSpaceHistory + "getFocusAreasCount", focusAreas);
        }, threadPoolExecutor);
        CompletableFuture<Void> focusAreasCallbackFuture = focusAreasFuture.thenAccept((result) -> {
            // log.info("[{}]重点领域预警历史查询成功：{}", title, result);
            mapResult.put("focusAreas", result);
        }).exceptionally((e) -> {
            log.error("[{}]重点领域预警历史查询异常，请求参数：{}，异常：", title, JSON.toJSONString(focusAreas), e);
            return null;
        });

        // 涉众预警
        InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO = new InvolveEveryoneStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, involveEveryoneStatisticsDTO);
        CompletableFuture<Integer> involveEveryoneFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            involveEveryoneStatisticsDTO.setTriggerType("3");
            return commonService.getObject(nameSpace + "getInvolveEveryoneCountHistory", involveEveryoneStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> involveEveryoneCallbackFuture = involveEveryoneFuture.thenAccept((result) -> {
            // log.info("[{}]涉众预警历史查询成功：{}", title, result);
            mapResult.put("involveEveryone", result);
        }).exceptionally((e) -> {
            log.error("[{}]涉众预警历史查询异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });
        CompletableFuture<Void> allQuery = CompletableFuture.allOf(eventRiskFactorCallbackFuture, groupEventCallbackFuture, focusAreasCallbackFuture, involveEveryoneCallbackFuture);
        CompletableFuture<Map<String, Object>> future = allQuery.thenApply((result) -> mapResult).exceptionally((e) -> {
            log.error("[{}]获取预警历史数量异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        future.join();

        // 结果处理
        Object eventRiskFactorResult = mapResult.get("eventRiskFactor");
        if (!Optional.ofNullable(eventRiskFactorResult).isPresent()) {
            mapResult.put("eventRiskFactor", 0);
            eventRiskFactorResult = 0;
        }
        Object groupEventResult = mapResult.get("groupEvent");
        if (!Optional.ofNullable(groupEventResult).isPresent()) {
            mapResult.put("groupEvent", 0);
            groupEventResult = 0;
        }
        Object involveEveryoneResult = mapResult.get("involveEveryone");
        if (!Optional.ofNullable(involveEveryoneResult).isPresent()) {
            mapResult.put("involveEveryone", 0);
            involveEveryoneResult = 0;
        }
        Object focusAreasResult = mapResult.get("focusAreas");
        if (!Optional.ofNullable(focusAreasResult).isPresent()) {
            mapResult.put("focusAreas", 0);
            focusAreasResult = 0;
        }
        mapResult.put("total", ((Integer) eventRiskFactorResult + (Integer) groupEventResult + (Integer) involveEveryoneResult + (Integer) focusAreasResult));
        // log.info("[{}]历史查询成功，用户：{}，统计结果：{}", title, reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }

    /**
     * 重点事统计
     *
     * @param reqWarningStatisticsVo 入参
     * @return 结果
     */
    private Map<String, Object> importantEventCount(ReqWarningStatisticsVo reqWarningStatisticsVo) {

        // 并行执行结果
        Map<String, Object> mapResult = new HashMap<>(8);
        // 根据是否重点事，查询不同事件办理状态（重点事查询办理中，历史重点事查询办结）
        String eventStatus = reqWarningStatisticsVo.getImportantEventStatus() == 2 ? "1" : "2";
        String title = reqWarningStatisticsVo.getImportantEventStatus() == 1 ? "重点事" : "非重点事";

        // 矛盾超期预警
        OverdueEventStatisticsDTO overdueEventStatisticsDTO = new OverdueEventStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, overdueEventStatisticsDTO);
        overdueEventStatisticsDTO.setStatus(eventStatus);
        CompletableFuture<Integer> overdueEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpace + "getOverdueEventCount", overdueEventStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> overdueEventCallbackFuture = overdueEventFuture.thenAccept((result) -> {
            // log.info("[{}]矛盾超期预警查询成功：{}", title, result);
            mapResult.put("overdueEvent", result);
        }).exceptionally((e) -> {
            log.error("[{}]矛盾超期预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(overdueEventStatisticsDTO), e);
            return null;
        });

        // 风险因子预警
        EventRiskFactorDTO eventRiskFactor = new EventRiskFactorDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, eventRiskFactor);
        eventRiskFactor.setStatus(eventStatus);
        CompletableFuture<Integer> eventRiskFactorFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            return commonService.getObject(nameSpaceRiskFactor + "getEventRiskFactorCount", eventRiskFactor);
        }, threadPoolExecutor);
        CompletableFuture<Void> eventRiskFactorCallbackFuture = eventRiskFactorFuture.thenAccept((result) -> {
            // log.info("[{}]风险因子预警查询成功：{}", title, result);
            mapResult.put("eventRiskFactor", result);
        }).exceptionally((e) -> {
            log.error("[{}]风险因子预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(eventRiskFactor), e);
            return null;
        });

        // 群体性事件预警
        GroupEventWarningDTO groupEventWarning = new GroupEventWarningDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, groupEventWarning);
        // 重点事查触发状态，历史重点事查解除
        groupEventWarning.setStatus(reqWarningStatisticsVo.getImportantEventStatus());
        CompletableFuture<Integer> groupEventFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            groupEventWarning.setTriggerType("2");
            return commonService.getObject(nameSpaceWaiting + "getGroupEventWarningCount", groupEventWarning);
        }, threadPoolExecutor);
        CompletableFuture<Void> groupEventCallbackFuture = groupEventFuture.thenAccept((result) -> {
            // log.info("[{}]群体性事件预警查询成功：{}", title, result);
            mapResult.put("groupEvent", result);
        }).exceptionally((e) -> {
            log.error("[{}]群体性事件预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(groupEventWarning), e);
            return null;
        });

        // 涉众预警
        InvolveEveryoneStatisticsDTO involveEveryoneStatisticsDTO = new InvolveEveryoneStatisticsDTO();
        BeanUtils.copyProperties(reqWarningStatisticsVo, involveEveryoneStatisticsDTO);
        involveEveryoneStatisticsDTO.setStatus(String.valueOf(reqWarningStatisticsVo.getImportantEventStatus()));
        CompletableFuture<Integer> involveEveryoneFuture = CompletableFuture.supplyAsync(() -> {
            DataSourceSwitcher.setSlave();
            DataSourceSwitcher.setSharding("all");
            involveEveryoneStatisticsDTO.setTriggerType("3");
            return commonService.getObject(nameSpace + "getInvolveEveryoneCount", involveEveryoneStatisticsDTO);
        }, threadPoolExecutor);
        CompletableFuture<Void> involveEveryoneCallbackFuture = involveEveryoneFuture.thenAccept((result) -> {
            // log.info("[{}]涉众预警查询成功：{}", title, result);
            mapResult.put("involveEveryone", result);
        }).exceptionally((e) -> {
            log.error("[{}]涉众预警查询异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        CompletableFuture<Void> allQuery = CompletableFuture.allOf(overdueEventCallbackFuture, eventRiskFactorCallbackFuture, groupEventCallbackFuture, involveEveryoneCallbackFuture);
        CompletableFuture<Map<String, Object>> future = allQuery.thenApply((result) -> mapResult).exceptionally((e) -> {
            log.error("[{}]获取数量异常，请求参数：{}，异常：", title, JSON.toJSONString(involveEveryoneStatisticsDTO), e);
            return null;
        });

        future.join();
        // log.info("[{}]同步后结果：{}", title, JSON.toJSONString(mapResult));

        Object overdueEventResult = mapResult.get("overdueEvent");
        if (!Optional.ofNullable(overdueEventResult).isPresent()) {
            mapResult.put("overdueEvent", 0);
            overdueEventResult = 0;
        }
        Object eventRiskFactorResult = mapResult.get("eventRiskFactor");
        if (!Optional.ofNullable(eventRiskFactorResult).isPresent()) {
            mapResult.put("eventRiskFactor", 0);
            eventRiskFactorResult = 0;
        }
        Object groupEventResult = mapResult.get("groupEvent");
        if (!Optional.ofNullable(groupEventResult).isPresent()) {
            mapResult.put("groupEvent", 0);
            groupEventResult = 0;
        }
        Object involveEveryoneResult = mapResult.get("involveEveryone");
        if (!Optional.ofNullable(involveEveryoneResult).isPresent()) {
            mapResult.put("involveEveryone", 0);
            involveEveryoneResult = 0;
        }
        mapResult.put("total", ((Integer) overdueEventResult + (Integer) eventRiskFactorResult + (Integer) groupEventResult + (Integer) involveEveryoneResult));
        // log.info("[{}]查询成功，用户：{}，统计结果：{}", title, reqWarningStatisticsVo.get_user().getLoginName(), JSON.toJSONString(mapResult));
        return mapResult;
    }

}
