import auth from '@/plugins/auth'
import router, { constantRoutes, dynamicRoutes } from '@/router'
import { getRouters } from '@/api/menu'
import Layout from '@/layout/index'
import ParentView from '@/components/ParentView'
import InnerLink from '@/layout/components/InnerLink'

// 匹配views里面所有的.vue文件
const modules = import.meta.glob('./../../views/**/*.vue')

const usePermissionStore = defineStore('permission', {
  state: () => ({
    routes: [],
    addRoutes: [],
    defaultRoutes: [],
    topbarRouters: [],
    sidebarRouters: [],
  }),
  actions: {
    setRoutes(routes) {
      this.addRoutes = routes
      this.routes = constantRoutes.concat(routes)
    },
    setDefaultRoutes(routes) {
      this.defaultRoutes = constantRoutes.concat(routes)
    },
    setTopbarRoutes(routes) {
      this.topbarRouters = routes
    },
    setSidebarRouters(routes) {
      this.sidebarRouters = routes
    },
    generateRoutes(roles) {
      return new Promise((resolve) => {
        // 向后端请求路由数据
        const navList = JSON.parse(sessionStorage.getItem('navList'))
        const data = transformRoutes(navList)
        const sdata = JSON.parse(JSON.stringify(data))
        const rdata = JSON.parse(JSON.stringify(data))
        const defaultData = JSON.parse(JSON.stringify(data))
        const sidebarRoutes = filterAsyncRouter(sdata)
        const rewriteRoutes = filterAsyncRouter(rdata, false, true)
        const defaultRoutes = filterAsyncRouter(defaultData)
        const asyncRoutes = filterDynamicRoutes(dynamicRoutes)
        asyncRoutes.forEach((route) => {
          router.addRoute(route)
        })
        this.setRoutes(rewriteRoutes)
        this.setSidebarRouters(constantRoutes.concat(sidebarRoutes))
        this.setDefaultRoutes(sidebarRoutes)
        this.setTopbarRoutes(defaultRoutes)
        resolve(rewriteRoutes)
      })
    },
  },
})

// 遍历后台传来的路由字符串，转换为组件对象
function filterAsyncRouter(asyncRouterMap, lastRouter = false, type = false) {
  return asyncRouterMap.filter((route) => {
    if (type && route.children) {
      route.children = filterChildren(route.children)
    }
    if (route.component) {
      // Layout ParentView 组件特殊处理
      if (route.component === 'Layout') {
        route.component = Layout
      } else if (route.component === 'ParentView') {
        route.component = ParentView
      } else if (route.component === 'InnerLink') {
        route.component = InnerLink
      } else {
        route.component = loadView(route.component)
      }
    }
    if (route.children != null && route.children && route.children.length) {
      route.children = filterAsyncRouter(route.children, route, type)
    } else {
      delete route['children']
      delete route['redirect']
    }
    return true
  })
}

function filterChildren(childrenMap, lastRouter = false) {
  var children = []
  childrenMap.forEach((el) => {
    el.path = lastRouter ? lastRouter.path + '/' + el.path : el.path
    if (el.children && el.children.length && el.component === 'ParentView') {
      children = children.concat(filterChildren(el.children, el))
    } else {
      children.push(el)
    }
  })
  return children
}

// 动态路由遍历，验证是否具备权限
export function filterDynamicRoutes(routes) {
  const res = []
  routes.forEach((route) => {
    if (route.permissions) {
      if (auth.hasPermiOr(route.permissions)) {
        res.push(route)
      }
    } else if (route.roles) {
      if (auth.hasRoleOr(route.roles)) {
        res.push(route)
      }
    }
  })
  return res
}

export const loadView = (view) => {
  let res
  for (const path in modules) {
    const dir = path.split('views/')[1].split('.vue')[0]
    if (dir === view) {
      res = () => modules[path]()
    }
  }
  return res
}

// 格式化路由数据结构
function transformRoutes(originalRoutes) {
  return originalRoutes.map((routeItem) => {
    const hasChildren = routeItem.submenu && routeItem.submenu.length > 0
    // 基础转换 - 所有路由都使用Layout作为父组件
    const transformedRoute = {
      path: routeItem.url,
      name: routeItem.menuname.replace(/\s+/g, '') + new Date().getTime(),
      hidden: false,
      type: routeItem.type,
      component: 'Layout', // 所有路由都使用Layout
      redirect: 'noRedirect',
      meta: {
        title: routeItem.menuname,
        icon: routeItem.icon,
        noCache: false,
        link: null,
        singleLevel: !hasChildren,
      },
      children: [], // 初始化children数组
    }

    // 处理有子路由的情况
    if (hasChildren) {
      transformedRoute.alwaysShow = true
      transformedRoute.children = routeItem.submenu.map((child) => ({
        path: child.url.startsWith('/') ? child.url.substring(1) : child.url,
        name: child.menuname.replace(/\s+/g, ''),
        hidden: false,
        component: `${routeItem.url.replace(/^\//, '')}/${child.url}/index`,
        meta: {
          title: child.menuname,
          icon: child.icon,
          noCache: false,
          link: null,
        },
      }))
    }
    // 处理没有子路由的情况 - 将自己作为唯一的子路由
    else {
      transformedRoute.children.push({
        path: '',
        name: `${routeItem.menuname.replace(/\s+/g, '')}`,
        hidden: false,
        component: `${routeItem.url.replace(/^\//, '')}/index`,
        meta: {
          title: routeItem.menuname,
          icon: routeItem.icon,
          noCache: false,
          link: null,
        },
      })
    }
    return transformedRoute
  })
}

export default usePermissionStore
