package com.ruoyi.inventory.service.impl;

import java.util.Collections;
import java.util.List;
import com.ruoyi.common.utils.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.util.ArrayList;
import java.util.UUID;

import com.ruoyi.common.utils.StringUtils;
import org.springframework.transaction.annotation.Transactional;
import com.ruoyi.inventory.domain.ReturnOrderItems;
import com.ruoyi.inventory.mapper.ReturnOrdersMapper;
import com.ruoyi.inventory.domain.ReturnOrders;
import com.ruoyi.inventory.service.IReturnOrdersService;

/**
 * 退库单主Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-12-03
 */
@Service
public class ReturnOrdersServiceImpl implements IReturnOrdersService 
{
    @Autowired
    private ReturnOrdersMapper returnOrdersMapper;

    /**
     * 查询退库单主
     * 
     * @param id 退库单主主键
     * @return 退库单主
     */
    @Override
    public ReturnOrders selectReturnOrdersById(String id)
    {
        return returnOrdersMapper.selectReturnOrdersById(id);
    }

    /**
     * 查询退库单主列表
     * 
     * @param returnOrders 退库单主
     * @return 退库单主
     */
    @Override
    public List<ReturnOrders> selectReturnOrdersList(ReturnOrders returnOrders)
    {
        return returnOrdersMapper.selectReturnOrdersList(returnOrders);
    }

    /**
     * 新增退库单主
     * 
     * @param returnOrders 退库单主
     * @return 结果
     */
    @Transactional
    @Override
    public int insertReturnOrders(ReturnOrders returnOrders)
    {
        if (StringUtils.isEmpty(returnOrders.getId())) {
            returnOrders.setId(UUID.randomUUID().toString());
        }
        returnOrders.setCreateTime(DateUtils.getNowDate());
        if (returnOrders.getIsUsed() == null) {
            returnOrders.setIsUsed(1);
        }
        int rows = returnOrdersMapper.insertReturnOrders(returnOrders);
        insertReturnOrderItems(returnOrders);
        return rows;
    }

    /**
     * 修改退库单主
     * 
     * @param returnOrders 退库单主
     * @return 结果
     */
    @Transactional
    @Override
    public int updateReturnOrders(ReturnOrders returnOrders)
    {
        returnOrders.setUpdateTime(DateUtils.getNowDate());
        returnOrdersMapper.deleteReturnOrderItemsByOrderId(returnOrders.getId());
        insertReturnOrderItems(returnOrders);
        return returnOrdersMapper.updateReturnOrders(returnOrders);
    }

    /**
     * 批量删除退库单主
     * 
     * @param ids 需要删除的退库单主主键
     * @return 结果
     */
    @Transactional
    @Override
    public int deleteReturnOrdersByIds(String[] ids)
    {
        returnOrdersMapper.deleteReturnOrderItemsByOrderIds(ids);
        return returnOrdersMapper.deleteReturnOrdersByIds(ids);
    }

    /**
     * 删除退库单主信息
     * 
     * @param id 退库单主主键
     * @return 结果
     */
    @Transactional
    @Override
    public int deleteReturnOrdersById(String id)
    {
        ReturnOrders returnOrders = returnOrdersMapper.selectReturnOrdersById(id);
        if (returnOrders != null && returnOrders.getOrderId() != null) {
            returnOrdersMapper.deleteReturnOrderItemsByOrderId(returnOrders.getOrderId());
        }
        return returnOrdersMapper.deleteReturnOrdersById(id);
    }

    @Override
    public List<ReturnOrderItems> selectExportList(ReturnOrders returnOrders) {
        return Collections.emptyList();
    }

    @Override
    public int confirmReturnOrdersByIds(String[] ids) {
        // todo 调用库存插入方法
        return returnOrdersMapper.confirmReturnOrdersByIds(ids);
    }

    /**
     * 新增退库单明细信息
     * 
     * @param returnOrders 退库单主对象
     */
    public void insertReturnOrderItems(ReturnOrders returnOrders)
    {
        List<ReturnOrderItems> returnOrderItemsList = returnOrders.getReturnOrderItemsList();
        String returnOrderId = returnOrders.getId();
        String orderId = returnOrders.getOrderId();
        if (StringUtils.isNotNull(returnOrderItemsList))
        {
            List<ReturnOrderItems> list = new ArrayList<ReturnOrderItems>();
            for (ReturnOrderItems returnOrderItems : returnOrderItemsList)
            {
                if (StringUtils.isEmpty(returnOrderItems.getId())) {
                    returnOrderItems.setId(UUID.randomUUID().toString());
                }
                returnOrderItems.setReturnOrderId(returnOrderId);
                returnOrderItems.setOrderId(orderId);
                returnOrderItems.setCreateTime(DateUtils.getNowDate());
                if (returnOrderItems.getIsUsed() == null) {
                    returnOrderItems.setIsUsed(1);
                }
                // 自动计算金额
                if (returnOrderItems.getActualQuantity() != null && returnOrderItems.getUnitPrice() != null) {
                    returnOrderItems.setTotalAmount(returnOrderItems.getActualQuantity() * returnOrderItems.getUnitPrice());
                }
                list.add(returnOrderItems);
            }
            if (list.size() > 0)
            {
                returnOrdersMapper.batchReturnOrderItems(list);
            }
        }
    }
}

