package com.ruoyi.inventory.controller;

import java.util.List;
import javax.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.inventory.domain.OutboundOrderItems;

import com.ruoyi.inventory.service.IOutboundOrderItemsService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 出库单明细Controller
 * 
 * @author ruoyi
 * @date 2025-12-03
 */
@RestController
@RequestMapping("/inventory/items")
public class OutboundOrderItemsController extends BaseController
{
    @Autowired
    private IOutboundOrderItemsService outboundOrderItemsService;

    /**
     * 查询出库单明细列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:list')")
    @GetMapping("/list")
    public TableDataInfo list(OutboundOrderItems outboundOrderItems)
    {
        startPage();
        List<OutboundOrderItems> list = outboundOrderItemsService.selectOutboundOrderItemsList(outboundOrderItems);
        return getDataTable(list);
    }

    /**
     * 导出出库单明细列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:export')")
    @Log(title = "出库单明细", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, OutboundOrderItems outboundOrderItems)
    {
        List<OutboundOrderItems> list = outboundOrderItemsService.selectOutboundOrderItemsList(outboundOrderItems);
        ExcelUtil<OutboundOrderItems> util = new ExcelUtil<OutboundOrderItems>(OutboundOrderItems.class);
        util.exportExcel(response, list, "出库单明细数据");
    }

    /**
     * 获取出库单明细详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        return success(outboundOrderItemsService.selectOutboundOrderItemsById(id));
    }

    /**
     * 新增出库单明细
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:add')")
    @Log(title = "出库单明细", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody OutboundOrderItems outboundOrderItems)
    {
        try {
            // 调用业务层校验+插入逻辑
            outboundOrderItemsService.insertOutboundOrderItems(outboundOrderItems);
            // 校验通过：返回成功（若依标准成功响应）
            return success("新增出库单明细成功");
        } catch (Exception e) {
            // 校验失败：捕获异常，返回"库存被修改请重新确认"（前端弹窗用）
            logger.error("新增出库单明细失败", e);
            return error("库存被修改请重新确认");
        }
    }

    /**
     * 修改出库单明细
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:edit')")
    @Log(title = "出库单明细", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody OutboundOrderItems outboundOrderItems)
    {
        try {
            // 调用业务层校验+插入逻辑
            outboundOrderItemsService.updateOutboundOrderItems(outboundOrderItems);
            // 校验通过：返回成功（若依标准成功响应）
            return success("新增出库单明细成功");
        } catch (Exception e) {
            // 校验失败：捕获异常，返回"库存被修改请重新确认"（前端弹窗用）
            logger.error("新增出库单明细失败", e);
            return error("库存被修改请重新确认");
        }
    }


    /**
     * 删除出库单明细
     */
    @PreAuthorize("@ss.hasPermi('inventory:items:remove')")
    @Log(title = "出库单明细", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(outboundOrderItemsService.deleteOutboundOrderItemsByIds(ids));
    }
}
