package com.ruoyi.inventory.service.impl;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.ruoyi.common.exception.ServiceException;
import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.inventory.mapper.OwnersMapper;
import com.ruoyi.inventory.domain.Owners;
import com.ruoyi.inventory.service.IOwnersService;

/**
 * 货主信息Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-11-28
 */
@Service
public class OwnersServiceImpl implements IOwnersService 
{
    @Autowired
    private OwnersMapper ownersMapper;

    /**
     * 查询货主信息
     * 
     * @param id 货主信息主键
     * @return 货主信息
     */
    @Override
    public Owners selectOwnersById(String id)
    {
        return ownersMapper.selectOwnersById(id);
    }

    /**
     * 查询货主信息列表
     * 
     * @param owners 货主信息
     * @return 货主信息
     */
    @Override
    public List<Owners> selectOwnersList(Owners owners)
    {
        return ownersMapper.selectOwnersList(owners);
    }

    /**
     * 新增货主信息
     * 
     * @param owners 货主信息
     * @return 结果
     */
    @Override
    public int insertOwners(Owners owners)
    {
        Long userId = SecurityUtils.getUserId();
        String operId = userId.toString();
        Date now = DateUtils.getNowDate();
        // 填充创建人、创建时间、修改人、修改时间
        owners.setCreateBy(operId);
        owners.setCreateTime(now);
        owners.setUpdateBy(operId);
        owners.setUpdateTime(now);
        // 填充创建用户编码和更新用户编码
        owners.setCreateUserCode(operId);
        owners.setUpdateUserCode(operId);
        // 设置默认值
        if (owners.getIsActive() == null)
        {
            owners.setIsActive(1L); // 默认激活
        }
        if (owners.getIsUsed() == null)
        {
            owners.setIsUsed(0L); // 默认未删除
        }
        if (owners.getSortNo() == null)
        {
            owners.setSortNo(0L); // 默认排序号
        }
        return ownersMapper.insertOwners(owners);
    }

    @Override
    public String importOwners(List<Owners> ownersList, Boolean isUpdateSupport, String operName)
    {
        if (StringUtils.isNull(ownersList) || ownersList.size() == 0)
        {
            throw new ServiceException("导入用户数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        Date now = DateUtils.getNowDate();
        // 获取当前登录用户ID
        Long userId = SecurityUtils.getUserId();
        String operId = userId.toString();
        for (Owners owners : ownersList)
        {
            try
            {
                owners.setId(UUID.randomUUID().toString());
                // 填充创建人、创建时间、修改人、修改时间
                owners.setCreateBy(operId);
                owners.setCreateTime(now);
                owners.setUpdateBy(operId);
                owners.setUpdateTime(now);
                // 填充创建用户编码和更新用户编码
                owners.setCreateUserCode(operId);
                owners.setUpdateUserCode(operId);
                // 设置默认值
                if (owners.getIsActive() == null)
                {
                    owners.setIsActive(1L); // 默认激活
                }
                if (owners.getIsUsed() == null)
                {
                    owners.setIsUsed(0L); // 默认未删除
                }
                if (owners.getSortNo() == null)
                {
                    owners.setSortNo(0L); // 默认排序号
                }
                ownersMapper.insertOwners(owners);
                successNum++;
                successMsg.append("<br/>" + successNum + "、用户 " + owners.getOwnerName() + " 导入成功");
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、账号 " + owners.getOwnerName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());

            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确");
            throw new ServiceException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 修改货主信息
     * 
     * @param owners 货主信息
     * @return 结果
     */
    @Override
    public int updateOwners(Owners owners)
    {
        Long userId = SecurityUtils.getUserId();
        String operId = userId.toString();
        Date now = DateUtils.getNowDate();
        // 填充修改人、修改时间和更新用户编码
        owners.setUpdateBy(operId);
        owners.setUpdateTime(now);
        owners.setUpdateUserCode(operId);
        return ownersMapper.updateOwners(owners);
    }

    /**
     * 批量删除货主信息
     * 
     * @param ids 需要删除的货主信息主键
     * @return 结果
     */
    @Override
    public int deleteOwnersByIds(String[] ids)
    {
        return ownersMapper.deleteOwnersByIds(ids);
    }

    /**
     * 删除货主信息信息
     * 
     * @param id 货主信息主键
     * @return 结果
     */
    @Override
    public int deleteOwnersById(String id)
    {
        return ownersMapper.deleteOwnersById(id);
    }
}
