package com.ruoyi.inventory.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import com.ruoyi.common.core.domain.TreeSelect;
import com.ruoyi.common.core.domain.entity.Materials;
import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.spring.SpringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.inventory.mapper.MaterialsCategoryMapper;
import com.ruoyi.common.core.domain.entity.MaterialsCategory;
import com.ruoyi.inventory.service.IMaterialsCategoryService;
import com.ruoyi.inventory.service.IMaterialsService;

/**
 * 物料分类Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-11-28
 */
@Service
public class MaterialsCategoryServiceImpl implements IMaterialsCategoryService 
{
    @Autowired
    private MaterialsCategoryMapper materialsCategoryMapper;
    
    @Autowired
    private IMaterialsService materialsService;

    /**
     * 查询物料分类
     * 
     * @param id 物料分类主键
     * @return 物料分类
     */
    @Override
    public MaterialsCategory selectMaterialsCategoryById(String id)
    {
        return materialsCategoryMapper.selectMaterialsCategoryById(id);
    }

    /**
     * 查询物料分类列表
     * 
     * @param materialsCategory 物料分类
     * @return 物料分类
     */
    @Override
    public List<MaterialsCategory> selectMaterialsCategoryList(MaterialsCategory materialsCategory)
    {
        return materialsCategoryMapper.selectMaterialsCategoryList(materialsCategory);
    }
    /**
     * 查询分类树结构
     *
     * @param materialsCategory 物料分类
     * @return 结果
     */
    @Override
    public List<TreeSelect> selectMaterialsCategoryTreeList(MaterialsCategory materialsCategory)
    {
        List<MaterialsCategory> materialsCategorys = SpringUtils.getAopProxy(this).selectMaterialsCategoryList(materialsCategory);
        return buildTreeSelect(materialsCategorys);
    }
    /**
     * 构建前端所需要树结构
     *
     * @param materialsCategorys 物料分类
     * @return 结果
     */
    @Override
    public List<MaterialsCategory> buildMaterialsCategoryTree(List<MaterialsCategory> materialsCategorys) {
        List<MaterialsCategory> returnList = new ArrayList<MaterialsCategory>();
//        List<String> tempList = materialsCategorys.stream().map(MaterialsCategory::getId).collect(Collectors.toList());
        for (MaterialsCategory materialsCategory : materialsCategorys) {
            // 如果是顶级节点， 遍历该父节点的所有子节点
            if (StringUtils.isNull(materialsCategory.getParentId()) || materialsCategory.getParentId().isEmpty()){
                recursionFn(materialsCategorys, materialsCategory);
                returnList.add(materialsCategory);
            }
        }
        if (returnList.isEmpty()){
            returnList = materialsCategorys;
        }
        return returnList;
    }
    /**
     * 递归列表
     */
    private void recursionFn(List<MaterialsCategory> materialsCategoryList, MaterialsCategory materialsCategory) {
        // 得到子节点列表
        List<MaterialsCategory> childList = getChildList(materialsCategoryList, materialsCategory);
        materialsCategory.setChildren(childList);
        
        // 为当前分类添加物料列表
        List<Materials> materialsList = materialsService.selectMaterialsByCategory(materialsCategory.getCategoryCode());
        materialsCategory.setMaterialsList(materialsList);

        // 如果有子节点，递归处理
        if (!childList.isEmpty()) {
            for (MaterialsCategory child : childList) {
                recursionFn(materialsCategoryList, child);
            }
        }
    }
    /**
     * 得到子节点列表
     */
    private List<MaterialsCategory> getChildList(List<MaterialsCategory> list, MaterialsCategory materialsCategory){
        List<MaterialsCategory> childList = new ArrayList<>();
        Iterator<MaterialsCategory> iterator = list.iterator();
        while (iterator.hasNext()){
            MaterialsCategory m = (MaterialsCategory) iterator.next();
            if (StringUtils.equals(m.getParentId(), materialsCategory.getId())){
                childList.add(m);
            }
        }
        return childList;
    }
    /**
     * 判断是否有子节点
     */
    private boolean hasChild(List<MaterialsCategory> list, MaterialsCategory m){
        return getChildList(list, m).size() > 0;
    }
    /**
     * 构建前端所需要下拉树结构
     *
     * @param materialsCategorys 物料分类
     * @return 结果
     */
    @Override
    public List<TreeSelect> buildTreeSelect(List<MaterialsCategory> materialsCategorys) {
        List<MaterialsCategory> materialsCategories = buildMaterialsCategoryTree(materialsCategorys);
        return materialsCategories.stream().map(category -> new TreeSelect(category, category.getMaterialsList())).collect(Collectors.toList());
    }

    /**
     * 新增物料分类
     * 
     * @param materialsCategory 物料分类
     * @return 结果
     */
    @Override
    public int insertMaterialsCategory(MaterialsCategory materialsCategory)
    {
        materialsCategory.setCreateTime(DateUtils.getNowDate());
        return materialsCategoryMapper.insertMaterialsCategory(materialsCategory);
    }

    /**
     * 修改物料分类
     * 
     * @param materialsCategory 物料分类
     * @return 结果
     */
    @Override
    public int updateMaterialsCategory(MaterialsCategory materialsCategory)
    {
        materialsCategory.setUpdateTime(DateUtils.getNowDate());
        return materialsCategoryMapper.updateMaterialsCategory(materialsCategory);
    }

    /**
     * 修改物料分类有效状态
     *
     * @param ids 物料分类
     * @return 结果
     */
    @Override
    public int updateCategoryIsUsedByIds(String[] ids) {
        return materialsCategoryMapper.updateCategoryIsUsedByIds(ids);
    }

    /**
     * 批量删除物料分类(暂无用)
     * 
     * @param ids 需要删除的物料分类主键
     * @return 结果
     */
    @Override
    public int deleteMaterialsCategoryByIds(String[] ids)
    {
        return materialsCategoryMapper.deleteMaterialsCategoryByIds(ids);
    }

    /**
     * 删除物料分类信息(暂无用)
     * 
     * @param id 物料分类主键
     * @return 结果
     */
    @Override
    public int deleteMaterialsCategoryById(String id)
    {
        return materialsCategoryMapper.deleteMaterialsCategoryById(id);
    }
}
