package com.ruoyi.inventory.service.impl;

import java.util.*;

import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.inventory.domain.Inventory;
import com.ruoyi.inventory.service.IInventoryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ruoyi.common.utils.StringUtils;
import org.springframework.transaction.annotation.Transactional;
import com.ruoyi.inventory.domain.ReturnOrderItems;
import com.ruoyi.inventory.mapper.ReturnOrdersMapper;
import com.ruoyi.inventory.domain.ReturnOrders;
import com.ruoyi.inventory.service.IReturnOrdersService;

/**
 * 退库单主Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-12-03
 */
@Service
public class ReturnOrdersServiceImpl implements IReturnOrdersService 
{
    @Autowired
    private ReturnOrdersMapper returnOrdersMapper;
    @Autowired
    private IInventoryService inventoryService;

    /**
     * 查询退库单主
     * 
     * @param id 退库单主主键
     * @return 退库单主
     */
    @Override
    public ReturnOrders selectReturnOrdersById(String id)
    {
        return returnOrdersMapper.selectReturnOrdersById(id);
    }

    /**
     * 查询退库单主列表
     * 
     * @param returnOrders 退库单主
     * @return 退库单主
     */
    @Override
    public List<ReturnOrders> selectReturnOrdersList(ReturnOrders returnOrders)
    {
        return returnOrdersMapper.selectReturnOrdersList(returnOrders);
    }

    /**
     * 新增退库单主
     * 
     * @param returnOrders 退库单主
     * @return 结果
     */
    @Transactional
    @Override
    public int insertReturnOrders(ReturnOrders returnOrders)
    {
        if (StringUtils.isEmpty(returnOrders.getId())) {
            returnOrders.setId(UUID.randomUUID().toString());
        }
        returnOrders.setCreateTime(DateUtils.getNowDate());
        if (returnOrders.getIsUsed() == null) {
            returnOrders.setIsUsed(1);
        }
        int rows = returnOrdersMapper.insertReturnOrders(returnOrders);
        insertReturnOrderItems(returnOrders);
        return rows;
    }

    /**
     * 修改退库单主
     * 
     * @param returnOrders 退库单主
     * @return 结果
     */
    @Transactional
    @Override
    public int updateReturnOrders(ReturnOrders returnOrders)
    {
        returnOrders.setUpdateTime(DateUtils.getNowDate());
        returnOrdersMapper.deleteReturnOrderItemsByOrderId(returnOrders.getId());
        insertReturnOrderItems(returnOrders);
        return returnOrdersMapper.updateReturnOrders(returnOrders);
    }

    /**
     * 批量删除退库单主
     * 
     * @param ids 需要删除的退库单主主键
     * @return 结果
     */
    @Transactional
    @Override
    public int deleteReturnOrdersByIds(String[] ids)
    {
        returnOrdersMapper.deleteReturnOrderItemsByOrderIds(ids);
        return returnOrdersMapper.deleteReturnOrdersByIds(ids);
    }

    /**
     * 删除退库单主信息
     * 
     * @param id 退库单主主键
     * @return 结果
     */
    @Transactional
    @Override
    public int deleteReturnOrdersById(String id)
    {
        ReturnOrders returnOrders = returnOrdersMapper.selectReturnOrdersById(id);
        if (returnOrders != null && returnOrders.getOrderId() != null) {
            returnOrdersMapper.deleteReturnOrderItemsByOrderId(returnOrders.getOrderId());
        }
        return returnOrdersMapper.deleteReturnOrdersById(id);
    }

    @Override
    public List<ReturnOrderItems> selectExportList(ReturnOrders returnOrders) {
        return Collections.emptyList();
    }

    @Override
    public int confirmReturnOrdersByIds(String[] ids) {
        // 调用库存插入方法
        int count = 0;
        if(ids.length > 0){
            //库存变更
            for (int i = 0; i < ids.length; i++) {
                String id = ids[i].trim();
                ReturnOrders returnOrders = returnOrdersMapper.selectReturnOrdersById(id);
                List<Inventory> inventoryList = trans(returnOrders);
                count = inventoryService.insertInventoryList(inventoryList);
            }

            //库存更新后，退库确认，更新退库状态
            count = returnOrdersMapper.confirmReturnOrdersByIds(ids);
        }
        return count;
    }

    private List<Inventory> trans(ReturnOrders returnOrders) {
        List<Inventory> inventoryList = new ArrayList<>();
        List<ReturnOrderItems> items = returnOrders.getReturnOrderItemsList();
        long i = 1L;
        Date now = DateUtils.getNowDate();
        for (ReturnOrderItems item : items) {
            Inventory inventory = new Inventory();
            inventory.setId(UUID.randomUUID().toString());
            inventory.setInventoryType(2L);//退库
            inventory.setOrderId(returnOrders.getOrderId());
            inventory.setMaterialId(item.getMaterialId());
            inventory.setBatchId(item.getBatchCode());
            inventory.setWarehousesCode(item.getWarehouseId());
            inventory.setLocationId(item.getLocationId());
            inventory.setOwnerId(returnOrders.getOwnerId());
            inventory.setQuantity(item.getActualQuantity().longValue());
            inventory.setLockedQuantity(0L);
//            inventory.setUnitWeight();
//            inventory.setUnitPrice(item.getUnitPrice());
//            inventory.setTotalWeight();
//            inventory.setTotalVolume();
            inventory.setProductionDate(returnOrders.getCreateTime());
            inventory.setExpirationDate(returnOrders.getCreateTime());
            inventory.setInventoryStatus(1L);
            inventory.setLastInboundTime(returnOrders.getCreateTime());
//            inventory.setLastOutboundTime(returnOrders.getCreateTime());
            inventory.setIsUsed(1L);
            inventory.setSortNo(i++);
            inventory.setCreateTime(now);
            inventory.setCreateUserCode(returnOrders.getOwnerId());

            inventoryList.add(inventory);
        }

        return inventoryList;
    }

    /**
     * 新增退库单明细信息
     * 
     * @param returnOrders 退库单主对象
     */
    public void insertReturnOrderItems(ReturnOrders returnOrders)
    {
        List<ReturnOrderItems> returnOrderItemsList = returnOrders.getReturnOrderItemsList();
        String returnOrderId = returnOrders.getId();
        String orderId = returnOrders.getOrderId();
        if (StringUtils.isNotNull(returnOrderItemsList))
        {
            List<ReturnOrderItems> list = new ArrayList<ReturnOrderItems>();
            for (ReturnOrderItems returnOrderItems : returnOrderItemsList)
            {
                if (StringUtils.isEmpty(returnOrderItems.getId())) {
                    returnOrderItems.setId(UUID.randomUUID().toString());
                }
                returnOrderItems.setReturnOrderId(returnOrderId);
                returnOrderItems.setOrderId(orderId);
                returnOrderItems.setCreateTime(DateUtils.getNowDate());
                if (returnOrderItems.getIsUsed() == null) {
                    returnOrderItems.setIsUsed(1);
                }
                // 自动计算金额
                if (returnOrderItems.getActualQuantity() != null && returnOrderItems.getUnitPrice() != null) {
                    returnOrderItems.setTotalAmount(returnOrderItems.getActualQuantity() * returnOrderItems.getUnitPrice());
                }
                list.add(returnOrderItems);
            }
            if (list.size() > 0)
            {
                returnOrdersMapper.batchReturnOrderItems(list);
            }
        }
    }
}

