package com.ruoyi.web.controller.inventory;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletResponse;

import com.ruoyi.common.core.domain.entity.Materials;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.inventory.service.IMaterialsService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;
import org.springframework.web.multipart.MultipartFile;

/**
 * 物料Controller
 *
 * @author ruoyi
 * @date 2025-11-28
 */
@RestController
@RequestMapping("/inventory/materials")
public class MaterialsController extends BaseController
{
    @Autowired
    private IMaterialsService materialsService;

    /**
     * 查询物料列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:list')")
    @GetMapping("/list")
    public TableDataInfo list(Materials materials)
    {
        startPage();
        List<Materials> list = materialsService.selectMaterialsList(materials);
        return getDataTable(list);
    }

    /**
     * 导出物料列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:export')")
    @Log(title = "物料", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, Materials materials)
    {
        List<Materials> list = materialsService.selectMaterialsList(materials);
        ExcelUtil<Materials> util = new ExcelUtil<Materials>(Materials.class);
        util.exportExcel(response, list, "物料数据");
    }

    /**
     * 获取物料详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        return success(materialsService.selectMaterialsById(id));
    }

    /**
     * 获取物料详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:query')")
    @GetMapping(value = "getMaterial/{id}")
    public AjaxResult getMaterialByCategory(@PathVariable("id") String id)
    {
        return success(materialsService.selectMaterialsByCategory(id));
    }

    /**
     * 新增物料
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:add')")
    @Log(title = "物料", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody Materials materials)
    {
        materials.setId(UUID.randomUUID().toString());
        return toAjax(materialsService.insertMaterials(materials));
    }

    /**
     * 修改物料
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:edit')")
    @Log(title = "物料", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody Materials materials)
    {
        return toAjax(materialsService.updateMaterials(materials));
    }

    /**
     * 删除物料  (修改物料有效性)
     */

    @PreAuthorize("@ss.hasPermi('inventory:materials:remove')")
    @Log(title = "物料", businessType = BusinessType.DELETE)
    @PutMapping("/{ids}")
    public AjaxResult updateIsUsed(@PathVariable String[] ids)
    {
        return toAjax(materialsService.updateMaterialsIsUsed(ids));
    }

    /**
     * 下载物料导入模板
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:importTemplate')")
    @Log(title = "物料", businessType = BusinessType.IMPORT)
    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response)
    {
        ExcelUtil<Materials> util = new ExcelUtil<Materials>(Materials.class);
        util.importTemplateExcel(response, "物料信息");
    }

    /**
     * 导入物料
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:import')")
    @Log(title = "物料信息导入", businessType = BusinessType.IMPORT)
    @PostMapping("/import")
    public AjaxResult importTemplate(MultipartFile file , boolean updateSupport) throws Exception
    {
        ExcelUtil<Materials> util = new ExcelUtil<Materials>(Materials.class);
        List<Materials> materialsList = util.importExcel(file.getInputStream());
        String operName = getUsername();
        String message = materialsService.importMaterials(materialsList, updateSupport, operName);
        return success(message);
    }

    /**
     * @description: 获取物料 的 sap_no Name 做成字典
     * @author cs
     * @date 2025/12/4
     * @version 1.0
     */
    @GetMapping(value = "/getMapList")
    @ResponseBody
    public List<Map<String, Object>> getMapList()
    {
        try {
            return materialsService.getMapList();
        } catch (Exception e) {
            return Collections.emptyList(); // 替代new ArrayList<>()，更高效
        }
    }


    /**
     * 删除物料  (暂时无用)
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials:remove')")
    @Log(title = "物料", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(materialsService.deleteMaterialsByIds(ids));
    }
}
