package com.ruoyi.inventory.service.impl;

import java.util.*;
import java.util.stream.Collectors;

import com.ruoyi.common.core.domain.entity.Materials;
import com.ruoyi.common.exception.ServiceException;
import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.bean.BeanUtils;
import com.ruoyi.inventory.domain.InboundOrderItems;
import com.ruoyi.inventory.domain.InboundOrders;
import com.ruoyi.inventory.domain.vo.InboundTemplateVO;
import com.ruoyi.inventory.domain.vo.OutboundOrdersSummaryVO;
import com.ruoyi.inventory.domain.vo.OutboundTemplateVO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.inventory.mapper.OutboundOrderItemsMapper;
import com.ruoyi.inventory.domain.OutboundOrderItems;
import com.ruoyi.inventory.service.IOutboundOrderItemsService;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 出库单明细Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-12-03
 */
@Service
public class OutboundOrderItemsServiceImpl implements IOutboundOrderItemsService 
{
    @Autowired
    private OutboundOrderItemsMapper outboundOrderItemsMapper;

    @Autowired
    private InventoryServiceImpl inventoryService;

    @Autowired
    private OutboundOrderLogServiceImpl outboundOrderLogService;
    /**
     * 查询出库单明细
     * 
     * @param id 出库单明细主键
     * @return 出库单明细
     */
    @Override
    public OutboundOrderItems selectOutboundOrderItemsById(String id)
    {
        return outboundOrderItemsMapper.selectOutboundOrderItemsById(id);
    }

    /**
     * 查询出库单明细列表
     * 
     * @param outboundOrderItems 出库单明细
     * @return 出库单明细
     */
    @Override
    public List<OutboundOrderItems> selectOutboundOrderItemsList(OutboundOrderItems outboundOrderItems)
    {
        return outboundOrderItemsMapper.selectOutboundOrderItemsList(outboundOrderItems);
    }

    @Override
    public List<OutboundOrderItems> selectOutboundOrderItemsStatistics(OutboundOrdersSummaryVO outboundOrderItems) {
        List<OutboundOrderItems> list = outboundOrderItemsMapper.selectOutboundOrderItemsStatistics(outboundOrderItems);
        return list;
    }


    @Override
    public int insertOutboundOrderItems(OutboundOrderItems outboundOrderItems)
    {
        outboundOrderItems.setCreateTime(DateUtils.getNowDate());
        return outboundOrderItemsMapper.insertOutboundOrderItems(outboundOrderItems);
    }


    @Override
    public int updateOutboundOrderItems(OutboundOrderItems outboundOrderItems)
    {
        outboundOrderItems.setUpdateTime(DateUtils.getNowDate());
        return outboundOrderItemsMapper.updateOutboundOrderItems(outboundOrderItems);
    }

    /**
     * 批量删除出库单明细
     * 
     * @param ids 需要删除的出库单明细主键
     * @return 结果
     */
    @Override
    public int deleteOutboundOrderItemsByIds(String[] ids)
    {
        return outboundOrderItemsMapper.deleteOutboundOrderItemsById(ids);
    }

    /**
     * 删除出库单明细信息
     * 
     * @param id 出库单明细主键
     * @return 结果
     */
    @Override
    public int deleteOutboundOrderItemsById(String id)
    {
        return outboundOrderItemsMapper.deleteOutboundOrderItemsById(id);
    }

    /**
     * 导入入库单明细信息
     *
     * @param  inboundOrdersList,isUpdateSupport,operName 入库单数据信息
     * @return 结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String importOutoundOrders(List<OutboundTemplateVO> inboundOrdersList, Boolean isUpdateSupport, String operName) {
        if (StringUtils.isNull(inboundOrdersList) || inboundOrdersList.size() == 0) {
            throw new ServiceException("导入数据不能为空！");
        }
        // 2. 初始化统计变量
        int totalMainSuccess = 0; // 成功的主表数量
        int totalMainFailure = 0; // 失败的主表数量
        int totalItemSuccess = 0; // 成功的明细数量
        int totalItemFailure = 0; // 失败的明细数量
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        Date now = DateUtils.getNowDate();
        Long userId = SecurityUtils.getUserId();
        String operId = userId.toString();

        // 3. 按入库单号分组（核心：同一入库单的多条明细归为一组）
        Map<String, List<OutboundTemplateVO>> orderGroupMap = inboundOrdersList.stream()
                .filter(vo -> StringUtils.isNotBlank(vo.getOrderId())) // 过滤无入库单号的无效行
                .collect(Collectors.groupingBy(OutboundTemplateVO::getOrderId));

        // 4. 遍历每个入库单分组处理
        for (Map.Entry<String, List<OutboundTemplateVO>> entry : orderGroupMap.entrySet()) {
            String orderId = entry.getKey();
            List<OutboundTemplateVO> voList = entry.getValue();
            OutboundOrderItems mainDO = null;
            List<InboundOrderItems> itemDOList = new ArrayList<>();

            try {
                // 4.1 处理主表（每个入库单号只处理一次主表）
                OutboundTemplateVO firstVO = voList.get(0); // 取第一条VO的主表信息
                // 检查入库单是否已存在
                OutboundOrderItems existMain = outboundOrderItemsMapper.selectOutboundOrderItemsById(orderId);
                if (existMain != null) {
                    if (!isUpdateSupport) {
                        // 不支持更新，跳过该入库单
                        totalMainFailure++;
                        failureMsg.append(String.format("入库单号【%s】已存在，且不支持更新，跳过导入；\n", orderId));
                        totalItemFailure += voList.size(); // 该单的明细全部失败
                        continue;
                    }
                    // 支持更新，复用已有主表ID
                    mainDO = existMain;
                    // 复制VO中的主表字段到已有主表（只更新可修改的字段）
                    BeanUtils.copyProperties(firstVO, mainDO, "id", "createBy", "createTime"); // 排除不可更新字段
                    mainDO.setUpdateBy(operId);
                    mainDO.setUpdateTime(now);
                    mainDO.setUpdateUserCode(operId);
                    // 更新主表
                    outboundOrderItemsMapper.updateOutboundOrderItems(mainDO);
                    totalMainSuccess++;
                    successMsg.append(String.format("入库单号【%s】已更新；\n", orderId));
                } else {
                    // 新增主表
                    mainDO = new OutboundOrderItems();
                    // 复制主表字段（只复制主表相关字段，避免物料字段污染）
                    BeanUtils.copyProperties(firstVO, mainDO,
                            "sapNo", "materialName", "plannedQuantity", "actualQuantity",
                            "plannedPackages", "materialUnit", "materialRemark"); // 排除子表字段
                    // 填充主表必填字段
                    mainDO.setId(UUID.randomUUID().toString());
                    mainDO.setOrderId(orderId);
                    mainDO.setCreateBy(operId);
                    mainDO.setCreateTime(now);
                    mainDO.setCreateUserCode(operId);
                    mainDO.setUpdateBy(operId);
                    mainDO.setUpdateTime(now);
                    mainDO.setUpdateUserCode(operId);
                    // 设置默认值
                    if (mainDO.getSortNo() == null) {
                        mainDO.setSortNo(0L);
                    }
                    if (mainDO.getItemStatus() == null) {
                        mainDO.setItemStatus(1L); // 默认草稿状态
                    }
                    // 插入主表
                    outboundOrderItemsMapper.insertOutboundOrderItems(mainDO);
                    totalMainSuccess++;
                    successMsg.append(String.format("入库单号【%s】已新增；\n", orderId));
                }

                // 4.2 处理子表明细（每条VO对应一条明细）
                for (OutboundTemplateVO vo : voList) {
                    InboundOrderItems itemDO = new InboundOrderItems();
                    // 复制子表字段（物料相关）
                    BeanUtils.copyProperties(vo, itemDO,
                            "orderId", "systemNo", "orderTypeId", "batchId"); // 排除主表字段
                    // 填充明细必填字段
                    itemDO.setId(UUID.randomUUID().toString());
                    Materials materials = new Materials();
                    materials.setSapNo(vo.getSapNo());
                    itemDO.setMaterialId(vo.getSapNo());
                    itemDO.setOrderId(orderId); // 关联入库单号
                    itemDO.setBatchId(mainDO.getBatchCode());
                    itemDO.setInboundOrderId(mainDO.getId()); // 关联主表ID（核心！）
                    itemDO.setCreateBy(operId);
                    itemDO.setCreateTime(now);
                    itemDO.setCreateUserCode(operId);
                    itemDO.setSortNo(0L);
                    // 校验物料字段（示例：必填sapNo）
                    if (StringUtils.isBlank(vo.getSapNo())) {
                        throw new ServiceException(String.format("入库单号【%s】的物料SAP号为空，明细导入失败", orderId));
                    }
                    System.out.println(itemDO);
                    itemDOList.add(itemDO);
                }

                // 4.3 批量插入明细
                if (!CollectionUtils.isEmpty(itemDOList)) {
                    int itemSuccess = outboundOrderItemsMapper.batchInsertOutboundOrderItems(itemDOList);
                    totalItemSuccess += itemSuccess;
                    totalItemFailure += (itemDOList.size() - itemSuccess);
                    successMsg.append(String.format("入库单号【%s】成功导入%d条物料明细；\n", orderId, itemSuccess));
                    if (itemDOList.size() - itemSuccess > 0) {
                        failureMsg.append(String.format("入库单号【%s】有%d条物料明细导入失败；\n", orderId, itemDOList.size() - itemSuccess));
                    }
                }

            } catch (Exception e) {
                // 单个入库单处理失败，统计错误
                totalMainFailure++;
                totalItemFailure += voList.size();
                failureMsg.append(String.format("入库单号【%s】处理失败：%s；\n", orderId, e.getMessage()));
            }
        }

        // 5. 结果汇总
        if (totalMainFailure > 0 || totalItemFailure > 0) {
            // 有失败数据，抛出异常提示
            String finalFailureMsg = String.format(
                    "导入结果：成功新增/更新%d个入库单，失败%d个；成功导入%d条明细，失败%d条。失败详情：%s",
                    totalMainSuccess, totalMainFailure, totalItemSuccess, totalItemFailure, failureMsg.toString()
            );
            throw new ServiceException(finalFailureMsg);
        } else {
            // 全部成功
            String finalSuccessMsg = String.format(
                    "恭喜您，数据已全部导入成功！共处理%d个入库单，成功导入%d条物料明细。详情：%s",
                    totalMainSuccess, totalItemSuccess, successMsg.toString()
            );
            return finalSuccessMsg;
        }
    }
}
