package com.ruoyi.web.controller.inventory;

import java.util.List;
import javax.servlet.http.HttpServletResponse;

import com.ruoyi.common.core.domain.entity.SysDept;
import com.ruoyi.common.utils.uuid.UUID;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.core.domain.entity.MaterialsCategory;
import com.ruoyi.inventory.service.IMaterialsCategoryService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 物料分类Controller
 * 
 * @author ruoyi
 * @date 2025-11-28
 */
@RestController
@RequestMapping("/inventory/materials_category")
public class MaterialsCategoryController extends BaseController
{
    @Autowired
    private IMaterialsCategoryService materialsCategoryService;

    /**
     * 查询物料分类列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:list')")
    @GetMapping("/list")
    public TableDataInfo list(MaterialsCategory materialsCategory)
    {
        startPage();
        List<MaterialsCategory> list = materialsCategoryService.selectMaterialsCategoryList(materialsCategory);
        return getDataTable(list);
    }

    /**
     * 导出物料分类列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:export')")
    @Log(title = "物料分类", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, MaterialsCategory materialsCategory)
    {
        List<MaterialsCategory> list = materialsCategoryService.selectMaterialsCategoryList(materialsCategory);
        ExcelUtil<MaterialsCategory> util = new ExcelUtil<MaterialsCategory>(MaterialsCategory.class);
        util.exportExcel(response, list, "物料分类数据");
    }

    /**
     * 获取物料分类详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        return success(materialsCategoryService.selectMaterialsCategoryById(id));
    }

    /**
     * 获取分类树结构
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:list')")
    @GetMapping("/categoryTree")
    public AjaxResult categoryTree(MaterialsCategory materialsCategory)
    {
        return success(materialsCategoryService.selectMaterialsCategoryTreeList(materialsCategory));
    }

    /**
     * 新增物料分类
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:add')")
    @Log(title = "物料分类", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody MaterialsCategory materialsCategory)
    {
        materialsCategory.setId(UUID.randomUUID().toString());
        return toAjax(materialsCategoryService.insertMaterialsCategory(materialsCategory));

    }

    /**
     * 修改物料分类
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:edit')")
    @Log(title = "物料分类", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody MaterialsCategory materialsCategory)
    {
        return toAjax(materialsCategoryService.updateMaterialsCategory(materialsCategory));
    }

    /**
     * 删除物料分类
     */
    @PreAuthorize("@ss.hasPermi('inventory:materials_category:remove')")
    @Log(title = "物料分类", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(materialsCategoryService.deleteMaterialsCategoryByIds(ids));
    }
}
