package com.ruoyi.inventory.service.impl;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.ruoyi.common.exception.ServiceException;
import com.ruoyi.common.utils.DateUtils;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.inventory.domain.Owners;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.inventory.mapper.MaterialsMapper;
import com.ruoyi.inventory.domain.Materials;
import com.ruoyi.inventory.service.IMaterialsService;

/**
 * 物料Service业务层处理
 * 
 * @author ruoyi
 * @date 2025-11-28
 */
@Service
public class MaterialsServiceImpl implements IMaterialsService 
{
    @Autowired
    private MaterialsMapper materialsMapper;

    /**
     * 查询物料
     * 
     * @param id 物料主键
     * @return 物料
     */
    @Override
    public Materials selectMaterialsById(String id)
    {
        return materialsMapper.selectMaterialsById(id);
    }

    /**
     * 查询物料列表
     * 
     * @param materials 物料
     * @return 物料
     */
    @Override
    public List<Materials> selectMaterialsList(Materials materials)
    {
        return materialsMapper.selectMaterialsList(materials);
    }

    /**
     * 新增物料
     * 
     * @param materials 物料
     * @return 结果
     */
    @Override
    public int insertMaterials(Materials materials)
    {
        materials.setCreateTime(DateUtils.getNowDate());
        return materialsMapper.insertMaterials(materials);
    }

    /**
     * 修改物料
     * 
     * @param materials 物料
     * @return 结果
     */
    @Override
    public int updateMaterials(Materials materials)
    {
        materials.setUpdateTime(DateUtils.getNowDate());
        return materialsMapper.updateMaterials(materials);
    }

    /**
     * 批量修改物料的使用状态
     *
     * @param ids 需要删除的物料主键
     * @return 结果
     */
    @Override
    public int deleteMaterialsByIds(String[] ids)
    {
        return materialsMapper.updateMaterialsIsUsedByIds(ids);
    }

    /**
     * 删除物料信息
     * 
     * @param id 物料主键
     * @return 结果
     */
    @Override
    public int deleteMaterialsById(String id)
    {
        return materialsMapper.deleteMaterialsById(id);
    }

    @Override
    public String importMaterials(List<Materials> materialsList, Boolean isUpdateSupport, String operName)
    {
        if (StringUtils.isNull(materialsList) || materialsList.size() == 0)
        {
            throw new ServiceException("导入用户数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        Date now = DateUtils.getNowDate();
        Long userId = SecurityUtils.getUserId();
        String operId = userId.toString();
        for (Materials materials : materialsList)
        {
            try
            {
                materials.setId(UUID.randomUUID().toString());
                // 填充创建人、创建时间、修改人、修改时间
                materials.setCreateBy(operId);
                materials.setCreateTime(now);
                // 填充创建用户编码和更新用户编码
                materials.setCreateUserCode(operId);
                // 设置默认值
                if (materials.getIsActive() == null)
                {
                    materials.setIsActive(1L); // 默认激活
                }
                if (materials.getIsUsed() == null)
                {
                    materials.setIsUsed(1L); // 默认未删除
                }
                if (materials.getSortNo() == null)
                {
                    materials.setSortNo(0L); // 默认排序号
                }
                materialsMapper.insertMaterials(materials);
                successNum++;
                successMsg.append("<br/>" + successNum + "、物料 " + materials.getMaterialName() + " 导入成功");
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、物料 " + materials.getMaterialName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());

            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确");
            throw new ServiceException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }
}
