package com.ruoyi.inventory.controller;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletResponse;

import com.ruoyi.inventory.domain.Owners;
import com.ruoyi.inventory.domain.vo.OwnerTemplateVO;
import com.ruoyi.inventory.domain.vo.StorageLocationsLocationTemplateVO;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.inventory.domain.StorageLocations;
import com.ruoyi.inventory.service.IStorageLocationsService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;
import org.springframework.web.multipart.MultipartFile;

/**
 * 库位Controller
 *
 * @author ruoyi
 * @date 2025-12-01
 */
@RestController
@RequestMapping("/inventory/locations")
public class StorageLocationsController extends BaseController
{
    @Autowired
    private IStorageLocationsService storageLocationsService;

    /**
     * 查询库位列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:list')")
    @GetMapping("/list")
    public TableDataInfo list(StorageLocations storageLocations)
    {
        startPage();
        List<StorageLocations> list = storageLocationsService.selectStorageLocationsList(storageLocations);
        return getDataTable(list);
    }

    /**
     * 导出库位列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:export')")
    @Log(title = "库位", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, StorageLocations storageLocations)
    {
        List<StorageLocations> list = storageLocationsService.selectStorageLocationsList(storageLocations);
        ExcelUtil<StorageLocations> util = new ExcelUtil<StorageLocations>(StorageLocations.class);
        util.exportExcel(response, list, "库位数据");
    }

    /**
     * 获取库位详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        return success(storageLocationsService.selectStorageLocationsById(id));
    }

    /**
     * 新增库位
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:add')")
    @Log(title = "库位", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody StorageLocations storageLocations)
    {
        return toAjax(storageLocationsService.insertStorageLocations(storageLocations));
    }

    /**
     * 修改库位
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:edit')")
    @Log(title = "库位", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody StorageLocations storageLocations)
    {
        return toAjax(storageLocationsService.updateStorageLocations(storageLocations));
    }

    /**
     * 删除库位
     */
    @PreAuthorize("@ss.hasPermi('inventory:locations:remove')")
    @Log(title = "库位", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(storageLocationsService.deleteStorageLocationsByIds(ids));
    }

    /**
     * @description: 获取库位表 的 location_code 编码 location_name 做成字典
     * @author cs
     * @date 2025/12/4
     * @version 1.0
     */
    @GetMapping(value = "/getMapList")
    @ResponseBody
    public List<Map<String, Object>> getMapList(StorageLocations storageLocations)
    {
        try {
            return storageLocationsService.getMapList(storageLocations);
        } catch (Exception e) {
            return Collections.emptyList(); // 替代new ArrayList<>()，更高效
        }
    }

    /**
     * 查询库位列表（无权限限制，用于选择器）
     */
    @GetMapping("/listForSelector")
    public TableDataInfo listForSelector(StorageLocations storageLocations)
    {
        startPage();
        List<StorageLocations> list = storageLocationsService.getStorageLocationsList(storageLocations);
        return getDataTable(list);
    }

    @PreAuthorize("@ss.hasPermi('inventory:owners:add')")
    @Log(title = "货主信息", businessType = BusinessType.IMPORT)
    @PostMapping("/import")
    public AjaxResult importTemplate(MultipartFile file , boolean updateSupport) throws Exception
    {
        ExcelUtil<StorageLocationsLocationTemplateVO> util = new ExcelUtil<StorageLocationsLocationTemplateVO>(StorageLocationsLocationTemplateVO.class);
        List<StorageLocationsLocationTemplateVO> storageLocationsLocationTemplateVOS = util.importExcel(file.getInputStream());
        String operName = getUsername();
        String message = storageLocationsService.importStorageLocationsLocation(storageLocationsLocationTemplateVOS, updateSupport, operName);
        return success(message);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response)
    {
        ExcelUtil<StorageLocationsLocationTemplateVO> util = new ExcelUtil<StorageLocationsLocationTemplateVO>(StorageLocationsLocationTemplateVO.class);
        util.importTemplateExcel(response, "货主数据");
    }
}
