package com.ruoyi.inventory.controller;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletResponse;

import com.ruoyi.common.config.WarehouseConfig;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.inventory.domain.Warehouses;
import com.ruoyi.inventory.service.IWarehousesService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 仓库Controller
 *
 * @author ruoyi
 * @date 2025-12-01
 */
@RestController
@RequestMapping("/inventory/warehouses")
public class WarehousesController extends BaseController
{
    @Autowired
    private IWarehousesService warehousesService;

    /**
     * 查询仓库列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:list')")
    @GetMapping("/list")
    public TableDataInfo list(Warehouses warehouses)
    {
        warehouses.setLocalId(WarehouseConfig.DEFAULT_WAREHOUSE_ID);
        startPage();
        List<Warehouses> list = warehousesService.selectWarehousesList(warehouses);
        return getDataTable(list);
    }

    /**
     * 导出仓库列表
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:export')")
    @Log(title = "仓库", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, Warehouses warehouses)
    {
        warehouses.setLocalId(WarehouseConfig.DEFAULT_WAREHOUSE_ID);
        List<Warehouses> list = warehousesService.selectWarehousesList(warehouses);
        ExcelUtil<Warehouses> util = new ExcelUtil<Warehouses>(Warehouses.class);
        util.exportExcel(response, list, "仓库数据");
    }

    /**
     * 获取仓库详细信息
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id)
    {
        return success(warehousesService.selectWarehousesById(id));
    }

    /**
     * 新增仓库
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:add')")
    @Log(title = "仓库", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody Warehouses warehouses)
    {
        return toAjax(warehousesService.insertWarehouses(warehouses));
    }

    /**
     * 修改仓库
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:edit')")
    @Log(title = "仓库", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody Warehouses warehouses)
    {
        return toAjax(warehousesService.updateWarehouses(warehouses));
    }

    /**
     * 删除仓库
     */
    @PreAuthorize("@ss.hasPermi('inventory:warehouses:remove')")
    @Log(title = "仓库", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids)
    {
        return toAjax(warehousesService.deleteWarehousesByIds(ids));
    }

    /**
     * @description: 获取仓库 的 warehouses_code 仓库编码 warehouses_name 做成字典
     * @author cs
     * @date 2025/12/4
     * @version 1.0
     */
    @GetMapping(value = "/getMapList")
    @ResponseBody
    public List<Map<String, Object>> getMapList()
    {
        try {
            return warehousesService.getMapList();
        } catch (Exception e) {
            return Collections.emptyList(); // 替代new ArrayList<>()，更高效
        }
    }

    /**
     * 查询仓库列表（无权限限制，用于选择器）
     */
    @GetMapping("/listForSelector")
    public TableDataInfo listForSelector(Warehouses warehouses)
    {
        warehouses.setLocalId(WarehouseConfig.DEFAULT_WAREHOUSE_ID);
        startPage();
        List<Warehouses> list = warehousesService.selectWarehousesList(warehouses);
        return getDataTable(list);
    }
}
