import { h, ref, reactive } from 'vue'
import { updateUserPwd } from '@/api/system/user'
import {
  ElMessageBox,
  ElMessage,
  ElForm,
  ElFormItem,
  ElInput
} from 'element-plus'
import type { FormInstance, FormRules, Action, MessageBoxState } from 'element-plus'
import useUserStore from '@/store/modules/user'
import { logoutUrl } from '@/api/oauth2'

const pwdRef = ref()
const user = reactive({
  oldPassword: '',
  newPassword: '',
  confirmPassword: ''
})
const equalToPassword = (rule, value, callback) => {
  if (user.newPassword !== value) {
    callback(new Error('两次输入的密码不一致'))
  } else {
    callback()
  }
}
const regex = new RegExp('(?=.*[0-9])(?=.*[a-zA-Z])(?=.*[^a-zA-Z0-9])')

const rules = reactive<FormRules>({
  oldPassword: [{ required: true, message: '旧密码不能为空', trigger: 'blur' }],
  newPassword: [{ required: true, message: '新密码不能为空', trigger: 'blur' }, { min: 6, max: 20, message: '长度在 6 到 20 个字符', trigger: 'blur' }, { pattern: regex, message: '密码需由特殊字符，字母及数字组成', trigger: 'blur' }],
  confirmPassword: [{ required: true, message: '确认密码不能为空', trigger: 'blur' }, { required: true, validator: equalToPassword, trigger: 'blur' }]
})

/** 提交按钮 */
function submit(action: Action, instance: MessageBoxState, done: () => void): void {
  const userStore = useUserStore()
  // 切换账号
  if (action === 'cancel') {
    userStore.ssoLogout()
    return
  }
  // 修改密码
  pwdRef.value.validate(valid => {
    if (valid) {
      instance.confirmButtonLoading = true
      updateUserPwd(user.oldPassword, user.newPassword).then(response => {
        // proxy.$modal.msgSuccess('修改成功')
        ElMessage({
          message: '修改密码成功',
          type: 'success'
        })
        setTimeout(() => {
          userStore.logout().then(() => {
            // location.href = `${location.origin}${import.meta.env.BASE_URL}`
            logoutUrl().then(res => {
              window.location.href = res + '?redirect_uri=' + window.location.href
            })
          })
        }, 1000)
        // done()
      }).finally(() => {
        instance.confirmButtonLoading = false
      })
    }
  })
}

function VNodeForm() {
  return h(
    ElForm,
    {
      ref: pwdRef,
      model: user,
      rules: rules,
      'label-width': '80px'
    },
    () => [
      h(
        ElFormItem,
        {
          label: '旧密码',
          prop: 'oldPassword'
        },
        () => h(
          ElInput,
          {
            type: 'password',
            showPassword: true,
            placeholder: '请输入旧密码',
            modelValue: user.oldPassword,
            autocomplete: 'new-password',
            onInput: (value) => { user.oldPassword = value }
          }
        )
      ),
      h(
        ElFormItem,
        {
          label: '新密码',
          prop: 'newPassword',
          style: { width: '100%' }
        },
        () => h(
          ElInput,
          {
            type: 'password',
            showPassword: true,
            placeholder: '请输入新密码',
            modelValue: user.newPassword,
            autocomplete: 'new-password',
            onInput: (value) => { user.newPassword = value }
          }
        )
      ),
      h(
        ElFormItem,
        {
          label: '确认密码',
          prop: 'confirmPassword'
        },
        () => h(
          ElInput,
          {
            type: 'password',
            showPassword: true,
            placeholder: '请确认新密码',
            modelValue: user.confirmPassword,
            autocomplete: 'new-password',
            onInput: (value) => { user.confirmPassword = value }
          }
        )
      )
    ]
  )
}

export function open(passwordStatus: string) {
  user.oldPassword = ''
  user.newPassword = ''
  user.confirmPassword = ''
  return ElMessageBox({
    title: passwordStatus === '1' ? '首次登录修改密码' : passwordStatus === '2' ? '密码已过期请重置' : '修改密码',
    customStyle: { width: '100%' },
    showClose: false,
    closeOnClickModal: false,
    showCancelButton: true,
    cancelButtonText: '退出登录',
    confirmButtonText: '保存密码',
    customClass: 'message-box-password-modify',
    message: VNodeForm,
    beforeClose: submit
  })
}
