import { ref, type Ref, reactive, computed } from 'vue'
import {
  getCenter
} from '../utils/index'
import {
  getDistance,
  getCenterByPoints
} from '../utils/algorithm'
import * as variables from './variables'
const {
  svgWrap,
  actives,
  x,
  y,
  oldX,
  oldY,
  scale,
  pageX,
  pageY,
  startX,
  startY,
  xSelection,
  ySelection,
  state
} = variables


let oldTiem = new Date().getTime()
let isTouchpad = false
let canMove = false

let oldDistance: number | undefined = 0 // 触摸时，上一次记录的2点距离
let distance = 0 // 触摸时，2点距离
let touchCenter: {
  offsetX?: number
  offsetY?: number
} = {} // 双指触摸缩放中心点

// 右键
function onMouseContextmenu(e: MouseEvent) {
  e.stopPropagation()
  e.preventDefault()
}
// 拖动 - 开始
function EventStart(e: any) {
  oldTiem = new Date().getTime()
  e.stopPropagation()
  e.preventDefault()
  pageX.value = e.pageX ?? e.changedTouches[0].pageX
  pageY.value = e.pageY ?? e.changedTouches[0].pageY
  oldX.value = x.value
  oldY.value = y.value
  canMove = true
}
// 拖动 - 中
function EventMove(e: any) {
  e.stopPropagation()
  if (!e.changedTouches) e.preventDefault()
  if (!canMove) return
  x.value = (e.pageX ?? e.changedTouches[0].pageX) - pageX.value + oldX.value
  y.value = (e.pageY ?? e.changedTouches[0].pageY) - pageY.value + oldY.value
}
// 拖动 - 结束
function EventEnd(e: any) {
  if (!e.changedTouches) e.preventDefault()
  // this.dragging = false
  canMove = false
  document.removeEventListener('mousemove', onMouseMove)
  document.removeEventListener('mouseup', onMouseUp)
  if (new Date().getTime() - oldTiem < 300 && (e.pageX ?? e.changedTouches[0].pageX) === pageX.value && (e.pageY ?? e.changedTouches[0].pageY) === pageY.value) {
    onClickSVG(e)
    return
  }
}
// 拖动 - 开始
function onMouseDown(e: MouseEvent) {
  EventStart(e)
  if (e.shiftKey) {
    // 圈选
    startX.value = (e.offsetX - x.value) / scale.value
    startY.value = (e.offsetY - y.value) / scale.value
    setMouseXYSelection(e)
    document.addEventListener('mousemove', onMouseMoveSelection, false)
    document.addEventListener('mouseup', onMouseUpSelection, false)
    state.value.showSelection = true
    return
  }
  document.addEventListener('mousemove', onMouseMove, false)
  document.addEventListener('mouseup', onMouseUp, false)
}
// 拖动 - 中
function onMouseMove(e: MouseEvent) {
  EventMove(e)
}
// 拖动 - 结束
function onMouseUp(e: MouseEvent) {
  EventEnd(e)
}
// 触摸 - 开始
function onTouchStart(e: TouchEvent) {
  EventStart(e)
  distance = 0
  document.addEventListener('touchmove', onTouchMove, false)
  document.addEventListener('touchend', onTouchEnd, false)
}
// 触摸 - 移动
function onTouchMove(e: TouchEvent) {
  if (e.touches?.length === 2) {
    onTouchPinchZoom(e)
    return
  }
  EventMove(e)
}
// 触摸 - 结束
function onTouchEnd(e: TouchEvent) {
  touchCenter = {}
  oldDistance = undefined
  EventEnd(e)
  if (e.touches?.length) return
  document.removeEventListener('touchmove', onTouchMove, false)
  document.removeEventListener('touchend', onTouchEnd, false)
}
// 圈选 - 开始
function setMouseXYSelection(e) {
  xSelection.value = (e.pageX - pageX.value - x.value + oldX.value) / scale.value + startX.value
  ySelection.value = (e.pageY - pageY.value - y.value + oldY.value) / scale.value + startY.value
}
// 圈选 - 中
function onMouseMoveSelection(e) {
  e.stopPropagation()
  e.preventDefault()
  if (!canMove) return
  if (!state.value.draggingSVG) {
    state.value.draggingSVG = true
  }
  setMouseXYSelection(e)
}
// 圈选 - 结束
function onMouseUpSelection(e) {
  canMove = false
  state.value.showSelection = false
  document.removeEventListener('mousemove', onMouseMoveSelection)
  document.removeEventListener('mouseup', onMouseUpSelection)
}

// 点击画布
function onClickSVG(e) {
  console.log('点击画布')
  if (e.target === svgWrap.value) {
    actives.value.splice(0, actives.value.length)
  }
}

// 修改缩放和坐标
function getZoomXY(sc: number, { offsetX, offsetY }) {
  // 如果通过触控板缩放
  // if (isTouchpad) {
  //   sc = sc / 5
  // }
  sc = sc * 0.2
  if (scale.value + sc < 0.1) {
    sc = 0.1 - scale.value
  }
  // 上次缩放量
  const s1 = scale.value
  scale.value += sc
  // 本次缩放量
  const s = scale.value
  // 对象在缩放后移动量
  const dx = (1 - s / s1) * (offsetX - x.value)
  const dy = (1 - s / s1) * (offsetY - y.value)
  // 得到缩放后对象偏移点
  x.value += dx
  y.value += dy
}
// 获取滚轮滚动的方向
function getDirect(e: WheelEvent): {
  direct: string
  placement: string
  deltaX: number
  deltaY: number
  // wheelData: number
} {
  const direct: string = ''
  const placement: string = e.deltaY < 0 ? 'up' : 'down'
  // if (!e.deltaX) {
  //   placement = e.deltaX > 0 ? 'up' : 'down'
  //   direct = 'vertical'
  // } else if (!e.deltaY) {
  //   placement = e.deltaY > 0 ? 'left' : 'right'
  //   direct = 'horizontal'
  // }
  return {
    direct,
    placement,
    deltaX: e.deltaX,
    deltaY: e.deltaY
    // wheelDelta: e.wheelDelta
  }
}
// 滚轮 - 滚动 - 上下左右
function onWheel(e: WheelEvent) {
  e.preventDefault()
  e.stopPropagation()
  if (e.ctrlKey) {
    onMouseWheelZoom(e)
    return
  }
  const { direct, placement, deltaX, deltaY } = getDirect(e)
  y.value = y.value - deltaY
  x.value = x.value - deltaX
}
// 滚轮 - 缩放
function onMouseWheelZoom(e: WheelEvent) {
  const { direct, placement, deltaX, deltaY } = getDirect(e)
  if (!Number.isInteger(e.deltaY)) {
    // 使用触摸板缩放
    isTouchpad = true
  }
  getZoomXY(Math.sign(-e.deltaY) * 0.1, e)
  isTouchpad = false
}
// 双指 - 缩放
function onTouchPinchZoom(e: TouchEvent) {
  const [p1, p2] = e.touches
  const [offsetX, offsetY] = getCenterByPoints([p1.pageX, p1.pageY], [p2.pageX, p2.pageY])
  let diffLeft = 0
  let diffTop = 0
  let parent = svgWrap.value?.parentElement
  if (touchCenter.offsetX === undefined) {
    while (parent) {
      diffLeft += parent?.offsetLeft ?? 0
      diffTop += parent?.offsetTop ?? 0
      parent = (parent?.offsetParent ?? parent.parentElement) as HTMLElement
    }
    touchCenter = { offsetX: offsetX - diffLeft, offsetY: offsetY - diffTop }
  }
  distance = getDistance([p1.pageX, p1.pageY], [p2.pageX, p2.pageY])
  if (oldDistance === undefined) oldDistance = distance
  getZoomXY(Math.sign((distance - oldDistance)) * 0.1, { offsetX: touchCenter.offsetX, offsetY: touchCenter.offsetY })
  oldDistance = distance
}
// 缩放+-
export function zoomSize(type: number) {
  if (!(svgWrap.value)) return
  // const { width, height } = svgWrap?.getBoundingClientRect()
  switch (type) {
    case 1:
      // 缩小
      getZoomXY(-0.5, {
        offsetX: svgWrap.value?.width.baseVal.value / 2,
        offsetY: svgWrap.value?.height.baseVal.value / 2
      })
      break
    case 2:
      // 放大
      getZoomXY(0.5, {
        offsetX: svgWrap.value?.width.baseVal.value / 2,
        offsetY: svgWrap.value?.height?.baseVal?.value / 2
      })
      break
    case 3:
      // 放大
      getZoomXY((1 - scale.value) * 5, {
        offsetX: svgWrap.value?.width.baseVal.value / 2,
        offsetY: svgWrap.value?.height?.baseVal?.value / 2
      })
      break
    default:
      // 重置
      // resetPosition()
      break
  }
}
// 重置位置缩放
export function resetPosition(nodeConfig) {
  // const nodeConfig = inject<any>('nodeConfig')
  // console.log(nodeConfig)
  const coords = getCenter(nodeConfig, svgWrap.value)
  x.value = coords.x
  y.value = coords.y
  scale.value = coords.scale
}

// 导出变量
// export const varibles = {
//   actives,
//   state,
//   scale,
//   oldScale,
//   x,
//   y,
//   oldX,
//   oldY,
//   pageX,
//   pageY,
//   startX,
//   startY,
//   xSelection,
//   ySelection
// }

// let svgWrap: SVGSVGElement | null = null
// 初始化
export function init(svgWrapRef: Ref<SVGSVGElement> | Ref<null>) {
  if (svgWrapRef?.value) {
    console.log('init svg')
    svgWrap.value = svgWrapRef.value
    svgWrap.value.addEventListener('mousedown', onMouseDown, false)
    svgWrap.value.addEventListener('touchstart', onTouchStart, false)
    svgWrap.value.addEventListener('contextmenu', onMouseContextmenu, false)
    svgWrap.value.addEventListener('wheel', onWheel, { passive: false })
  }
}
export function destroy() {
  svgWrap.value?.removeEventListener('mousedown', onMouseDown)
  svgWrap.value?.removeEventListener('wheel', onWheel)
}
